package cn.gtmap.realestate.supervise.server.utils;

import cn.gtmap.estateplat.core.ex.AppException;
import cn.gtmap.realestate.supervise.model.FileMessage;
import cn.gtmap.realestate.supervise.server.model.XmlValidateResult;
import cn.gtmap.realestate.supervise.utils.FileUtil;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.xml.serialize.OutputFormat;
import org.apache.xml.serialize.XMLSerializer;
import org.dom4j.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import javax.xml.XMLConstants;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;
import java.io.*;
import java.lang.annotation.Annotation;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author <a href="mailto:tianjian@gtmap.cn">tianjian</a>
 * @version 1.0, 2017/3/23
 * @description xml工具封装
 */
public class XmlUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(XmlUtil.class);

    /**
     * 解析XML，获取Head里面指定节点值
     *
     * @param fileContent
     * @param preEstateNum 原不动产单元号
     * @param preCertID    原不动产权证号
     * @return
     * @throws DocumentException
     */
    public static Map getXMLValueMap(String fileContent, String preEstateNum, String preCertID)
            throws DocumentException {
        Document doc;
        Map valueMap = new HashMap();
        doc = DocumentHelper.parseText(fileContent);
        Element root = doc.getRootElement();
        Element head = root.element("Head");
        Element bdcdyhElement = head.element(preEstateNum);
        String bdcdyh = bdcdyhElement.getTextTrim();
        if (null != bdcdyh && !bdcdyh.isEmpty()) {
            valueMap.put("bdcdyh", bdcdyh);//不动产单元号
        }
        Element bdcqzhElement = head.element(preCertID);
        String bdcqzh = bdcqzhElement.getTextTrim();
        if (null != bdcqzh && !bdcqzh.isEmpty()) {
            valueMap.put("bdcqzh", bdcqzh);//不动产权证号
        }

        return valueMap;
    }

    /**
     * 获取报文头不动产单元号
     *
     * @param fileContent
     * @param estateNum
     * @return
     * @throws DocumentException
     */
    public static String getBdcdyh(String fileContent, String estateNum)
            throws DocumentException {
        Document doc;
        doc = DocumentHelper.parseText(fileContent);
        Element root = doc.getRootElement();
        Element head = root.element("Head");
        Element bdcdyhElement = head.element(estateNum);
        String bdcdyh = bdcdyhElement == null ? "" : bdcdyhElement.getTextTrim();

        return bdcdyh;
    }

    /**
     * @param xmlStr 需要验证的xml字符串
     * @param xsdStr 验证xml的xsd字符串
     * @return 包含验证结果的对象
     */
    public static XmlValidateResult checkXmlByXsd(String xmlStr, String xsdStr) {

        // 创建返回值类，默认为失败
        XmlValidateResult vs = new XmlValidateResult();

        if (xmlStr == null || xsdStr == null) {
            return vs;
        }
        String schemaLanguage = XMLConstants.W3C_XML_SCHEMA_NS_URI;
        SchemaFactory factory = SchemaFactory.newInstance(schemaLanguage);

//        SchemaFactory factory = SchemaFactory.newInstance("http://www.w3.org/2001/XMLSchema");

        // 包装待验证的xml字符串为Reader
        Reader xmlReader = new BufferedReader(new StringReader(xmlStr));

        // 保障Schema xsd字符串为Reader
        Reader xsdReader = new BufferedReader(new StringReader(xsdStr));
        try {

            // 构造Schema Source
            Source xsdSource = new StreamSource(xsdReader);

            // 解析作为Schema的指定源并以Schema形式返回它
            Schema schema = factory.newSchema(xsdSource);

            // 根据Schema检查xml文档的处理器,创建此 Schema的新 validator
            Validator validator = schema.newValidator();

            // 构造待验证xml Source
            Source xmlSource = new StreamSource(xmlReader);

            // 执行验证
            validator.validate(xmlSource);

            // 设置验证通过
            vs.setValidated(true);
            return vs;
        } catch (SAXException ex) {
            LOGGER.error("-----XmlUtil.checkXmlByXsd SAXException in !:{}", ex);
            // 设置验证失败
            vs.setValidated(false);
            String errorMsg = ex.getMessage();
            // 设置验证失败信息
            vs.setErrorMsg(errorMsg.substring(errorMsg.indexOf(":") + 1));
            return vs;
        } catch (IOException e) {

            LOGGER.error("---------XmlUtil.checkXmlByXsd IOException in !:{}", e);
            // 设置验证失败
            vs.setValidated(false);

            String errorMsg = e.getMessage();
            // 设置验证失败信息
            vs.setErrorMsg(errorMsg.substring(errorMsg.indexOf(":") + 1));
            return vs;

        }
    }

    /**
     * xml内容格式化
     *
     * @param unformattedXml
     * @return
     */
    public static String format(String unformattedXml) {
        try {
            final org.w3c.dom.Document document = parseXmlFile(unformattedXml);
            OutputFormat format = new OutputFormat(document);
            format.setLineWidth(65);
            format.setIndenting(true);
            format.setIndent(2);
            Writer out = new StringWriter();
            XMLSerializer serializer = new XMLSerializer(out, format);
            serializer.serialize(document);
            return out.toString();
        } catch (IOException e) {
            LOGGER.error("--------XmlUtil.format IOException in !:{}", e);
            throw new AppException(e, 2005);
        }
    }

    public static org.w3c.dom.Document parseXmlFile(String in) {
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        DocumentBuilder db = null;
        org.w3c.dom.Document resDoc;
        try {
            db = dbf.newDocumentBuilder();
        } catch (ParserConfigurationException e) {
            LOGGER.error("----------------XmlUtil.parseXmlFile ParserConfigurationException in !--{}:", e);
            throw new AppException(e, 2003);
        }
        InputSource is = new InputSource(new StringReader(in));
        try {
            resDoc = db.parse(is);
        } catch (SAXException e) {
            LOGGER.error("----------------XmlUtil.parseXmlFile SAXException in !-------" + e);
            throw new AppException(e, 2004);
        } catch (IOException e) {
            LOGGER.error("----------------XmlUtil.parseXmlFile IOException in !-------" + e);
            throw new AppException(e, 2005);
        }

        return resDoc;
    }


    /**
     * fileMessage 转换成String
     *
     * @param fileMessage
     * @return
     */
    public static String getContent(FileMessage fileMessage) {

        String content = "";
        try {
            content = new String(fileMessage.getContent(), "UTF-8");
        } catch (UnsupportedEncodingException e) {
            LOGGER.error("--------------XmlUtil.getContent UnsupportedEncodingException in-------" + e);
            throw new AppException(e, 2002);
        }
        return content;
    }


    public static String getFileByParam(String basePath, String coderArea) {
        String coderPath = basePath + "\\" + coderArea;
        File coderResFile = new File(coderPath);
        if (!coderResFile.exists()) {
            coderResFile.mkdirs();
        }
        return coderPath;
    }


    public static String getTextByXpath(String xpath, String xml) {
        Document document = null;
        try {
            document = DocumentHelper.parseText(xml);
            Node destNode = document.selectSingleNode(xpath);
            if (destNode == null) {
                return null;
            }
            return destNode.getText();
        } catch (DocumentException e) {
            LOGGER.error("getTextByXpath.DocumentException in!{}", e);
        }
        return null;
    }


    /**
     * @param fileContent   需要抽取的字符串
     * @param parentElement 父节点名称
     * @return
     * @throws
     * @description 获取xml特定节点的信息 对节点的样式有比较严格的规范
     * 获取的样板如下
     * <head>
     * <name>person</name>
     * <sex>sex</sex>
     * </head>
     * 可以获取head并返回包含key分别为name和sex value分别为person和sex的map
     */
    public static Map<String, String> getParamValueByElement(String fileContent, String parentElement) throws DocumentException {
        Document doc;
        doc = DocumentHelper.parseText(fileContent);
        Map<String, String> data = new HashMap<String, String>();
        Element root = doc.getRootElement();
        Element head = root.element(parentElement);
        List<Element> list = head.elements();
        for (Element e : list) {
            String name = e.getName().trim();
            String value = e.getText().trim();
            data.put(name, value);
        }
        return data;
    }


    public static String convertStreamToString(InputStream is) throws IOException {

        BufferedReader reader = new BufferedReader(new InputStreamReader(is, "UTF-8"));
        StringBuilder sb = new StringBuilder();

        String line = null;
        try {
            while ((line = reader.readLine()) != null) {
                sb.append(line + "\n");
            }
        } catch (IOException e) {
            LOGGER.error("--------------XmlUtil.convertStreamToString IOException in-------" + e);
            throw new AppException(e, 2005);
        } finally {
            is.close();
        }
        return sb.toString();
    }


    public static <T> T converyToJavaBean(String xml, Class<T> c) {
        T t = null;
        try {
            JAXBContext context = JAXBContext.newInstance(c);
            Unmarshaller unmarshaller = context.createUnmarshaller();
            t = (T) unmarshaller.unmarshal(new StringReader(xml));
        } catch (Exception e) {
            LOGGER.error("XmlUtil.converyToJavaBean Exception in!{}", e);
            throw new AppException(2027);
        }

        return t;
    }


    public static String entityToXml(Object object) {
        String xmlStr = "";
        try {
            if (object != null) {
                Annotation[] aos = object.getClass().getAnnotations();
                Boolean bol = false;
                for (Annotation ao : aos) {
                    if (ao.annotationType().equals(XmlRootElement.class)) {
                        bol = true;
                        break;
                    }
                }
                if (bol) {
                    JAXBContext jaxbContext = JAXBContext.newInstance(object.getClass());
                    Marshaller jaxbMarshaller = jaxbContext.createMarshaller();
                    jaxbMarshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
                    StringWriter writer = new StringWriter();
                    jaxbMarshaller.marshal(object, writer);
                    xmlStr = writer.toString().replaceAll("\\s*standalone=\"yes\"", "");
                }
            }
        } catch (Exception e) {
            LOGGER.error("entityToXml exception in!{}", e);
        }
        return xmlStr;
    }

    public static String getFileByByte(File file) throws UnsupportedEncodingException {
        byte[] fileBytes = new byte[0];
        fileBytes = FileUtil.getBytes(file);
        String encodeType = "";
        try {
            encodeType = getEncodeStr(file.getPath());
        } catch (Exception e) {
            LOGGER.error("getFileByByte 解析异常:{}", e);
        }
        if (StringUtils.equals(encodeType, "UTF8")) {//utf-8 带BOM的去掉前三位
            String str = "";
            try {
                deleteUTF8BOM(file);
                file.renameTo(file);
                str = readFile2String(file, "UTF-8");
            } catch (Exception e) {
                LOGGER.error("getFileByByte2 解析异常:{}", e);
            }
            return str;
        }
        return new String(fileBytes, "UTF-8");
    }

    public static void deleteUTF8BOM(File file) {
        try {
            byte[] bom = {-17, -69, -65};
            byte[] fileByte = org.apache.commons.io.FileUtils.readFileToByteArray(file);
            if ((fileByte != null) && (fileByte.length >= 3) &&
                    (fileByte[0] == bom[0]) && (fileByte[1] == bom[1]) && (fileByte[2] == bom[2]))
                org.apache.commons.io.FileUtils.writeByteArrayToFile(file, Arrays.copyOfRange(fileByte, 3, fileByte.length));
        } catch (Exception e) {
            LOGGER.error("deleteUTF8BOM异常!{}", e);
        }

    }

    public static String readFile2String(File file, String encoding) {
        String data = null;
        try {
            data = FileUtils.readFileToString(file, encoding);
        } catch (IOException e) {
            LOGGER.error("readFile2String解析异常:{}", e);
        }
        return data;
    }

    public static String getEncodeStr(String fullFileName) {
        String encodeType = "";
        BufferedInputStream bis = null;
        FileInputStream fi = null;
        try {
            fi = new FileInputStream(fullFileName);
            bis = new BufferedInputStream(fi);
            byte[] head = new byte[3];
            int line = 0;
            while ((line = bis.read(head)) > 0) {
            }
            if (head[0] == -17 && head[1] == -69 && head[2] == -65) { //带BOM
                encodeType = "UTF8";
            }
        } catch (FileNotFoundException e) {
            LOGGER.error("文件不存在:{}", e);
        } catch (IOException e) {
            LOGGER.error("getEncodeStr.io解析异常:{}", e);
        } finally {
            if (bis != null) {
                try {
                    bis.close();
                } catch (IOException e) {
                    LOGGER.error("getEncodeStr解析异常:{}", e);
                }
            }
            if (null != fi) {
                try {
                    fi.close();
                } catch (IOException e) {
                    LOGGER.error("getEncodeStr.IOException 解析异常:{}", e);
                }
            }
        }
        return encodeType;
    }

    /**
     * 陕西需求转换qxdm工具
     * @param qhdm
     * @param bz
     * @return
     */
    public static String getExhangeQxdm(String qhdm, String bz) {
        String exchangeQxdm = "";
        if (StringUtils.isNotBlank(qhdm)) {
            String[] qhdms = qhdm.split("\\|");
            for (int i = 0; i < qhdms.length; i++) {
                String qhdmStr = qhdms[i];
                if (StringUtils.isNotBlank(bz) && bz.indexOf(qhdmStr) > -1) {
                    exchangeQxdm = qhdmStr;
                    break;
                }
            }
        }
        return exchangeQxdm;
    }
}
