package cn.gtmap.realestate.supervise.server.quartz.es;

import cn.gtmap.estateplat.core.ex.AppException;
import cn.gtmap.realestate.supervise.server.es.SearchService;
import cn.gtmap.realestate.supervise.server.utils.XmlUtil;
import cn.gtmap.realestate.supervise.utils.FileUtil;
import cn.gtmap.realestate.supervise.utils.XmlStemFilter;
import com.gtis.config.AppConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Configurable;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * @author <a href="mailto:gaofeng@gtmap.cn">gaofeng</a>
 * @version 1.0, 2017/2/13
 * @description 定时器处理中心端报文到ES
 */
@Component
@Configurable
public class EsScheduledTasks {


    private static final Logger LOGGER = LoggerFactory.getLogger(EsScheduledTasks.class);


    /**
     * @author <a href="mailto:tianjian@gtmap.cn">tianjian</a>
     * @description 成功报文存放路径
     */
    String successResponsePath = AppConfig.getProperty("supervise.response.success.path");

    /**
     * @author <a href="mailto:tianjian@gtmap.cn">tianjian</a>
     * @description 失败报文存放路径
     */
    String failResponsePath = AppConfig.getProperty("supervise.response.fail.path");

    /**
     * @author <a href="mailto:tianjian@gtmap.cn">tianjian</a>
     * @description 消息存放路径
     */
    String messagePath = AppConfig.getProperty("supervise.xml.save.path");

    String areaCode = AppConfig.getProperty("supervise.area.code");

    @Autowired
    SearchService searchService;

    /**
     * @param
     * @return void
     * @author <a href="mailto:gaofeng@gtmap.cn">gaofeng</a>
     * @description 定时获取RabbitMq 服务器信息报文信息到ES
     */
    public void handleEsInfos() {

        if (null != areaCode && !areaCode.isEmpty()) {

            String[] code = areaCode.split(",");

            for (int i = 0; i < code.length; i++) {
                String tempCode = code[i];
                initFailResFiles(tempCode);
                initMessageFiles(tempCode);
                initSuccessRepFiles(tempCode);
            }

        }

    }


    /**
     * 初始化失败响应报文
     *
     * @return
     */
    private void initFailResFiles(String areaCodeStr) {

        List<File> failRepFiles = getFiles(failResponsePath + areaCodeStr);
        if (null != failRepFiles && !failRepFiles.isEmpty()) {
            for (File failFile : failRepFiles) {
                String fileName = failFile.getName();
                String content = null;
                try {
                    content = getFileByByte(failFile);
                } catch (UnsupportedEncodingException e) {
                    LOGGER.error("---------EsScheduledTasks.initFailResFiles.UnsupportedEncodingException in----");
                    throw new AppException(e, 2006);
                }
                saveResInfoToEs(content, fileName);
                File file = new File(failResponsePath + areaCodeStr + "//" + fileName);
                boolean flag = false;
                if (file.exists()) {
                    flag = file.delete();
                }
                if (flag) {
                    LOGGER.info("=============删除文件成功======");
                }

            }
        }
    }

    /**
     * 初始化业务报文
     *
     * @return
     */
    private void initSuccessRepFiles(String areaCodeStr) {

        List<File> sucessRepFiles = getFiles(successResponsePath + areaCodeStr);
        if (null != sucessRepFiles && !sucessRepFiles.isEmpty()) {
            for (File sucessFile : sucessRepFiles) {
                String fileName = sucessFile.getName();
                String content = null;
                try {
                    content = getFileByByte(sucessFile);
                } catch (UnsupportedEncodingException e) {
                    LOGGER.error("---------EsScheduledTasks.initSuccessRepFiles.UnsupportedEncodingException in----");
                    throw new AppException(e, 2006);
                }
                saveResInfoToEs(content, fileName);
                File file = new File(successResponsePath + areaCodeStr + "\\" + fileName);
                boolean flag = false;
                if (file.exists()) {
                    flag = file.delete();
                }
                if (flag) {
                    LOGGER.info("=============删除文件成功======");
                }

            }
        }
    }

    /**
     * 初始化成功响应报文
     *
     * @return
     */
    private void initMessageFiles(String areaCodeStr) {

        List<File> messageFiles = getFiles(messagePath + areaCodeStr);

        if (null != messageFiles && !messageFiles.isEmpty()) {
            for (File messageFile : messageFiles) {
                String fileName = messageFile.getName();
                String content = null;
                try {
                    content = getFileByByte(messageFile);
                } catch (UnsupportedEncodingException e) {
                    LOGGER.error("---------EsScheduledTasks.initMessageFiles.UnsupportedEncodingException in----");
                    throw new AppException(e, 2006);
                }
                Map<String, Object> json = new HashMap<>();
                json.put("fileName", fileName);
                json.put("message", content);
                searchService.mainCreateIndex(json, "");
                File file = new File(messagePath + areaCodeStr + "\\" + fileName);
                boolean flag = false;
                if (file.exists()) {
                    flag = file.delete();
                }
                if (flag) {
                    LOGGER.info("=============删除文件成功======");
                }
            }
        }
    }

    /**
     * @param path
     * @return
     */
    public List<File> getFiles(String path) {
        List<File> filess = new ArrayList<File>();
        File file = new File(path);
        if (file.exists()) {
            XmlStemFilter xmlStemFilter = new XmlStemFilter();
            File[] files = file.listFiles();
            if (null == files || files.length == 0) {
                LOGGER.info("----文件夹是空的!");
                return null;
            } else {
                for (File file2 : files) {
                    if (file2.isDirectory()) {
                        String tempPath = file2.getAbsolutePath();
                        List<File> fileList = getFiles(tempPath);
                        if (null != fileList && !fileList.isEmpty()) {
                            for (File fileTemp : fileList) {
                                if (xmlStemFilter.accept(fileTemp)) {
                                    filess.add(fileTemp);
                                }

                            }

                        }

                    } else {
                        if (xmlStemFilter.accept(file2)) {
                            filess.add(file2);
                        }

                    }
                }
            }
        } else {
            file.mkdirs();
        }
        return filess;
    }


    /**
     * @param file:需要转换的文件
     * @return 返回文件的字符串
     * @author <a href="mailto:tianjian@gtmap.cn">tianjian</a>
     * @description 将文件获取为UTF-8编码的字符串
     */
    private String getFileByByte(File file) throws UnsupportedEncodingException {
        byte[] fileBytes = new byte[0];
        fileBytes = FileUtil.getBytes(file);
        return new String(fileBytes, "UTF-8");
    }


    /**
     * 保存响应报文到es
     *
     * @param resInfo
     * @param fileName
     */
    public void saveResInfoToEs(String resInfo, String fileName) {
        Map<String, Object> json = new HashMap<>();
        json.put("fileName", fileName);
        json.put("message", resInfo);
        String respCode = null;
        try {
            respCode = XmlUtil.getTextByXpath("/responseMessage/responseCode", resInfo);
            if (null == respCode || respCode.isEmpty()) {
                respCode = XmlUtil.getTextByXpath("/respond/ResponseCode", resInfo);
            }
        } catch (Exception e) {
            LOGGER.error("---------EsScheduledTasks.saveResInfoToEs.Exception in--------");
            throw new AppException(e, 2005);
        }
        json.put("respCode", respCode);
        searchService.mainCreateIndex(json, "");
    }

}

