package cn.gtmap.realestate.supervise.server.common.impl.check;

import cn.gtmap.realestate.supervise.model.FileMessage;
import cn.gtmap.realestate.supervise.model.MessageClient;
import cn.gtmap.realestate.supervise.server.common.impl.AbstractCheckMessage;
import cn.gtmap.realestate.supervise.server.common.impl.CheckResultManage;
import cn.gtmap.realestate.supervise.server.model.XmlValidateResult;
import cn.gtmap.realestate.supervise.server.utils.XsdUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.xml.sax.ErrorHandler;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import javax.xml.XMLConstants;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;
import java.io.*;
import java.nio.charset.Charset;
import java.util.List;

/**
 * @author <a href="mailto:tianjian@gtmap.cn">tianjian</a>
 * @version 1.0, 2017/7/4
 * @description xsd校验
 */
@Service
public class XsdCheck extends AbstractCheckMessage implements ErrorHandler {

    private static final Logger LOGGER = LoggerFactory.getLogger(XsdCheck.class);

    // 错误信息
    private String ERRORMSG = "";

    /**
     * @author <a href="mailto:tianjian@gtmap.cn">tianjian</a>
     * @description 存放检查结果的工具类
     */
    @Autowired
    CheckResultManage checkResultManage;

    /**
     * 初始化该验证是否对后面的验证有影响，如果该验证不通过不进行下面的验证设置为true比如用户验证
     */
    public XsdCheck() {
        super(true, "200103");
    }


    /**
     * @param messages 需要验证的文件
     * @return 验证结果的封装类
     * @author <a href="mailto:tianjian@gtmap.cn">tianjian</a>
     * @description 验证用户的xml文档和业务规范xsd文件是否一致
     */
    @Override
    public CheckResultManage checkMessage(MessageClient messages) {
        List<FileMessage> fileMessages = messages.getMessageContents();
        for (FileMessage fileMessage : fileMessages) {
            XmlValidateResult result = CheckXmlFile(fileMessage.toString(), fileMessage.getRectype(), fileMessage.getFileName());
            if (!result.isValidated()) {
                checkResultManage.putErrorCheckResult(fileMessage, "2023:" + result.getErrorMsg());
            }
        }
        return checkResultManage;
    }


    /**
     * @param fileName:文件名称
     * @param serverNumber:业务类型
     * @param fileString:报文内容
     * @return 检查结果
     * @author <a href="mailto:tianjian@gtmap.cn">tianjian</a>
     * @description xsd检查的返回封装类
     */
    private XmlValidateResult CheckXmlFile(String fileString, String serverNumber, String fileName) {

        String xsdString = XsdUtil.getTemplateByRecType(serverNumber);
        if (xsdString == null) {
            LOGGER.info("XML文件名称为: " + fileName + " ,未发现可以验证它的XSD文件");
            return null;
        }

        XmlValidateResult result = checkXmlByXsd(fileString, xsdString);
        if (!result.isValidated()) {
            LOGGER.info("XML文件名称为: " + fileName + " 验证失败, 原因如下: " + result.getErrorMsg());
        }
        return result;
    }


    @Override
    public void warning(SAXParseException exception) throws SAXException {
        if (ERRORMSG.length() < 1000000) {
            ERRORMSG += "警告：" + exception + "\n";
        } else {
            ERRORMSG += "提示：错误信息过长，此处忽略......" + "\n";
        }

    }

    @Override
    public void error(SAXParseException exception) throws SAXException {
        if (ERRORMSG.length() < 1000000) {
            ERRORMSG += "错误：" + exception + "\n";
        } else {
            ERRORMSG += "提示：错误信息过长，此处忽略......" + "\n";
        }
    }

    @Override
    public void fatalError(SAXParseException exception) throws SAXException {
        if (ERRORMSG.length() < 1000000) {
            ERRORMSG += "严重：" + exception + "\n";
        } else {
            ERRORMSG += "提示：错误信息过长，此处忽略......" + "\n";
        }
    }

    /**
     * @param xmlStr 需要验证的xml字符串
     * @param xsdStr 验证xml的xsd字符串
     * @return 包含验证结果的对象
     */
    public XmlValidateResult checkXmlByXsd(String xmlStr, String xsdStr) {
        ERRORMSG = "";
        // 创建返回值类，默认为失败
        XmlValidateResult vs = new XmlValidateResult();
        String schemaLanguage = XMLConstants.W3C_XML_SCHEMA_NS_URI;
        SchemaFactory schemaFactory = SchemaFactory.newInstance(schemaLanguage);

        try {
            // 解析XML的RecType 获取XSD文件
            // 解析XML的RecType 获取XSD文件
            Reader xsdReader = new BufferedReader(new StringReader(xsdStr));
            Source xsdSource = new StreamSource(xsdReader);
            InputStream inputSource = new ByteArrayInputStream(xmlStr.getBytes(Charset.forName("UTF-8")));
            Source source = new StreamSource(inputSource);
            Schema schema = schemaFactory.newSchema(xsdSource);
            Validator validator = schema.newValidator();
            validator.setErrorHandler(this);
            // 验证
            validator.validate(source);
            vs.setValidated(true);
        } catch (SAXException e) {
            LOGGER.error("error:", e);
            vs.setValidated(false);
        } catch (Exception e) {
            LOGGER.error("error:", e);
            vs.setValidated(false);
        }
        if (StringUtils.isNotBlank(ERRORMSG)) {
            vs.setErrorMsg(ERRORMSG);
            vs.setValidated(false);
        }
        return vs;
    }
}
