/**
 *  自定义漫游
 * @author by <a href="mailto:chaomashi@gmail.com">Du</a>
 * @version v1.0 2019/12/26/14:52  Copyright gtmap Corp
 */
define([
    'global/utils/MapUtils',
    'knockout',
    'Cesium'
], function (MapUtils, ko, Cesium) {

    var flagName;
    // 所有按钮
    var roamTypes = [
        {
            name: 'moveForward',
            icon: 'icon-f-move-forward',
            label: '向前漫游'
        },
        {
            name: 'moveUp',
            icon: 'icon-f-move-up',
            label: '向上漫游'
        },
        {
            name: 'moveBackward',
            icon: 'icon-f-move-backward',
            label: '向后漫游'
        },
        {
            name: 'moveLeft',
            icon: 'icon-f-move-left',
            label: '向左漫游'
        },
        {
            name: 'rotateLeft',
            icon: 'icon-f-reset',
            label: '旋转'
        },
        {
            name: 'moveRight',
            icon: 'icon-f-move-right',
            label: '向右漫游'
        },
        {
            name: 'Stop',
            icon: 'icon-f-move-stop',
            label: '停止'
        },
        {
            name: 'moveDown',
            icon: 'icon-f-move-down',
            label: '向下漫游'
        }
    ];

    var flags = {
        moveForward: false,
        moveBackward: false,
        moveUp: false,
        moveDown: false,
        moveLeft: false,
        moveRight: false,
        rotateLeft: false
    };

    var instance,
        me = {
            // init fun
            init: function () {
                me.startEvent();
            },
            // open event
            onOpen: function () {
                me.startEvent();
            },
            // close event
            onClose: function () {
                me.stopEvent();
            },
            // pause event
            onPause: function () {
                me.stopEvent();
            },

            startEvent: function () {
                var viewer = MapUtils.viewer,
                    handler = new Cesium.ScreenSpaceEventHandler(viewer.scene.canvas);

                $("body").on('keydown', me.onKeydown);
                $("body").on('keyup', me.onKeyup);
                viewer.clock.onTick.addEventListener(me.onClockOnTickEvent);

                // 右键停止
                handler.setInputAction(function (movement) {
                    removeClass();
                    flagName = undefined;
                }, Cesium.ScreenSpaceEventType.RIGHT_CLICK);

                me.handler = handler;
            },

            stopEvent: function () {
                $("body").off('keydown', me.onKeydown);
                $("body").off('keyup', me.onKeyup);
                MapUtils.viewer.clock.onTick.removeEventListener(me.onClockOnTickEvent);
                me.handler.removeInputAction(Cesium.ScreenSpaceEventType.RIGHT_CLICK);
            },

            // on key down event
            onKeydown: function (e) {
                flagName = getFlagForKeyCode(e.keyCode);
                if (typeof flagName !== 'undefined') {
                    flags[flagName] = true;
                }
            },

            // on key up event
            onKeyup: function (e) {
                flagName = getFlagForKeyCode(e.keyCode);
                if (typeof flagName !== 'undefined') {
                    flags[flagName] = false;
                }
            },

            onClockOnTickEvent: function (clock) {

                var viewer = MapUtils.viewer;
                var camera = viewer.camera,
                    cameraHeight = viewer.scene.globe.ellipsoid.cartesianToCartographic(camera.position).height,
                    // cameraHeight = position.height,
                    moveRate = cameraHeight / 100.0;

                if (flagName && flags[flagName]) {
                    flagName === "rotateLeft" ? rotateLeft() : camera[flagName](moveRate);
                }

                function rotateLeft() {
                    var delTime = Cesium.JulianDate.secondsDifference(viewer.clock.currentTime, viewer.clock.startTime);
                    var heading = Cesium.Math.toRadians(delTime * 360 / 36) + me.initialHeading;
                    var center = me.center;
                    viewer.scene.camera.setView({
                        destination: Cesium.Cartesian3.fromDegrees(center.longitude, center.latitude, center.height),
                        orientation: {
                            heading: heading,
                            pitch: me.pitch
                        }
                    });

                    if (Cesium.JulianDate.compare(viewer.clock.currentTime, viewer.clock.stopTime) >= 0) {
                        viewer.clock.onTick.removeEventListener(me.onClockOnTickEvent());
                    }
                }
            }
        };

    // 获取按键code
    function getFlagForKeyCode(keyCode) {
        switch (keyCode) {
            case 'W'.charCodeAt(0):
                return 'moveForward';
            case 'S'.charCodeAt(0):
                return 'moveBackward';
            case 'Q'.charCodeAt(0):
                return 'moveUp';
            case 'E'.charCodeAt(0):
                return 'moveDown';
            case 'D'.charCodeAt(0):
                return 'moveRight';
            case 'A'.charCodeAt(0):
                return 'moveLeft';
            default:
                return undefined;
        }
    }

    // 注册id
    ko.applyBindings(new RoamViewModel(), document.getElementById("roamGlobal"));

    function RoamViewModel() {
        var self = this;

        self.roamTypes = ko.observableArray(roamTypes);

        // 激活组件
        self.activeRoam = function (type) {
            var name = type.name;
            if (flagName !== name && name !== 'Stop') {
                $('#' + name).addClass('active-roam');
                removeClass();
            }

            if (name === 'Stop') {
                removeClass();
                flagName = undefined;
            } else {
                flagName = type.name;
                flags[flagName] = true;
                me.center = getCameraCenter();
            }
        }
    }

    function getCameraCenter() {
        var viewer = MapUtils.viewer,
            position = MapUtils.viewer.scene.globe.ellipsoid.cartesianToCartographic(viewer.camera.position),
            longitude = Cesium.Math.toDegrees(position.longitude),
            latitude = Cesium.Math.toDegrees(position.latitude),
            height = position.height;

        var startTime = Cesium.JulianDate.fromDate(new Date());
        viewer.clock.startTime = startTime.clone();
        viewer.clock.currentTime = startTime.clone();
        viewer.clock.clockRange = Cesium.ClockRange.CLAMPED;
        viewer.clock.clockStep = Cesium.ClockStep.SYSTEM_CLOCK;
        me.initialHeading = viewer.camera.heading;
        me.pitch = viewer.camera.pitch;
        return {longitude: longitude, latitude: latitude, height: height}
    }

    function removeClass() {
        if ($('#' + flagName).hasClass('active-roam')) {
            $('#' + flagName).removeClass('active-roam');
        }
    }

    /**
     * get instance
     *
     * @returns {*}
     */
    me.getInstance = function () {
        if (void 0 === instance) {
            instance = me;
        }
        return instance;
    };
    return me;
});