define([
    'Cesium',
    'lodash',
    "global/core/Popup",
    "text!global/utils/NearbySearchUtils/Index.html",
    "css!global/utils/NearbySearchUtils/Index"
], function (Cesium, _, Popup, html) {
    var $that;

    function NearbySearchUtils() {
        $that = this;
        // 用于在移除历史搜索范围圈时不会再次触发画圈事件的标记
        $that._surroundEnabled = false;
        $that._viewer = undefined;
        // 周边搜用到的鼠标监听
        $that._surroundHandler = undefined;
        // 周边搜用到的配置，一般配置于tpl文件，通过初始化获取
        $that._surroundOptions = undefined;
        // 周边搜用到的中心点
        $that._surroundPoint = {
            lon: undefined,
            lat: undefined,
            name: undefined
        };
        // 更改周边搜范围popup标记
        $that._surroundPopup = undefined;
        // 周边搜范围entity
        $that._resultSurroundRange = undefined;
        // 跟随鼠标动态绘制周边搜范围entity
        $that._activeSurroundRange = undefined;
        // 鼠标按下状态标记
        $that._leftdown = false;
        // 搜索框jQuery
        $that._searchBarElem = undefined;
        // 用于搜索的回调函数
        $that._callback = undefined;

        // 初始化状态
        $that._inited = false;
    }

    Object.defineProperties(NearbySearchUtils.prototype, {
        surroundOptions: {
            get: function () {
                return $that._surroundOptions;
            },
            set: function (val) {
                $that._surroundOptions = val;
            }
        },
        surroundPoint: {
            get: function () {
                return $that._surroundPoint;
            }
        },
        inited: {
            get: function () {
                return $that._inited;
            }
        }
    });

    /**
     * 周边搜功能初始化
     * @param elem 需要添加周边搜内容的jQuery对象
     * @param option 周边搜配置，一般配置于tpl文件
     * @param data 周边搜数据，一般包含坐标，搜索关键词(name)
     * @param viewer
     * @param callback 用于搜索的回调函数(可选)
     * @param searchBarElem 搜索框元素jQuery对象(可选)
     */
    NearbySearchUtils.prototype.init = function (elemid, option, data, viewer, callback, searchBarElemid) {
        // 移除页面上已经存在的周边搜内容，保证唯一性
        if ($that._inited) {
            $that.destroy();
        }
        if (elemid) {
            if (option && data && viewer) {
                // viewer
                $that._viewer = viewer;
                // 暂存周边搜配置
                $that._surroundOptions = option;
                // 经纬度坐标
                $that._surroundPoint.lon = data.lon;
                $that._surroundPoint.lat = data.lat;
                // 周边搜范围，初始化为配置中的默认范围
                $that._surroundPoint.distance = option.defaultDistance;
                // 周边搜用到的鼠标监听
                $that._surroundHandler = new Cesium.ScreenSpaceEventHandler($that._viewer.scene.canvas);
                // 周边搜关键词
                $that._surroundPoint.name = data.name;
                if (data.pageSize) {
                    $that._surroundPoint.pageSize = data.pageSize;
                }
                if (data.pageIndex) {
                    $that._surroundPoint.pageIndex = data.pageIndex;
                }
                // 周边搜类别id
                $that._surroundPoint.typeId = undefined;
                if (searchBarElemid) {
                    $that._searchBarElem = "#" + searchBarElemid;
                }
                // 用于搜索的回调函数
                $that._callback = callback;

                // 添加周边搜内容
                $("#" + elemid).append(html);
                //绑定点击事件
                initClickEvent();

                // 初始化完毕标记
                $that._inited = true;
            } else {
                console.log("缺少必要配置。");
            }
        } else {
            console.log("绑定元素失败。");
        }
    };

    /**
     * 清除页面上的周边搜内容
     */
    NearbySearchUtils.prototype.destroy = function () {
        if ($that._inited) {
            // 移除页面上已经存在的周边搜内容
            $("#surroundSearch").remove();

            // 防止清除历史周边搜时触发新的搜索
            $that._surroundEnabled = false;

            // 删除历史周边搜范围
            if ($that._resultSurroundRange) {
                $that._viewer.entities.remove($that._resultSurroundRange);
                $that._resultSurroundRange = undefined;
            }
            if ($that._activeSurroundRange) {
                $that._viewer.entities.remove($that._activeSurroundRange);
                $that._activeSurroundRange = undefined;
            }

            // 关闭监听事件
            closeMouseEvent($that._surroundHandler);

            // 清除改变周边搜范围标签
            if ($that._surroundPopup) {
                $($that._surroundPopup).trigger('click');
                $that._surroundPopup = undefined;
            }

            // 重置周边搜用到的中心点
            $that._surroundPoint = {
                lon: undefined,
                lat: undefined,
                name: undefined
            };

            // 重置鼠标按下状态标记
            $that._leftdown = false;

            // 重置搜索框元素记录
            $that._searchBarElem = undefined;

            // 重置初始化状态
            $that._inited = false;
        }
    };

    /**
     * 绑定相关事件
     */
    function initClickEvent() {
        $(document).off('click', '.surroundSearchAll');
        $(document).off('click', '.search-surround-border');
        $(document).on('click', '.surroundSearchAll', clickEvent);
        $(document).on('click', '.search-surround-border', clickEvent);
    }

    /**
     * 周边搜索点击事件
     */
    function clickEvent() {
        // 防止清除历史周边搜时触发新的搜索
        $that._surroundEnabled = false;

        // 删除历史周边搜范围
        if ($that._resultSurroundRange) {
            $that._viewer.entities.remove($that._resultSurroundRange);
            $that._resultSurroundRange = undefined;
        }
        if ($that._activeSurroundRange) {
            $that._viewer.entities.remove($that._activeSurroundRange);
            $that._activeSurroundRange = undefined;
        }

        // 关闭监听事件
        closeMouseEvent($that._surroundHandler);

        // 清除改变周边搜范围标签
        if ($that._surroundPopup) {
            $($that._surroundPopup).trigger('click');
            $that._surroundPopup = undefined;
        }

        $that._surroundEnabled = true;

        // 重置搜索范围为默认范围
        $that._surroundPoint.distance = $that._surroundOptions.defaultDistance;

        // 设置搜索类型
        var typeId = $(this).attr("typeId");
        if (typeof typeId !== undefined && typeId !== false) {
            $that._surroundPoint.typeId = $(this).attr("typeId");
            $that._surroundPoint.type = $(this).find('span').html();
        } else {
            $that._surroundPoint.typeId = undefined;
            $that._surroundPoint.type = undefined;
        }

        // 绘制默认周边搜范围
        $that._resultSurroundRange = $that._viewer.entities.add({
            position: Cesium.Cartesian3.fromDegrees($that._surroundPoint.lon, $that._surroundPoint.lat),
            ellipse: {
                semiMinorAxis: $that._surroundPoint.distance,
                semiMajorAxis: $that._surroundPoint.distance,
                material: Cesium.Color.BLUE.withAlpha(0.3)
            }
        });

        // 飞到周边搜范围
        var promise = $that._viewer.zoomTo($that._resultSurroundRange);

        // 调用搜索
        if ($that._callback && typeof $that._callback === "function") {
            $that._callback($that._surroundPoint, $that._surroundOptions);
        } else {
            if ($that._searchBarElem) {
                if ($that._surroundPoint.type) {
                    $($that._searchBarElem).val("在 " + $that._surroundPoint.name + " 周边搜索 " + $that._surroundPoint.type);
                } else {
                    $($that._searchBarElem).val("在 " + $that._surroundPoint.name + " 周边搜索");
                }
            }
            // SearchResultUtils._searchData({
            //     instanceId: $that._surroundOptions.instanceId,
            //     lon: $that._surroundPoint.lon,
            //     lat: $that._surroundPoint.lat,
            //     distance: Math.round($that._surroundPoint.distance),
            //     pageSize: $that._surroundPoint.pageSize,
            //     pageIndex: $that._surroundPoint.pageIndex
            // }, $that._surroundOptions.url);
        }

        // 绘制周边搜范围标签，并绑定改变周边搜范围事件
        promise.then(function () {
            setTimeout(function () {
                // 周边搜范围中心转屏幕坐标
                var center = Cesium.SceneTransforms.wgs84ToWindowCoordinates(
                    $that._viewer.scene,
                    Cesium.Cartesian3.fromDegrees($that._surroundPoint.lon, $that._surroundPoint.lat)
                );
                // 改变周边搜范围标签id
                var tempId = _.uniqueId() + "Draw";
                // 创建改变周边搜范围标签
                Popup.open({
                    content: '<div class="draw-measure-pop"> <span>' + Math.round($that._surroundPoint.distance) + '</span>' +
                        '<span class="icon-f-dragbox-fullscreen measure-close-icon defaultSurroundRange" id="' + tempId + '"></span></div>',
                    position: {
                        x: center.x + 200 < $(document).width() ? (center.x + 200 > 0 ? center.x + 200 : 0) : $(document).width() - 100,
                        y: center.y < $(document).height() ? (center.y > 0 ? center.y : 0) : $(document).height() - 100
                    },
                    offset: "top",
                    hasTriangle: false,
                    hasClose: false
                });
                // 存储周边搜范围标签id
                $that._surroundPopup = '#' + tempId;
                // 注册改变周边搜范围事件
                $($that._surroundPopup).on('click', changeSurroundRange);
            }, 500);
        });
    }

    /**
     * 改变周边搜范围事件
     * @param e 事件对象
     */
    function changeSurroundRange(e) {
        // 清除旧Popup和存储的id
        Popup.close(e);
        $that._surroundPopup = undefined;

        //清除旧监听事件
        closeMouseEvent($that._surroundHandler);

        // 清除旧周边搜范围
        if ($that._resultSurroundRange) {
            $that._viewer.entities.remove($that._resultSurroundRange);
            $that._resultSurroundRange = undefined;
        }
        if ($that._activeSurroundRange) {
            $that._viewer.entities.remove($that._activeSurroundRange);
            $that._activeSurroundRange = undefined;
        }

        // 绘制新周边搜范围及绑定监听事件
        if ($that._surroundEnabled) {
            // 初始化动态范围为旧范围
            $that._surroundPoint.dynamicDistance = $that._surroundPoint.distance;
            // 旧范围转为范围的动态监听，用于实时动态绘制范围
            $that._surroundPoint.distance = new Cesium.CallbackProperty(function () {
                return $that._surroundPoint.dynamicDistance;
            }, false);
            // 创建动态范围entity
            $that._activeSurroundRange = $that._viewer.entities.add({
                position: Cesium.Cartesian3.fromDegrees($that._surroundPoint.lon, $that._surroundPoint.lat),
                ellipse: {
                    semiMinorAxis: $that._surroundPoint.distance,
                    semiMajorAxis: $that._surroundPoint.distance,
                    material: Cesium.Color.BLUE.withAlpha(0.3)
                }
            });
            // 判断并记录鼠标状态，是否进入了长按左键的拖动状态
            $that._surroundHandler.setInputAction(function (mouseMove) {
                $that._leftDown = true;
            }, Cesium.ScreenSpaceEventType.LEFT_DOWN);
            $that._surroundHandler.setInputAction(function (mouseMove) {
                $that._leftDown = false;
            }, Cesium.ScreenSpaceEventType.LEFT_UP);

            // 允许拖动缩放地图方便调整周边搜范围
            $that._surroundHandler.setInputAction(function (mouseMove) {
                // 动态范围跟随鼠标移动实时动态更新
                if (!$that._leftDown) {
                    if ($that._surroundPoint.lon) {
                        // 鼠标所在位置
                        var newPosition = $that._viewer.scene.pickPosition(mouseMove.endPosition);
                        if (Cesium.defined(newPosition)) {
                            // 更新动态范围
                            $that._surroundPoint.dynamicDistance = Cesium.Cartesian3.distance(
                                Cesium.Cartesian3.fromDegrees($that._surroundPoint.lon, $that._surroundPoint.lat),
                                newPosition
                            );
                        }
                    }
                }
                // 生成跟随鼠标移动的提示popup窗
                mouseMovePop(mouseMove.endPosition);
            }, Cesium.ScreenSpaceEventType.MOUSE_MOVE);

            // 确定新搜索范围
            $that._surroundHandler.setInputAction(function (click) {
                // 关闭监听事件
                closeMouseEvent($that._surroundHandler);
                $that._surroundPoint.distance = Cesium.Cartesian3.distance(
                    Cesium.Cartesian3.fromDegrees($that._surroundPoint.lon, $that._surroundPoint.lat),
                    $that._viewer.scene.pickPosition(click.position));
                $that._resultSurroundRange = $that._viewer.entities.add({
                    position: Cesium.Cartesian3.fromDegrees($that._surroundPoint.lon, $that._surroundPoint.lat),
                    ellipse: {
                        semiMinorAxis: $that._surroundPoint.distance,
                        semiMajorAxis: $that._surroundPoint.distance,
                        material: Cesium.Color.BLUE.withAlpha(0.3)
                    }
                });
                // 清除动态范围entity
                if ($that._activeSurroundRange) {
                    $that._viewer.entities.remove($that._activeSurroundRange);
                    $that._activeSurroundRange = undefined;
                }
                // 关闭悬浮提示
                var popId = document.getElementById("drawMouseMove");
                popId.style.display = "none";

                // 调用搜索
                if ($that._callback && typeof $that._callback === "function") {
                    $that._callback($that._surroundPoint, $that._surroundOptions);
                } else {
                    if ($that._searchBarElem) {
                        if ($that._surroundPoint.type) {
                            $that._searchBarElem.val("在 " + $that._surroundPoint.name + " 周边搜索 " + $that._surroundPoint.type);
                        } else {
                            $that._searchBarElem.val("在 " + $that._surroundPoint.name + " 周边搜索");
                        }
                    }
                    // SearchResultUtils._searchData({
                    //     instanceId: $that._surroundOptions.instanceId,
                    //     lon: $that._surroundPoint.lon,
                    //     lat: $that._surroundPoint.lat,
                    //     distance: Math.round($that._surroundPoint.distance),
                    //     pageSize: $that._surroundPoint.pageSize,
                    //     pageIndex: 1
                    // }, $that._surroundOptions.url);
                }

                // 创建改变周边搜范围标签
                var tempId = _.uniqueId() + "Draw";
                Popup.open({
                    content: '<div class="draw-measure-pop"> <span>' + Math.round($that._surroundPoint.distance) + 'm</span>' +
                        '<span class="icon-f-dragbox-fullscreen measure-close-icon" id="' + tempId + '"></span></div>',
                    position: click.position,
                    offset: "top",
                    hasTriangle: false,
                    hasClose: false
                });
                $that._surroundPopup = '#' + tempId;

                // 绑定改变周边搜范围事件
                $($that._surroundPopup).on('click', changeSurroundRange);
            }, Cesium.ScreenSpaceEventType.RIGHT_CLICK);
        }
    }

    /**
     * 清除周边搜监听事件
     * @param handler Cesium屏幕监听handler
     */
    function closeMouseEvent(handler) {
        handler.removeInputAction(Cesium.ScreenSpaceEventType.LEFT_CLICK);
        handler.removeInputAction(Cesium.ScreenSpaceEventType.MOUSE_MOVE);
        handler.removeInputAction(Cesium.ScreenSpaceEventType.RIGHT_CLICK);
        handler.removeInputAction(Cesium.ScreenSpaceEventType.LEFT_DOUBLE_CLICK);
        handler.removeInputAction(Cesium.ScreenSpaceEventType.LEFT_DOWN);
        handler.removeInputAction(Cesium.ScreenSpaceEventType.LEFT_UP);
        // 关闭悬浮事件
        var popId = document.getElementById("drawMouseMove");
        if (popId) popId.style.display = "none";
    }

    /**
     * 动态绘制周边搜范围鼠标悬浮pop窗
     * @param position 鼠标位置
     * @param type
     */
    function mouseMovePop(position) {
        var popId = document.getElementById("drawMouseMove");
        if (!popId) {
            var popElement = document.createElement("div");
            popElement.innerHTML = '<div class="draw-mouse-move-popup" id="drawMouseMove">右击确定范围</div>';
            var body = document.getElementsByClassName('cesium-viewer')[0];
            body.appendChild(popElement);
            popId = document.getElementById("drawMouseMove");
        }
        popId.style.display = "block";
        popId.style.left = position.x + 15 + "px";
        popId.style.top = position.y + 15 + "px";
        popId.style.width = "auto";
        popId.style.height = "auto";
        popId.style.backgroundColor = "#ffffff";
        popId.style.position = "absolute";
        popId.style.color = "#7a8188";
        popId.style.border = "1px solid red";
        popId.style.padding = "5px";
    }

    return new NearbySearchUtils;
});