/**
 *  地图通用类
 * @author by <a href="mailto:chaomashi@gmail.com">Du</a>
 * @version v1.0 2019/5/20/10:15  Copyright gtmap Corp
 */

define(['Cesium',
    'lodash',
    'global/core/index',
    'global/utils/WidgetUtils',
    'global/core/ArcGisMapServerImageryProvider',
    'global/core/ArcGisMapServerVectorTileImageryProvider',
    'global/core/ArcGISTiledElevationTerrainProvider',
    'global/core/WebMapTileServiceImageryProvider',
    'global/core/TiandituGeoWTFS',
    'global/core/GeoServerVectorTileImageryProvider',
    'global/core/ArcGisMapServerDynamicImageryProvider'
], function (Cesium,
             _,
             Utils,
             WidgetUtils,
             ArcGisMapServerImageryProvider,
             ArcGisMapServerVectorTileImageryProvider,
             ArcGISTiledElevationTerrainProvider,
             WebMapTileServiceImageryProvider,
             TiandituGeoWTFS,
             GeoServerVectorTileImageryProvider,
             ArcGisMapServerDynamicImageryProvider) {
    'use strict';
    var that, currentVisibleLayers = [];

    /**
     * @export mapUtils
     * @constructor
     */
    function MapUtils() {
        this._mapContainer = Utils.config.GLOBAL_CONTAINER;
        this._viewer = undefined;
        this._baseImageryLayers = undefined;
        this._polygonDrawHandler = undefined;
        this._zoomLevel = undefined;
        this._isShowModel = false;
        this._modelLayer = undefined;
        that = this;
    }

    Object.defineProperties(MapUtils.prototype, {

        mapContainer: {
            get: function () {
                return this._mapContainer;
            }
        },

        viewer: {
            get: function () {
                return this._viewer;
            },
            set: function (value) {
                this._viewer = value
            }
        },

        baseImageryLayers: {
            get: function () {
                return this._baseImageryLayers;
            }
        },

        polygonDrawHandler: {
            get: function () {
                return this._polygonDrawHandler;
            }
        },

        zoomLevel: {
            get: function () {
                return this._zoomLevel;
            }
        },

        isShowModel: {
            get: function () {
                return this._isShowModel;
            },
            set: function (isShow) {
                this._isShowModel = isShow;

                if (!isShow && this._modelLayer.hasOwnProperty('layerName')) {
                    this._viewer.scene.layers.remove(this._modelLayer.layerName);
                    delete this._modelLayer.modelLayerDetails;
                }
            }
        },

        modelLayer: {
            get: function () {
                return this._modelLayer;
            }
        },
        poiLayer: {
            get: function () {
                return this._poiLayer;
            }
        }


    });

    /**
     *  set viewer
     * @param viewer
     */
    MapUtils.prototype.setViewer = function (viewer) {
        if (!Cesium.defined(viewer)) {
            Utils.error('layer options cannot null')
        }
        this._viewer = Cesium.defaultValue(viewer, '');
        this.addMapEvent(this._viewer);
    };


    MapUtils.prototype.addMapEvent = function (viewer) {
        // viewer.scene.imageryLayers.layerAdded.addEventListener(isAdded);
        viewer.scene.imageryLayers.layerRemoved.addEventListener(isRemoved);
        // function isAdded(e) {
        //     alert("a");
        // }
        function isRemoved(layer) {
            for (var i = currentVisibleLayers.length - 1; i >= 0; i--) {
                var tmpLayer = currentVisibleLayers[i];
                var url = "";
                try {
                    url = layer.imageryProvider._resource._url;
                } catch (e) {
                }
                var tmpUrl = tmpLayer.url;
                if (url.indexOf(tmpUrl) != -1) {
                    currentVisibleLayers.splice(i, 1);
                }
            }
        }
    }

    MapUtils.prototype.getCurrentLayers = function () {
        return currentVisibleLayers;
    };

    /**
     *添加当前图层参数
     */
    function addCurrentLayerConfig(_layer) {
        var layer = JSON.parse(JSON.stringify(_layer));
        var isExist = false;
        $(currentVisibleLayers).each(function (i, e) {
            if (!layer.url && layer.layerName == e.layerName) {
                isExist = true;
            }

            if (layer.url && e.url == layer.url) {
                isExist = true;
            }
        })
        if (!isExist) {
            currentVisibleLayers.push(layer);
        }
    }

    /**
     * get
     */
    MapUtils.prototype.getMapView = function () {
        var viewer = this._viewer;
        var camera = viewer.camera;
        var y = Cesium.Math.toDegrees(camera.positionCartographic.latitude);
        var x = Cesium.Math.toDegrees(camera.positionCartographic.longitude);
        var cartesian3 = camera.positionWC;
        var cartographic = Cesium.Ellipsoid.WGS84.cartesianToCartographic(cartesian3);
        var height = cartographic.height;
        //获取角度
        var heading = Cesium.Math.toDegrees(camera.heading).toFixed(2);
        var pitch = Cesium.Math.toDegrees(camera.pitch).toFixed(2);
        var roll = Cesium.Math.toDegrees(camera.roll).toFixed(2);

        return {
            z: height,
            heading: heading,
            pitch: pitch,
            roll: roll,
            x: x,
            y: y
        };
    }


    var entity = null;

    /**
     * set
     */
    MapUtils.prototype.setMapView = function (dt) {
        if (!dt) {
            return;
        }
        var viewer = this._viewer;
        viewerflyToLonLat(dt);

        function viewerflyToLonLat(dt) {
            if (entity) {
                viewer.entities.remove(entity);
            }
            entity = new Cesium.Entity({
                id: 'flyTmp',
                position: Cesium.Cartesian3.fromDegrees(dt.x, dt.y, dt.z),
                point: {
                    pixelSize: 10,
                    color: Cesium.Color.WHITE.withAlpha(0.9),
                    outlineColor: Cesium.Color.WHITE.withAlpha(0.9),
                    outlineWidth: 1
                }
            });
            viewer.entities.add(entity);
            viewer.flyTo(entity, {
                offset: {
                    heading: Cesium.Math.toRadians(dt.heading),
                    pitch: Cesium.Math.toRadians(dt.pitch),
                    range: Cesium.Math.toRadians(dt.roll)
                }
            });
        }

    }

    MapUtils.prototype.getCurrentExtent = function () {
        var viewer = this._viewer;
        // 范围对象
        var extent = {};

        // 得到当前三维场景
        var scene = viewer.scene;

        // 得到当前三维场景的椭球体
        var ellipsoid = scene.globe.ellipsoid;
        var canvas = scene.canvas;

        // canvas左上角
        var car3_lt = viewer.camera.pickEllipsoid(new Cesium.Cartesian2(0, 0), ellipsoid);

        // canvas右下角
        var car3_rb = viewer.camera.pickEllipsoid(new Cesium.Cartesian2(canvas.width, canvas.height), ellipsoid);

        // 当canvas左上角和右下角全部在椭球体上
        if (car3_lt && car3_rb) {
            var carto_lt = ellipsoid.cartesianToCartographic(car3_lt);
            var carto_rb = ellipsoid.cartesianToCartographic(car3_rb);
            extent.xmin = Cesium.Math.toDegrees(carto_lt.longitude);
            extent.ymax = Cesium.Math.toDegrees(carto_lt.latitude);
            extent.xmax = Cesium.Math.toDegrees(carto_rb.longitude);
            extent.ymin = Cesium.Math.toDegrees(carto_rb.latitude);
        }

        // 当canvas左上角不在但右下角在椭球体上
        else if (!car3_lt && car3_rb) {
            var car3_lt2 = null;
            var yIndex = 0;
            do {
                // 这里每次10像素递加，一是10像素相差不大，二是为了提高程序运行效率
                yIndex <= canvas.height ? yIndex += 10 : canvas.height;
                car3_lt2 = viewer.camera.pickEllipsoid(new Cesium.Cartesian2(0, yIndex), ellipsoid);
            } while (!car3_lt2);
            var carto_lt2 = ellipsoid.cartesianToCartographic(car3_lt2);
            var carto_rb2 = ellipsoid.cartesianToCartographic(car3_rb);
            extent.xmin = Cesium.Math.toDegrees(carto_lt2.longitude);
            extent.ymax = Cesium.Math.toDegrees(carto_lt2.latitude);
            extent.xmax = Cesium.Math.toDegrees(carto_rb2.longitude);
            extent.ymin = Cesium.Math.toDegrees(carto_rb2.latitude);
        }

        // 获取高度
        extent.height = Math.ceil(viewer.camera.positionCartographic.height);
        return extent;
    }

    /**
     * 世界坐标转经纬度
     * @param cartographic
     */
    MapUtils.prototype.cartesianToDegrees = function (cartographic) {
        var coord = new Cesium.Cartographic.fromCartesian(cartographic),
            longitude = Cesium.Math.toDegrees(coord.longitude),
            latitude = Cesium.Math.toDegrees(coord.latitude),
            height = coord.height;
        return [longitude, latitude, height]
    };

    /**
     *  get viewer
     */
    MapUtils.prototype.getViewer = function () {
        return this._viewer;
    };

    /**
     * 白模展示
     * @param isShow true show  false hidden
     */
    MapUtils.prototype.showModel = function (isShow) {
        this._isShowModel = isShow;
    };

    /**
     *removeSpmTerrain
     */
    MapUtils.prototype.removeSpmTerrain = function () {
        this._viewer.scene.terrainProvider = new Cesium.EllipsoidTerrainProvider({});
    };

    /**
     * modelLayer
     * @param modelLayer
     */
    MapUtils.prototype.setModelLayer = function (modelLayer) {
        this._modelLayer = modelLayer;
    };

    /**
     * modelLayer
     * @param modelLayer
     */
    MapUtils.prototype.setPoiLayer = function (poiLayer) {
        this._poiLayer = poiLayer;
    };


    /**
     * 截取url
     * @param url
     * @private
     * @return url and params
     */
    MapUtils.prototype.splitUrl = function (url) {
        var _url = _.clone(url);
        var obj = {},
            search = url.substring(url.lastIndexOf("?") + 1),
            reg = /([^?&=]+)=([^?&=]*)/g;
        obj.url = _url.split("?")[0];
        search.replace(reg, function (rs, $1, $2) {
            var name = decodeURIComponent($1);
            var val = decodeURIComponent($2);
            val = String(val);
            obj[name] = val;
            return rs;
        });
        return obj;
    }

    /**
     *  createGlobalLayer
     * @param options
     * @param viewer
     * @returns {ImageryProvider|*}
     */
    MapUtils.prototype.createGlobalLayer = function (options, viewer) {
        viewer = viewer || this._viewer;
        var params = {}, type = options.type, url = options.url;

        var geographicTilingScheme_4490 = Utils.config.tilingScheme['EPSG:4490'];
        addCurrentLayerConfig(options);
        switch (type) {
            case "spm_tile":
                return new Cesium.SuperMapImageryProvider({
                    url: url
                });
            case "ags_rest":
                return new ArcGisMapServerDynamicImageryProvider({
                    url: url,
                    viewer: this.viewer
                });
            case "ags_tile":
                return new ArcGisMapServerImageryProvider(_.assign(options, {
                    url: url,
                    layer: options.name,
                    tilingScheme: geographicTilingScheme_4490,
                    ellipsoid: Utils.config.ellipsolid["EPSG:4490"]
                }));
            case "ags_terr":
                return new ArcGISTiledElevationTerrainProvider({
                    url: url
                });
            case "ags_vt":
                return new ArcGisMapServerVectorTileImageryProvider({
                    url: url,
                    layer: options.name,
                    tilingScheme: geographicTilingScheme_4490,
                    maximumLevel: 16
                });
            case "wmts":
            case "geo_wmts":

                params = {
                    url: url,
                    layer: options.name,
                    style: Cesium.defaultValue(options.style, 'default'),
                    crs: options.crs,
                    type: type,
                    tileMatrixSetID: options.tileMatrixSetID
                };

                if (options.hasOwnProperty('extent')) {
                    params.rectangle = Cesium.Rectangle.fromDegrees(options.extent.xmin, options.extent.ymin, options.extent.xmax, options.extent.ymax)
                }

                if (options.hasOwnProperty('subdomains')) {
                    params.subdomains = options.subdomains;
                }

                if (options.hasOwnProperty('token')) {
                    params.token = options.token;
                }

                if (options.hasOwnProperty('format')) {
                    params.format = options.format;
                }

                if (options.hasOwnProperty('zoomOffset')) {
                    params.zoomOffset = options.zoomOffset;
                }


                return new WebMapTileServiceImageryProvider(params);
            case "geo_vt":

                params = {
                    url: url,
                    layer: options.name,
                    style: Cesium.defaultValue(options.style, ''),
                    crs: options.crs,
                    tileMatrixSetID: options.tileMatrixSetID
                };

                if (options.hasOwnProperty('extent')) {
                    params.rectangle = Cesium.Rectangle.fromDegrees(options.extent.xmin, options.extent.ymin, options.extent.xmax, options.extent.ymax)
                }

                if (options.hasOwnProperty('token')) {
                    params.token = options.token;
                }

                if (options.hasOwnProperty('format')) {
                    params.format = options.format;
                }

                if (options.hasOwnProperty('zoomOffset')) {
                    params.zoomOffset = options.zoomOffset;
                }

                if (options.hasOwnProperty('sld')) {
                    params.sld = options.sld;
                }

                return new GeoServerVectorTileImageryProvider(params);
            case "wms":
                params = this.splitUrl(options.url);
                return new Cesium.WebMapServiceImageryProvider({
                    url: params.url,
                    srs: params.srs || 'EPSG:4490',
                    layers: params.layers,
                    rectangle: Cesium.Rectangle.fromDegrees(options.extent.xmin, options.extent.ymin, options.extent.xmax, options.extent.ymax),
                    parameters: {
                        transparent: true,
                        format: 'image/png'
                    }
                });
            case "open_tile":
                return new Cesium.createTileMapServiceImageryProvider({
                    url: url,
                    layer: 'openStreetMap',
                    style: 'default',
                    tileMatrixSetID: '2km',
                    maximumLevel: 5,
                    format: 'image/png'
                });
            case "tms":
                return new Cesium.UrlTemplateImageryProvider({
                    url: options.url,
                    subdomains: ["0", "1", "2", "3", "4", "5", "6", "7"],
                    tilingScheme: new Cesium.WebMercatorTilingScheme,
                    maximumLevel: 18
                });
            case "spm_terr":
                viewer.scene.terrainProvider.isCreateSkirt = true;
                return new Cesium.CesiumTerrainProvider({
                    url: url,
                    isSct: true
                });
            case "mljs_tile":
                options["layers"] = options.index;
                return new ArcGisMapServerImageryProvider(_.assign(options, {
                    url: url,
                    layer: options.name,
                    tilingScheme: geographicTilingScheme_4490,
                    ellipsoid: Utils.config.ellipsolid["EPSG:4490"]
                }));
            case"spm_s3m":
                var scene = viewer.scene;
                if (options.hasOwnProperty('capables')) {
                    options = options.capables[0];
                }

                var promise = scene.addS3MTilesLayerByScp(url, {
                    name: options.name
                });
                try {
                    Cesium.when(promise, function (layers) {
                        options.modelLayerDetails = layers;

                        var style = layers.style3D;

                        if (options.selectEnabled) {
                            layers.selectEnabled = true;
                            layers.setQueryParameter({
                                url: options.dataURL,
                                dataSourceName: options.dataSourceName,
                                dataSetName: options.dataSetName,
                                keyWord: options.keyWord
                            });
                        } else {
                            layers.selectEnabled = false;
                        }

                        if (options.hasOwnProperty("fillForeColor")) {
                            //设置图层风格
                            style.fillForeColor = Cesium.Color.fromCssColorString(options.fillForeColor.color);
                            style.fillForeColor.alpha = options.fillForeColor.alpha;
                            //设置后需刷新图层
                            layers.refresh();
                        }

                        if (options.hasOwnProperty("wireFrameMode")) {
                            style.lineColor = Cesium.Color.BLACK;
                            style.lineWidth = 0.6;
                            style.fillStyle = options.fillStyle;
                            layers.wireFrameMode = options.wireFrameMode;
                            layers.refresh();
                        }

                        if (options.hasOwnProperty("selectedColor")) {
                            // 设置选中图层显示的高亮颜色
                            layers.selectedColor = options.selectedColor;
                        }

                        if (options.hasOwnProperty("hasLight")) {
                            // 设置图层是否开启光照效果
                            layers.hasLight = options.hasLight;
                        }

                        if (options.hasOwnProperty("objsUnVisible")) {
                            layers.setVisibleInViewport(options.objsUnVisible, false);
                        }

                        if (options.hasOwnProperty("objsVisible")) {
                            layers.setObjsVisible(options.objsVisible, true);
                        }

                        if (options.hasOwnProperty("visible")) {
                            // 设置图层可见性
                            layers.visible = options.visible;
                        }

                        layers.visibleDistanceMax = 5000; // 图层的最大可见距离值
                        layers.lodRangeScale = 0.6; // 图层的LOD层级切换距离缩放系数

                    });
                } catch (e) {
                    if (scene.cesiumWidget._showRenderLoopErrors) {
                        scene.cesiumWidget.showErrorPanel('加载SCP失败，请检查网络连接状态或者url地址是否正确？', undefined, e);
                    }
                }
                break;
            case "tdt_wtfs":
                try {
                    // viewer.scene.globe.depthTestAgainstTerrain = false;
                    var tiandituWTFS = new TiandituGeoWTFS(_.assign(Utils.config.tiandituWTFS.options, {
                        viewer: viewer,
                        url: options.url
                    }));
                    tiandituWTFS.initTDT(Utils.config.tiandituWTFS.resolution);
                } catch (e) {
                    Utils.error(e);
                }
                return undefined;
            case "tianditu":
                return new Cesium.TiandituImageryProvider({
                    mapStyle: Cesium.TiandituMapsStyle[options.name],
                    token: '4a00a1dc5387b8ed8adba3374bd87e5e'
                });
            default:
                Utils.error('不支持:  [' + type + '] 类型图层！');
                break;
        }
    };


    /**
     * getHeight
     */
    MapUtils.prototype.getHeight = function () {
        if (this._viewer) {
            var scene = this._viewer.scene,
                ellipsoid = scene.globe.ellipsoid;
            return ellipsoid.cartesianToCartographic(this._viewer.camera.position).height;
        }
    };

    /**
     * setZoomLevel
     * @param level
     */
    MapUtils.prototype.setZoomLevel = function (level) {
        this._zoomLevel = level;
    };


    MapUtils.prototype.getCameraFocus = function (terria, inWorldCoordinates, result) {
        var scene = terria.scene;
        var camera = scene.camera;
        var rayScratch = new Cesium.Ray();

        var unprojectedScratch = new Cesium.Cartographic();

        if (scene.mode === Cesium.SceneMode.MORPHING) {
            return undefined;
        }

        if (!Cesium.defined(result)) {
            result = new Cesium.Cartesian3();
        }

        if (Cesium.defined(terria.trackedEntity)) {
            result = terria.trackedEntity.position.getValue(terria.clock.currentTime, result);
        } else {
            rayScratch.origin = camera.positionWC;
            rayScratch.direction = camera.directionWC;
            result = scene.globe.pick(rayScratch, scene, result);
        }

        if (!Cesium.defined(result)) {
            return undefined;
        }

        if (scene.mode === Cesium.SceneMode.SCENE2D || scene.mode === Cesium.SceneMode.COLUMBUS_VIEW) {
            result = camera.worldToCameraCoordinatesPoint(result, result);

            if (inWorldCoordinates) {
                result = scene.globe.ellipsoid.cartographicToCartesian(scene.mapProjection.unproject(result, unprojectedScratch), result);
            }
        } else {
            if (!inWorldCoordinates) {
                result = camera.worldToCameraCoordinatesPoint(result, result);
            }
        }

        return result;
    };
    /**
     * 获取服务层级
     * @return {*}
     */
    MapUtils.prototype.getLayerLevel = function () {
        return this._zoomLevel;
    };
    /**
     * 获取当前可视域范围
     */
    MapUtils.prototype.getExtent = function () {

        return this._viewer.camera.computeViewRectangle(this._viewer.scene.globe.ellipsoid)
    };

    /**
     * 获取底图
     */
    MapUtils.prototype.setBaseImageryLayers = function (layers, isBaseLayerChange) {
        // 当前底图
        this._baseImageryLayers = layers;
        // 当前底图id
        var layersIds = [];
        $.each(layers, function (i, lyr) {
            layersIds.push(lyr.layerId);
        });
        // 获取底图配置
        var baseLayerConfig = WidgetUtils._appConfig.baseLayer;
        // 所有底图图层
        var baseLayers = [];
        $.each(baseLayerConfig, function (i, obj) {
            $.each(obj.layers, function (j, lyr) {
                baseLayers.push(lyr);
            });
        });

        //  快速切换时触发
        if (isBaseLayerChange) {
            $.each(baseLayers, function (i, val) {
                $.each(layers, function (j, lyr) {
                    if (val.id === lyr.layerId) {
                        WidgetUtils.checkNode(val.id, true, false);
                    }
                    if (val.id !== lyr.layerId && layersIds.indexOf(val.id) === -1) {
                        WidgetUtils.checkNode(val.id, false, false);
                    }
                });
            });
        }
    };

    /**
     * 设置生成地图上模型的样式
     * @param entity
     * @param type
     */
    MapUtils.prototype.setEntityStyle = function (entity, type) {
        var height = 30;
        if (type === "region") {
            height = 500;
        } else if (type === "building") {
            height = 2;
        }

        $.each(entity, function (index, value) {
            value.polygon.material.color = Cesium.Color.LIGHTSKYBLUE.withAlpha(0.7);
            value.polygon.outline = true;
            value.polygon.extrudedHeight = height;
            value.polygon.outlineWidth = 100;
            value.polygon.outlineColor = Cesium.Color.LIGHTSKYBLUE;
            //value.polygon.outlineColor = Cesium.Color.fromBytes(10, 142, 222, 0.9);
            // 不闭合面，不然底部会和图层重叠导致闪烁
            value.polygon.closeBottom = false;
        });
    };

    /**
     * 兼容修改全部的entity样式
     * @param entities
     */
    MapUtils.prototype.setAllStyle = function (entities, _height, alpha, clampToGround) {
        $.each(entities, function (count, entity) {
            if (entity.point) {
                console.log("point")
            } else if (entity.polyline) {
                entity.polyline.material.color = (Cesium.Color.RED);
                entity.polyline.width.setValue(4);
                entity.polyline.clampToGround.setValue(true);
                console.log("polyline")
            } else if (entity.polygon) {
                var height = 10;
                var color = Cesium.Color.LIGHTSKYBLUE.withAlpha(alpha || 0.7);
                if (entity.properties.__GEOAREA) {
                    if (entity.properties.__GEOAREA.getValue() > 50000000)
                        height = 300;
                    else if (entity.properties.__GEOAREA.getValue() < 50000000 && entity.properties.__GEOAREA.getValue() > 5000000)
                        height = 50;
                }
                if (_height) {
                    height = _height;
                }
                if (clampToGround) {
                    height = undefined;
                }
                if (entity.name === "房产") {
                    var high = entity.properties.high._value;
                    var floor = entity.properties.Floor2._value;
                    if (high === 0 && floor === '0') {
                        height = 3
                    } else if (high === 0 && floor !== '0') {
                        height = 3 * parseInt(floor);
                    } else if (high !== 0) {
                        height = high;
                    }
                    entity.polygon.zindex = 100;
                    color = Cesium.Color.LIGHTSKYBLUE;
                }

                entity.polygon.material.color = color;
                entity.polygon.outline = true;
                entity.polygon.extrudedHeight = height;
                entity.polygon.outlineColor = Cesium.Color.LIGHTSKYBLUE;
                //entity.polygon.outlineColor = Cesium.Color.RED;
                // 不闭合面，不然底部会和图层重叠导致闪烁
                entity.polygon.closeBottom = false;
                console.log("polygon")
            }
        })
    };

    /**
     * 根据超图绘制方法生成featureCollection格式结果
     * @param polygon
     */
    MapUtils.prototype.superMapPolygonToFeatureCollection = function (polygon) {
        var coordinates = [];
        $.each(polygon.object.positions, function (index, value) {
            var point = Cesium.Cartographic.fromCartesian(value);
            var temp = [];
            temp.push(Cesium.Math.toDegrees(point.longitude));
            temp.push(Cesium.Math.toDegrees(point.latitude));
            coordinates.push(temp);
        });
        coordinates.push(coordinates[0]);
        return {
            "type": "FeatureCollection",
            "features": [
                {
                    "type": "Feature",
                    "geometry": {
                        "type": "Polygon",
                        "coordinates": [coordinates]
                    }
                }
            ]
        };
    };

    MapUtils.prototype.drawGeometryPolygonToFeatureCollection = function (points) {
        var coordinates = [];
        $.each(points, function (index, value) {
            var point = Cesium.Cartographic.fromCartesian(value);
            var temp = [];
            temp.push(Cesium.Math.toDegrees(point.longitude));
            temp.push(Cesium.Math.toDegrees(point.latitude));
            coordinates.push(temp);
        });
        coordinates.push(coordinates[0]);
        return {
            "type": "FeatureCollection",
            "features": [
                {
                    "type": "Feature",
                    "geometry": {
                        "type": "Polygon",
                        "coordinates": [coordinates]
                    }
                }
            ]
        };
    };
    /**
     * 存放绘面监听，方便清除方法
     * @param handler
     */
    MapUtils.prototype.setPolygonDrawHandler = function (handler) {
        this._polygonDrawHandler = handler;
    };

    MapUtils.prototype.polygonToLine = function (geojson) {
        var allFeatures = [];
        $(geojson.features).each(function (i, feature) {
            var geometry = feature.geometry,
                properties = feature.properties;
            var features = [];
            switch (geometry.type.toLowerCase()) {
                case 'polygon':
                    features = polygonToLineString(geometry.coordinates, properties);
                case 'multipolygon':
                    features = multiPolygonToLineString(geometry.coordinates, properties);
                default:
                    return feature;
            }
            allFeatures.push(features);
        });
        var temp = {};
        temp.type = "FeatureCollection";
        temp.features = allFeatures;
        return temp;
    };

    MapUtils.prototype.startCursorCrosshair = function () {
        var cesiumWidget = this._viewer.cesiumWidget.container;
        cesiumWidget.style.cursor = 'crosshair';
    };

    MapUtils.prototype.stopCursorCrosshair = function () {
        var cesiumWidget = this._viewer.cesiumWidget.container;
        cesiumWidget.style.cursor = '';
    };

    function polygonToLineString(coordinates, properties) {
        return ($(coordinates).map(function (i, coordinate) {
            return turf.lineString(coordinate, properties);
        })).get(0);
    }

    function multiPolygonToLineString(coordinates, properties) {
        var arrs = [];
        $(coordinates).map(function (i, coordinate) {
            var feature = polygonToLineString(coordinate, properties);
            arrs.push(feature);
        })
        return arrs;
    }

    return new MapUtils
});
