/**
 *  viewer Extend
 * @author by <a href="mailto:chaomashi@gmail.com">Du</a>
 * @version v1.0 2019/8/21/16:30  Copyright gtmap Corp
 */

define([
    'Cesium',
    'lodash'
], function (
    Cesium,
    _
) {
    // 'use strict'; 去除否则不能 set readOnly property

    /**
     * 进行扩展展示
     * @param viewer
     * @constructor
     */
    function ViewerExtend(viewer) {
        if (!viewer) {
            throw new Cesium.DeveloperError('viewer is required.');
        }

        var scene = viewer.scene;
        var IEVersion = getBrowserType();
        // 无法直接更改属性、使用覆盖合并修改
        // scene.screenSpaceCameraController = _.assignIn(scene.screenSpaceCameraController, new Cesium.ScreenSpaceCameraController(scene));
        // set viewer params
        scene.globe.enableLighting = true;
        // 是否展示大气层
        scene.globe.showGroundAtmosphere = false;
        scene.globe.baseColor = Cesium.Color.WHITE;
        scene.skyAtmosphere.show = false;
        if (!IEVersion) {
            scene.postProcessStages.fxaa.enabled = false;
        }
        // 该值在0.66~1.33之间地图清晰度最高。
        scene.globe.maximumScreenSpaceError = 4 / 3;
        scene.highDynamicRange = false;

        //相机的高度的最小值
        scene.screenSpaceCameraController.minimumZoomDistance = 5;
        //相机高度的最大值
        scene.screenSpaceCameraController.maximumZoomDistance = 22000000.0;

        this._viewer = viewer;
        scene.cameraClone = new Cesium.Camera(scene);

        // TODO other vierwer extend or Copy fun
        return _.assign(this._viewer, {
            flyToViewer: _flyToViewer,
            zoomToViewer: _zoomToViewer,

            // TODO other supermap extend
            superMap: {}
        });
    }

    function getBrowserType() {
        var isIE11;
        navigator.userAgent.indexOf('Trident') > -1 && navigator.userAgent.indexOf("rv:11.0") > -1 ? isIE11 = true : isIE11 = false;
        return isIE11
    }

    /**
     * Flies the camera to the provided entity, entities, or data source.
     * If the data source is still in the process of loading or the visualization is otherwise still loading,
     * this method waits for the data to be ready before performing the flight.
     *
     * <p>The offset is heading/pitch/range in the local east-north-up reference frame centered at the center of the bounding sphere.
     * The heading and the pitch angles are defined in the local east-north-up reference frame.
     * The heading is the angle from y axis and increasing towards the x axis. Pitch is the rotation from the xy-plane. Positive pitch
     * angles are above the plane. Negative pitch angles are below the plane. The range is the distance from the center. If the range is
     * zero, a range will be computed such that the whole bounding sphere is visible.</p>
     *
     * <p>In 2D, there must be a top down view. The camera will be placed above the target looking down. The height above the
     * target will be the range. The heading will be determined from the offset. If the heading cannot be
     * determined from the offset, the heading will be north.</p>
     *
     * @param {Entity|Entity[]|EntityCollection|DataSource|ImageryLayer|Cesium3DTileset|TimeDynamicPointCloud|Promise.<Entity|Entity[]|EntityCollection|DataSource|ImageryLayer|Cesium3DTileset|TimeDynamicPointCloud>} target The entity, array of entities, entity collection, data source, Cesium3DTileset, point cloud, or imagery layer to view. You can also pass a promise that resolves to one of the previously mentioned types.
     * @param {Object} [options] Object with the following properties:
     * @param {Number} [options.duration=3.0] The duration of the flight in seconds.
     * @param {Number} [options.maximumHeight] The maximum height at the peak of the flight.
     * @param {HeadingPitchRange} [options.offset] The offset from the target in the local east-north-up reference frame centered at the target.
     * @returns {Promise.<Boolean>} A Promise that resolves to true if the flight was successful or false if the target is not currently visualized in the scene or the flight was cancelled. //TODO: Cleanup entity mentions
     */
    function _flyToViewer(target, options) {
        return zoomToOrFly(this, target, options, true);
    }

    /**
     * Asynchronously sets the camera to view the provided entity, entities, or data source.
     * If the data source is still in the process of loading or the visualization is otherwise still loading,
     * this method waits for the data to be ready before performing the zoom.
     *
     * <p>The offset is heading/pitch/range in the local east-north-up reference frame centered at the center of the bounding sphere.
     * The heading and the pitch angles are defined in the local east-north-up reference frame.
     * The heading is the angle from y axis and increasing towards the x axis. Pitch is the rotation from the xy-plane. Positive pitch
     * angles are above the plane. Negative pitch angles are below the plane. The range is the distance from the center. If the range is
     * zero, a range will be computed such that the whole bounding sphere is visible.</p>
     *
     * <p>In 2D, there must be a top down view. The camera will be placed above the target looking down. The height above the
     * target will be the range. The heading will be determined from the offset. If the heading cannot be
     * determined from the offset, the heading will be north.</p>
     *
     * @param {Entity|Entity[]|EntityCollection|DataSource|ImageryLayer|Cesium3DTileset|TimeDynamicPointCloud|Promise.<Entity|Entity[]|EntityCollection|DataSource|ImageryLayer|Cesium3DTileset|TimeDynamicPointCloud>} target The entity, array of entities, entity collection, data source, Cesium3DTileset, point cloud, or imagery layer to view. You can also pass a promise that resolves to one of the previously mentioned types.
     * @param {HeadingPitchRange} [offset] The offset from the center of the entity in the local east-north-up reference frame.
     * @returns {Promise.<Boolean>} A Promise that resolves to true if the zoom was successful or false if the target is not currently visualized in the scene or the zoom was cancelled.
     */
    function _zoomToViewer(target, offset) {
        var options = {
            offset: offset
        };
        return zoomToOrFly(this, target, options, false);
    }

    function zoomToOrFly(that, zoomTarget, options, isFlight) {

        //>>includeStart('debug', pragmas.debug);
        if (!Cesium.defined(zoomTarget)) {
            throw new Cesium.DeveloperError('zoomTarget is required.');
        }
        //>>includeEnd('debug');

        cancelZoom(that);

        //We can't actually perform the zoom until all visualization is ready and
        //bounding spheres have been computed.  Therefore we create and return
        //a deferred which will be resolved as part of the post-render step in the
        //frame that actually performs the zoom
        var zoomPromise = Cesium.when.defer();
        that._zoomPromise = zoomPromise;
        that._zoomIsFlight = isFlight;
        that._zoomOptions = options;

        Cesium.when(zoomTarget, function (zoomTarget) {
            //Only perform the zoom if it wasn't cancelled before the promise resolved.
            if (that._zoomPromise !== zoomPromise) {
                return;
            }

            //If the zoom target is a rectangular imagery in an ImageLayer
            if (zoomTarget instanceof Cesium.ImageryLayer) {
                zoomTarget.getViewableRectangle().then(function (rectangle) {
                    return Cesium.computeFlyToLocationForRectangle(rectangle, that.scene);
                }).then(function (position) {
                    //Only perform the zoom if it wasn't cancelled before the promise was resolved
                    if (that._zoomPromise === zoomPromise) {
                        that._zoomTarget = position;
                    }
                });
                return;
            }

            //If the zoom target is a Cesium3DTileset
            if (zoomTarget instanceof Cesium.Cesium3DTileset) {
                that._zoomTarget = zoomTarget;
                return;
            }

            //If the zoom target is a TimeDynamicPointCloud
            if (zoomTarget instanceof Cesium.TimeDynamicPointCloud) {
                that._zoomTarget = zoomTarget;
                return;
            }

            //If the zoom target is a data source, and it's in the middle of loading, wait for it to finish loading.
            if (zoomTarget.isLoading && Cesium.defined(zoomTarget.loadingEvent)) {
                var removeEvent = zoomTarget.loadingEvent.addEventListener(function () {
                    removeEvent();

                    //Only perform the zoom if it wasn't cancelled before the data source finished.
                    if (that._zoomPromise === zoomPromise) {
                        that._zoomTarget = zoomTarget.entities.values.slice(0);
                    }
                });
                return;
            }

            //Zoom target is already an array, just copy it and return.
            if (Cesium.isArray(zoomTarget)) {
                that._zoomTarget = zoomTarget.slice(0);
                return;
            }

            //If zoomTarget is an EntityCollection, this will retrieve the array
            zoomTarget = Cesium.defaultValue(zoomTarget.values, zoomTarget);

            //If zoomTarget is a DataSource, this will retrieve the array.
            if (Cesium.defined(zoomTarget.entities)) {
                zoomTarget = zoomTarget.entities.values;
            }

            //Zoom target is already an array, just copy it and return.
            if (Cesium.isArray(zoomTarget)) {
                that._zoomTarget = zoomTarget.slice(0);
            } else {
                //Single entity
                that._zoomTarget = [zoomTarget];
            }
        });

        that.scene.requestRender();
        return zoomPromise.promise;
    }


    function cancelZoom(viewer) {
        var zoomPromise = viewer._zoomPromise;
        if (Cesium.defined(zoomPromise)) {
            clearZoom(viewer);
            zoomPromise.resolve(false);
        }
    }

    function clearZoom(viewer) {
        viewer._zoomPromise = undefined;
        viewer._zoomTarget = undefined;
        viewer._zoomOptions = undefined;
    }

    Object.defineProperties(ViewerExtend.prototype, {

        viewer: {
            get: function () {
                return this._viewer;
            }
        }

    });


    return ViewerExtend;
});