/**
 *  TiandituGeoWTFS
 * @example:new TiandituGeoWTFS({
                viewer: viewer,
                url: url,
                subdomains: ['8', '9', '10', '11'],
                metadata: {
                    boundBox: {
                        minX: -180,
                        minY: -90,
                        maxX: 180,
                        maxY: 90
                    },
                    minLevel: n,
                    maxLevel: i
                },
                aotuCollide: true,
                collisionPadding: [5, 10, 8, 5],
                serverFirstStyle: true,
                labelGraphics: {
                    font: "28px sans-serif",
                    fontSize: 28,
                    fillColor: Cesium.Color.DARKGREY,
                    scale: .5,
                    outlineColor: Cesium.Color.BLACK,
                    outlineWidth: 3.125,
                    style: Cesium.LabelStyle.FILL_AND_OUTLINE,
                    showBackground: false,
                    backgroundColor: Cesium.Color.RED,
                    backgroundPadding: new Cesium.Cartesian2(10, 10),
                    horizontalOrigin: Cesium.HorizontalOrigin.MIDDLE,
                    verticalOrigin: Cesium.VerticalOrigin.TOP,
                    eyeOffset: Cesium.Cartesian3.ZERO,
                    pixelOffset: new Cesium.Cartesian2(0, 8)
                },
                billboardGraphics: {
                    horizontalOrigin: Cesium.HorizontalOrigin.CENTER,
                    verticalOrigin: Cesium.VerticalOrigin.CENTER,
                    eyeOffset: Cesium.Cartesian3.ZERO,
                    pixelOffset: Cesium.Cartesian2.ZERO,
                    alignedAxis: Cesium.Cartesian3.ZERO,
                    color: Cesium.Color.WHITE,
                    rotation: 0,
                    scale: 1,
                    width: 18,
                    height: 18
                }
            })
 *
 * @author by <a href="mailto:chaomashi@gmail.com">Du</a>
 * @version v1.0 2019/9/10/16:50  Copyright gtmap Corp
 */

define(['Cesium', 'tp/protobuf/protobuf.min'], function (Cesium, ProtoBuf) {

    "use strict";
    var that;

    function TiandituGeoWTFS(options) {
        options = Cesium.defaultValue(options, Cesium.defaultValue.EMPTY_OBJECT);
        this.proxy = options.proxy;
        this.viewer = options.viewer;
        this.url = options.url;
        this.metadata = options.metadata;
        this.roadMetadata = options.roadMetadata;
        this.roadUrl = options.roadUrl;
        this.labelGraphics = options.labelGraphics ? options.labelGraphics : {};
        this.billboardGraphics = options.billboardGraphics ? options.billboardGraphics : {};
        this.aotuCollide = !!options.aotuCollide;
        this.collisionPadding = options.collisionPadding ? options.collisionPadding : [3, 5, 3, 5];
        this.serverFirstStyle = !!options.serverFirstStyle;
        this.subdomains = options.subdomains;
        Cesium.isArray(this.subdomains) ?
            this.subdomains = this.subdomains.slice() :
            Cesium.defined(this.subdomains)
            && this.subdomains.length > 0 ?
                this.subdomains = this.subdomains.split("") :
                this.subdomains = ["a", "b", "c"];
        this.tileCache = [];
        this.labelCache = [];
        this._isInitial = true;
        this._latelyGrid = [];
        this._latelyRefreshStamp = 0;
        var guid = Cesium.createGuid();
        this._UUID = "GEO_WTFS_LABEL_" + guid;
        this._UUIDRoad = "GEO_WTFS_LABEL_ROAD_" + guid;
        this.viewer.camera.percentageChanged = .08;
        this.bindEvent();
        that = this;
    }

    function n(t, e) {
        return e.minX >= t.minX && e.minX <= t.maxX && e.minY >= t.minY && e.minY <= t.maxY || (e.maxX >= t.minX && e.maxX <= t.maxX && e.maxY >= t.minY && e.maxY <= t.maxY || (e.minX >= t.minX && e.minX <= t.maxX && e.maxY >= t.minY && e.maxY <= t.maxY || e.maxX >= t.minX && e.maxX <= t.maxX && e.minY >= t.minY && e.minY <= t.maxY))
    }

    function o(t, e) {
        var i = t.x,
            n = t.y,
            o = t.width,
            a = t.height,
            r = e.x,
            s = e.y,
            l = e.width,
            h = e.height;
        return !(i >= r && i >= r + l) && (!(i <= r && i + o <= r) && (!(n >= s && n >= s + h) && !(n <= s && n + a <= s)))
    }

    var a = function (t) {
        for (var e = 0, i = 0; i < t.length; i++) {
            null != t.charAt(i).match(/[^\x00-\xff]/gi) ? e += 2 : e += 1
        }
        return e
    };

    /**
     * 解析瓦片数据
     * @param data
     * @returns {{enumGeometryType: *[], enumZCoordType: *[], stringTable: Array, pois: Array}}
     * @private
     */
    function _analyzeTile(data) {

        var params = {
                stringTable: [],
                pois: [],
                enumGeometryType: [
                    {
                        ePoint: 0
                    },
                    {
                        eMultiLineString: 1
                    },
                    {
                        ePolygon: 2
                    }
                ],
                enumZCoordType: [
                    {
                        eCloseGround: 0
                    },
                    {
                        eCloseSeaSurface: 1
                    },
                    {
                        eRelativelyGround: 2
                    },
                    {
                        eAbsolute: 3
                    }]
            },

            errorLoadProto = function () {
                if (!ProtoBuf) {
                    throw new Cesium.RuntimeError("ProtoBuf.js is not present. Please see www/index.html for manual setup instructions.");
                }
                return ProtoBuf.loadProto("option optimize_for = LITE_RUNTIME;package GEOPOI;enum enumGeometryType {ePoint = 0;eMultiLineString = 1;ePolygon = 2;} ;message PBPOI{required uint64 OID = 1;required string Name =2;repeated double Coordinates =3 [packed=true];required enumGeometryType GeometryType = 4;optional int32 Interate = 5;optional int32 SymbolID = 10  [default = 0];optional double DisplayHeight = 11 [default = 32];optional uint32 ShiningColor=12 [default =0];optional uint32\tFontNameIndex=13 [default =0];optional int32\tFontSize=14 [default =18];optional uint32\tFontColor=15 [default =0];};message StringTable {repeated string s = 1;}message PBPOITile{required int64 Version = 1;required int64 TileKey = 2;required StringTable StringTable = 3;repeated PBPOI POIS = 4;};", "GEOPOI", "GeoPOI.proto").build("GEOPOI").PBPOITile
            }(),
            loadProto = function () {
                if (!ProtoBuf) {
                    throw new Cesium.RuntimeError("ProtoBuf.js is not present. Please see www/index.html for manual setup instructions.");
                }

                return ProtoBuf.loadProto("option optimize_for = LITE_RUNTIME;package GEOPOI;enum enumGeometryType {ePoint = 0;eMultiLineString = 1;ePolygon = 2;};enum enumZCoordType {eCloseGround = 0;eCloseSeaSurface = 1;eRelativelyGround = 2;eAbsolute = 3;};message PBPOI{required uint64 OID = 1;required string Name =2;repeated double Coordinates =3 [packed=true];required enumGeometryType GeometryType = 4;optional int32 Interate = 5;optional int32 SymbolID = 10  [default = 0];optional double DisplayHeight = 11 [default = 32];optional uint32 ShiningColor=12 [default =0];optional uint32\tFontNameIndex=13 [default =0];optional int32\tFontSize=14 [default =18];optional uint32\tFontColor=15 [default =0];optional enumZCoordType ZCoordType = 16 [default = eAbsolute];};message StringTable {repeated string s = 1;}message PBPOITile{required int64 Version = 1;required int64 TileKey = 2;required StringTable StringTable = 3;repeated PBPOI POIS = 4;};", "GEOPOI", "GeoPOI2.proto").build("GEOPOI").PBPOITile
            }();

        params.pois.length = 0;
        var result;
        try {

            result = loadProto.decode(data)

        } catch (t) {
            console.log(data.message);
            result = errorLoadProto.decode(data);
        }

        params.version = parseInt(result.Version.toString());
        params.titleKey = parseInt(result.TileKey.toString());
        params.stringTable = [];

        result.StringTable.s.forEach(function (value) {
            params.stringTable.push(value.toString());
        });

        result.POIS.forEach(function (poi) {
            var geo = {
                oid: parseInt(poi.OID.toString()),
                name: poi.Name.toString(),
                symbolID: parseInt(poi.SymbolID.toString()),
                displayHeight: poi.DisplayHeight,
                shiningColor: poi.ShiningColor,
                fontNameIndex: poi.FontNameIndex,
                fontSize: poi.FontSize,
                fontColor: poi.FontColor,
                geometryType: poi.GeometryType,
                interate: poi.Interate,
                coordinate: poi.Coordinates
            };
            if (poi.hasOwnProperty('ZCoordType')) {
                geo.zCoordType = poi.ZCoordType
            }
            params.pois.push(geo);
        });
        return params
    }


    Cesium.defineProperties(TiandituGeoWTFS.prototype, {});


    TiandituGeoWTFS.prototype.getCacheTile = function (t, e, i, n) {
        for (var o = 0; o < this.tileCache.length; o++)
            if (this.tileCache[o].x === t && this.tileCache[o].y === e && this.tileCache[o].z === i && this.tileCache[o].t === n)
                return this.tileCache[o];
        return null
    };

    TiandituGeoWTFS.prototype.addCacheTile = function (tile) {
        this.tileCache.length > 999 && this.tileCache.splice(0, 500);
        this.removeCacheTile(tile.x, tile.y, tile.z, tile.t);
        this.tileCache.push(tile);
    };

    TiandituGeoWTFS.prototype.removeCacheTile = function (t, e, i, n) {
        for (var o = 0; o < this.tileCache.length; o++)
            if (this.tileCache[o].x === t && this.tileCache[o].y === e && this.tileCache[o].z === i && this.tileCache[o].t === n) {
                this.tileCache.splice(o, 1);
                break
            }
    };

    TiandituGeoWTFS.prototype.getCacheLabel = function (t) {
        for (var e = 0; e < this.labelCache.length; e++)
            if (this.labelCache[e].name === this._UUID && this.labelCache[e].oid === t)
                return this.labelCache[e];
        return null
    };

    TiandituGeoWTFS.prototype.addCacheLabel = function (tile) {
        this.labelCache.length > 999 && this.labelCache.splice(0, 250);
        this.removeCacheLabel(tile.oid);
        tile.timestamp = new Date().getTime();
        this.labelCache.push(tile);
    };

    TiandituGeoWTFS.prototype.removeCacheLabel = function (t) {
        for (var e = 0; e < this.labelCache.length; e++)
            if (this.labelCache[e].name === this._UUID && this.labelCache[e].oid === t) {
                this.labelCache.splice(e, 1);
                break
            }
    };

    TiandituGeoWTFS.prototype.getTileUrl = function () {
        return (this.proxy ? this.proxy.proxy : "") + this.url + "/GetTiles?lxys={z},{x},{y}"
    };

    TiandituGeoWTFS.prototype.getIcoUrl = function () {
        return (this.proxy ? this.proxy.proxy : "") + this.url + "/GetIcon?id={id}"
    };

    TiandituGeoWTFS.prototype.getRoadTileUrl = function () {
        return (this.proxy ? this.proxy.proxy : "") + this.roadUrl
    };

    TiandituGeoWTFS.prototype._moveEnd = function () {
        clearTimeout(this._timer);

        var surface = this.viewer.scene.globe._surface;

        if (surface._tilesToRender.length === 0 || surface._tileLoadQueueHigh.length > 3) {
            this._timer = setTimeout(function () {
                this._moveEnd();
            }, 100);
        } else {
            var tilesRender = this.getTilesToRender();
            if (this.compareArray(tilesRender, this._latelyGrid)) {
                return;
            }
            this._queueCall(tilesRender);
            this.delaySynchronous();
        }
    };

    TiandituGeoWTFS.prototype._changed = function () {
        // 获取1970 年 1 月 1 日午夜（GMT 时间）之间的毫秒数
        var time = new Date().getTime();
        this._latelyRefreshStamp > 0 && time - this._latelyRefreshStamp > 300 ?
            this._moveEnd() :
            this.collisionDetection();
    };

    TiandituGeoWTFS.prototype._queueCall = function (tilesRender) {

        this._latelyGrid = tilesRender;
        this._latelyRefreshStamp = new Date().getTime();

        tilesRender.forEach(function (tile) {
            if (that.metadata && n(that.metadata.boundBox, tile.boundBox)) {
                if (that.metadata.minLevel > tile.level + 1 || that.metadata.maxLevel < tile.level + 1) {
                    return;
                }

                var cacheTile = that.getCacheTile(tile.x, tile.y, tile.level + 1, 0);
                if (cacheTile) {
                    that.addLabelAndIco(cacheTile);
                } else {
                    var url = that.getTileUrl().replace("{z}", tile.level + 1).replace("{y}", tile.y).replace("{x}", tile.x).replace("{s}", that.subdomains[(tile.x + tile.y + tile.level) % that.subdomains.length]);
                    var xhr = new XMLHttpRequest;
                    xhr.open("GET", url, true);
                    xhr.responseType = "arraybuffer";
                    xhr.onload = function () {
                        if (!(xhr.status < 200 || xhr.status >= 300)) {
                            var data = that.CutString(xhr.response);
                            if (data) {
                                var _res = _analyzeTile(data);
                                _res.x = this.tile.x;
                                _res.y = this.tile.y;
                                _res.z = this.tile.z;
                                _res.t = 0;
                                that.addCacheTile(_res);
                                that.addLabelAndIco(_res);
                            } else {
                                var _tile = {
                                    x: this.tile.x,
                                    y: this.tile.y,
                                    z: this.tile.z,
                                    t: 0
                                };
                                that.addCacheTile(_tile);
                                that.delaySynchronous();
                            }
                        }
                    };

                    xhr.onerror = function (err) {
                        console.error(err)
                    };

                    xhr.send();

                    xhr.tile = {
                        x: tile.x,
                        y: tile.y,
                        z: tile.level + 1
                    }
                }
            }

            if (that.roadMetadata && n(that.roadMetadata.boundBox, tile.boundBox)) {
                if (that.roadMetadata.minLevel > tile.level + 1 || that.roadMetadata.maxLevel < tile.level + 1) {
                    return;
                }

                var _cacheTile = that.getCacheTile(tile.x, tile.y, tile.level + 1, 1);
                if (_cacheTile) {
                    that.addLabelAndIco(_cacheTile);
                } else {
                    var h = that.getRoadTileUrl().replace("{z}", tile.level + 1).replace("{y}", tile.y).replace("{x}", tile.x);
                    var c = new XMLHttpRequest;
                    c.open("GET", h, !0),
                        c.responseType = "json",
                        c.onload = function () {
                            if (!(c.status < 200 || c.status >= 300)) {
                                var t = c.response;
                                if (t) {
                                    var i = {
                                        pois: t.map(function (t, e, i) {
                                            return {
                                                oid: t.LabelPoint.X + "_" + t.LabelPoint.Y,
                                                name: t.Feature.properties.Name,
                                                coordinate: [t.LabelPoint.X, t.LabelPoint.Y, t.LabelPoint.Z ? t.LabelPoint.Z : 0]
                                            }
                                        }),
                                        x: this.tile.x,
                                        y: this.tile.y,
                                        z: this.tile.z,
                                        t: 1
                                    };
                                    that.addCacheTile(i),
                                        that.addLabelAndIco(i)
                                } else {
                                    i = {
                                        x: this.tile.x,
                                        y: this.tile.y,
                                        z: this.tile.z,
                                        t: 1
                                    };
                                    that.addCacheTile(i);
                                    that.delaySynchronous();
                                }
                            }
                        }
                        ,
                        c.onerror = function (t) {
                            console.error(t)
                        }
                        ,
                        c.send(),
                        c.tile = {
                            x: tile.x,
                            y: tile.y,
                            z: tile.level + 1
                        }
                }
            }
        })
    };

    TiandituGeoWTFS.prototype.initTDT = function (tdt) {
        var initLevel = 0;
        that._queueCall(tdt);
        var n = setInterval(function () {
            if (initLevel > 3) {
                that._isInitial = false;
                clearInterval(n)
            } else if (initLevel % 2 === 0 && that.aotuCollide) {
                that.collisionDetection();
                initLevel++;
            }
        }, 600);
        return this;
    };

    TiandituGeoWTFS.prototype.getTilesToRender = function () {
        // var tilesRender = this.viewer.scene.globe._surface._tilesToRender.map(function (value) {
        //     return {
        //         x: value.x,
        //         y: value.y,
        //         level: value.level,
        //         boundBox: {
        //             minX: Cesium.Math.toDegrees(value.rectangle.west),
        //             minY: Cesium.Math.toDegrees(value.rectangle.south),
        //             maxX: Cesium.Math.toDegrees(value.rectangle.east),
        //             maxY: Cesium.Math.toDegrees(value.rectangle.north)
        //         }
        //     }
        // }).sort(function (t, e) {
        //     return e.level - t.level
        // });
        //
        // var level = [tilesRender[0].level];
        //
        // tilesRender.forEach(function (value, index) {
        //     value.level !== level[level.length - 1] && level.push(value.level);
        //     if (level.length > 4) {
        //         tilesRender.splice(index);
        //     }
        // });
        //
        // return tilesRender
        for (var e = this.viewer.scene.globe._surface._tilesToRender.map(function (e, i, n) {
            return {
                x: e.x,
                y: e.y,
                level: e.level,
                boundBox: {
                    minX: Cesium.Math.toDegrees(e.rectangle.west),
                    minY: Cesium.Math.toDegrees(e.rectangle.south),
                    maxX: Cesium.Math.toDegrees(e.rectangle.east),
                    maxY: Cesium.Math.toDegrees(e.rectangle.north)
                }
            }
        }).sort(function (t, e) {
            return e.level - t.level
        }), i = [e[0].level], n = 0; n < e.length; n++)
            e[n].level !== i[i.length - 1] && i.push(e[n].level),
            i.length > 4 && (e.splice(n),
                n--);
        return e
    };

    TiandituGeoWTFS.prototype.addLabelAndIco = function (t) {
        if (t.pois)
            for (var e = 0; e < t.pois.length; e++) {
                var i = this.getCacheLabel(t.pois[e].oid);
                i || (i = this.createLabel(t.pois[e], t));
                this.addCacheLabel(i);
            }
        this.delaySynchronous()
    };

    TiandituGeoWTFS.prototype.delaySynchronous = function () {

        clearTimeout(this._timer2);
        this._timer2 = setTimeout(function () {
            that.synchronousLabel();
        }, 100)
    };

    TiandituGeoWTFS.prototype.synchronousLabel = function () {
        for (var t = 0; t < this.labelCache.length; t++)
            this.labelCache[t].timestamp >= this._latelyRefreshStamp && !this.viewer.entities.contains(this.labelCache[t]) && (this._isInitial && this.aotuCollide && (this.labelCache[t].show = !1),
                this.viewer.entities.add(this.labelCache[t]));
        if (!this._isInitial) {
            for (t = 0; t < this.viewer.entities.values.length; t++)
                !this.viewer.entities.values[t].name || this.viewer.entities.values[t].name !== this._UUID && this.viewer.entities.values[t].name !== this._UUIDRoad || this.viewer.entities.values[t].timestamp < this._latelyRefreshStamp && (this.viewer.entities.remove(this.viewer.entities.values[t]),
                    t--);
            this.aotuCollide && this.collisionDetection()
        }
    };

    TiandituGeoWTFS.prototype.createLabel = function (e, i) {
        if (e) {
            var n = {
                show: !0,
                position: Cesium.Cartesian3.fromDegrees(e.coordinate[0], e.coordinate[1], e.coordinate[2]),
                label: {
                    text: e.name
                }
            };
            this.serverFirstStyle && undefined !== e.fontSize ? (n.label.font = e.fontSize + "px ",
                undefined !== e.fontNameIndex && i.stringTable && i.stringTable[e.fontNameIndex] ? n.label.font += i.stringTable[e.fontNameIndex] : n.label.font += "sans-serif",
            this.labelGraphics.bold && (n.label.font = "bold " + n.label.font)) : this.labelGraphics.font && (n.label.font = this.labelGraphics.font),
                this.serverFirstStyle && undefined !== e.fontColor ? n.label.fillColor = Cesium.Color.fromCssColorString(this.HexadecimalConversion(e.fontColor)) : this.labelGraphics.fillColor && (n.label.fillColor = this.labelGraphics.fillColor),
                this.serverFirstStyle && undefined !== e.shiningColor ? n.label.outlineColor = Cesium.Color.fromCssColorString(this.HexadecimalConversion(e.shiningColor)) : this.labelGraphics.outlineColor && (n.label.outlineColor = this.labelGraphics.outlineColor),
                this.serverFirstStyle && undefined !== e.outlineWidth ? n.label.outlineWidth = e.outlineWidth : undefined !== this.labelGraphics.outlineWidth && (n.label.outlineWidth = this.labelGraphics.outlineWidth),
                this.serverFirstStyle && undefined !== e.showBackground ? n.label.showBackground = e.showBackground : undefined !== this.labelGraphics.showBackground && (n.label.showBackground = this.labelGraphics.showBackground),
                this.serverFirstStyle && undefined !== e.backgroundColor ? n.label.backgroundColor = e.backgroundColor : undefined !== this.labelGraphics.backgroundColor && (n.label.backgroundColor = this.labelGraphics.backgroundColor),
                this.serverFirstStyle && undefined !== e.backgroundPadding ? n.label.backgroundPadding = e.backgroundPadding : undefined !== this.labelGraphics.backgroundPadding && (n.label.backgroundPadding = this.labelGraphics.backgroundPadding),
                i.t ? (n.label.verticalOrigin = t.VerticalOrigin.CENTER,
                    n.label.horizontalOrigin = t.HorizontalOrigin.CENTER) : (this.serverFirstStyle && undefined !== e.verticalOrigin ? n.label.verticalOrigin = e.verticalOrigin : undefined !== this.labelGraphics.verticalOrigin && (n.label.verticalOrigin = this.labelGraphics.verticalOrigin),
                    this.serverFirstStyle && undefined !== e.horizontalOrigin ? n.label.horizontalOrigin = e.horizontalOrigin : undefined !== this.labelGraphics.horizontalOrigin && (n.label.horizontalOrigin = this.labelGraphics.horizontalOrigin)),
                this.serverFirstStyle && undefined !== e.eyeOffset ? n.label.eyeOffset = e.eyeOffset : undefined !== this.labelGraphics.eyeOffset && (n.label.eyeOffset = this.labelGraphics.eyeOffset),
                this.serverFirstStyle && undefined !== e.pixelOffset ? n.label.pixelOffset = e.pixelOffset : undefined !== this.labelGraphics.pixelOffset && (n.label.pixelOffset = this.labelGraphics.pixelOffset),
                this.serverFirstStyle && undefined !== e.style ? n.label.style = e.style : undefined !== this.labelGraphics.style && (n.label.style = this.labelGraphics.style),
                this.serverFirstStyle && undefined !== e.scale ? n.label.scale = e.scale : undefined !== this.labelGraphics.scale && (n.label.scale = this.labelGraphics.scale),
                n.label.disableDepthTestDistance = this.labelGraphics.disableDepthTestDistance,
            undefined !== e.symbolID && e.symbolID > -1 && (n.billboard = {
                image: this.getIcoUrl().replace("{id}", e.symbolID).replace("{s}", this.subdomains[(i.x + i.y + i.z) % this.subdomains.length])
            },
                this.serverFirstStyle && undefined !== e.displayHeight ? (n.billboard.width = e.displayHeight,
                    n.billboard.height = e.displayHeight) : undefined === this.billboardGraphics.width && undefined === this.billboardGraphics.height || (n.billboard.width = this.billboardGraphics.width,
                    n.billboard.height = this.billboardGraphics.height),
                i.t ? (n.label.verticalOrigin = t.VerticalOrigin.BOTTOM,
                    n.label.horizontalOrigin = t.HorizontalOrigin.CENTER) : (this.serverFirstStyle && undefined !== e.verticalOrigin ? n.billboard.verticalOrigin = e.verticalOrigin : undefined !== this.billboardGraphics.verticalOrigin && (n.billboard.verticalOrigin = this.billboardGraphics.verticalOrigin),
                    this.serverFirstStyle && undefined !== e.horizontalOrigin ? n.billboard.horizontalOrigin = e.horizontalOrigin : undefined !== this.billboardGraphics.horizontalOrigin && (n.billboard.horizontalOrigin = this.billboardGraphics.horizontalOrigin)),
                this.serverFirstStyle && undefined !== e.eyeOffset ? n.billboard.eyeOffset = e.eyeOffset : undefined !== this.billboardGraphics.eyeOffset && (n.billboard.eyeOffset = this.billboardGraphics.eyeOffset),
                this.serverFirstStyle && undefined !== e.pixelOffset ? n.billboard.pixelOffset = e.pixelOffset : undefined !== this.billboardGraphics.pixelOffset && (n.billboard.pixelOffset = this.billboardGraphics.pixelOffset),
                this.serverFirstStyle && undefined !== e.rotation ? n.billboard.rotation = e.rotation : undefined !== this.billboardGraphics.rotation && (n.billboard.rotation = this.billboardGraphics.rotation),
                this.serverFirstStyle && undefined !== e.alignedAxis ? n.billboard.alignedAxis = e.alignedAxis : undefined !== this.billboardGraphics.alignedAxis && (n.billboard.alignedAxis = this.billboardGraphics.alignedAxis),
                this.serverFirstStyle && undefined !== e.color ? n.billboard.color = e.color : undefined !== this.billboardGraphics.color && (n.billboard.color = this.billboardGraphics.color),
                this.serverFirstStyle && undefined !== e.scale ? n.billboard.scale = e.scale : undefined !== this.billboardGraphics.scale && (n.billboard.scale = this.billboardGraphics.scale),
                n.billboard.disableDepthTestDistance = this.billboardGraphics.disableDepthTestDistance);
            var o = new Cesium.Entity(n);
            return o.name = i.t ? this._UUIDRoad : this._UUID,
                o.oid = e.oid,
                o.xyz = i.x + "_" + i.y + "_" + (i.z - 1),
                o
        }
    };

    TiandituGeoWTFS.prototype.getPropertyValue = function (t, e, i, n) {
        return undefined !== e[t] ? e[t] : undefined !== i[t] ? i[t] : n
    };

    TiandituGeoWTFS.prototype.collisionDetection = function () {
        if (this.debug) {
            for (var e = document.getElementsByClassName("debug"), i = 0; i < e.length; i++)
                this.viewer.container.removeChild(e[i]),
                    i--;
        }

        var n = [];
        for (i = 0; i < this.viewer.entities.values.length; i++)
            if (this.viewer.entities.values[i].name && (this.viewer.entities.values[i].name === this._UUID || this.viewer.entities.values[i].name === this._UUIDRoad)) {
                var a = Cesium.SceneTransforms.wgs84ToWindowCoordinates(this.viewer.scene, this.viewer.entities.values[i].position.getValue(0));
                this.viewer.entities.values[i].show = !!a;
                var r = this.getLabelReact({
                    point: a,
                    entity: this.viewer.entities.values[i]
                });
                this.viewer.entities.values[i].collisionBox = r;
                for (var s = null, l = 0; l < n.length; l++)
                    if (n[l].xyz === this.viewer.entities.values[i].xyz) {
                        s = n[l];
                        break
                    }
                if (s || (s = {
                    xyz: this.viewer.entities.values[i].xyz,
                    entities: []
                },
                    n.push(s)),
                    s.entities.push(this.viewer.entities.values[i]),
                    this.debug) {
                    var h = document.createElement("div");
                    h.setAttribute("class", "debug"),
                        h.style.cssText = "position:absolute; top:" + r.y + "px; left:" + r.x + "px; width:" + r.width + "px; height:" + r.height + "px; border: solid 1px red; color: rgba(0,0,0,0);",
                        h.innerHTML = this.viewer.entities.values[i].label.text.getValue(0),
                        this.viewer.container.appendChild(h)
                }
            }
        for (var c = 0; c < n.length; c++)
            for (i = 0; i < n[c].entities.length; i++)
                if (n[c].entities[i].show)
                    for (l = i + 1; l < n[c].entities.length; l++)
                        n[c].entities[l].show && o(n[c].entities[i].collisionBox, n[c].entities[l].collisionBox) && (n[c].entities[l].show = !1);
        var p = [];
        for (c = 0; c < n.length; c++)
            for (i = 0; i < n[c].entities.length; i++)
                n[c].entities[i].show && p.push(n[c].entities[i]);
        for (i = 0; i < p.length; i++)
            if (p[i].show)
                for (l = i + 1; l < p.length; l++)
                    p[l].show && o(p[i].collisionBox, p[l].collisionBox) && (p[l].show = !1)
    };

    TiandituGeoWTFS.prototype.bindEvent = function () {
        this.viewer.scene.camera.moveEnd.addEventListener(this._moveEnd, this);
        this.viewer.scene.camera.changed.addEventListener(this._changed, this);
    };

    TiandituGeoWTFS.prototype.unbindEvent = function () {
        this.viewer.scene.camera.moveEnd.removeEventListener(this._moveEnd, this);
        this.viewer.scene.camera.changed.removeEventListener(this._changed, this);
    };

    TiandituGeoWTFS.prototype.activate = function () {
        this._latelyGrid = [];
        this._moveEnd();
    };

    TiandituGeoWTFS.prototype.destroy = function () {
        for (var t = 0; t < this.viewer.entities.values.length; t++) {
            !this.viewer.entities.values[t].name || this.viewer.entities.values[t].name !== this._UUID && this.viewer.entities.values[t].name !== this._UUIDRoad || (this.viewer.entities.remove(this.viewer.entities.values[t]),
                t--);
        }

        this.viewer.camera.percentageChanged = .5;
        this.unbindEvent();
        this.handler = this.handler && this.handler.destroy();
        this.proxy = undefined;
        this.viewer = undefined;
        this.url = undefined;
        this.labelGraphics = undefined;
        this.billboardGraphics = undefined;
        this.aotuCollide = undefined;
        this.collisionPadding = undefined;
        this.tileCache = undefined;
        this.labelCache = undefined;
        this._latelyGrid = undefined;
        this._latelyRefreshStamp = undefined;
        this._roadTileset = undefined;
    };

    TiandituGeoWTFS.prototype.compareArray = function (tilesRender, latelyGrid) {
        var i = false, o = false;
        for (var n = 0; n < tilesRender.length; n++) {
            for (var a = 0; a < latelyGrid.length; a++)
                if (tilesRender[n].x === latelyGrid[a].x && tilesRender[n].y === latelyGrid[a].y && tilesRender[n].level === latelyGrid[a].level) {
                    o = false;
                    break
                }
            if (!o) {
                i = !0;
                break
            }
        }
        return !i
    };

    TiandituGeoWTFS.prototype.getLabelReact = function (t) {
        var e = parseInt(t.entity.label.font);
        e = e > 0 ? e : 15;
        for (var i = t.entity.label.text.getValue(0).split("\n"), n = 0, o = i.length, r = 0; r < i.length; r++) {
            var s = a(i[r]) / 2;
            s > n && (n = s)
        }
        var l = t.entity.billboard ? t.entity.billboard.width.getValue(0) * t.entity.billboard.scale.getValue(0) : 1
            ,
            h = t.entity.billboard ? t.entity.billboard.height.getValue(0) * t.entity.billboard.scale.getValue(0) : 1;
        return {
            x: (t.point ? t.point.x : -999) - l / 2 - this.collisionPadding[3],
            y: (t.point ? t.point.y : -999) - h / 2 - this.collisionPadding[0],
            width: e * t.entity.label.scale.getValue(0) * n + t.entity.label.pixelOffset.getValue(0).x + l + this.collisionPadding[1],
            height: e * t.entity.label.scale.getValue(0) * (o + .5 * (o - 1)) + t.entity.label.pixelOffset.getValue(0).y + h / 2 + this.collisionPadding[2]
        }
    };

    TiandituGeoWTFS.prototype.CutString = function (data) {
        if (!data) {
            return "";
        }

        return data.byteLength <= 28 ? "" : data.slice(19, data.byteLength - 9);
    };

    TiandituGeoWTFS.prototype.HexadecimalConversion = function (t) {
        if (4278190080 === t)
            return "#000000";
        var e = 4278190080 | parseInt(-Number(t));
        t = "";
        if ((e = e.toString(16).substring(1)).length < 6)
            for (var i = 6 - e.length, n = 0; n < i; n++)
                t += "0";
        return "#" + t + e
    };


    return TiandituGeoWTFS
});