/**
 * @author by <a href="mailto:chaomashi@gmail.com">C</a>
 * @version v1.0 2019/2/14/10:21  Copyright gtmap Corp
 */
define(["mapV",
    "global/core/MapVLayerProvider",
    "CompositeView",
    "global/utils/MapUtils",
    "Cesium",
    "mustache",
    'global/core/EventBus',
    "echarts"
], function (mapV, MapVLayer, CompositeView, MapUtils, Cesium, mustache, EventBus, echarts) {
    /*
    * 存放echarts渲染的可视化模型参数
    *
    * scatterOption：散点(态势)图          数据格式：value(权值) coords(坐标) name(名称)
    *
    * linesOption：折线图                  数据格式： fromName   toName    coords:[[y,x],[y,x]]
    *
    * flowOption: 流线图
    *
    * mapVHeat: mapv热力图
    *
    *mapVHoneycomb：蜂窝图
    *
    * mapVLines ： 点构造首尾闭合线
    *
    * mapVDynamivPoints ： 动态点
    *
    * mapVPolygon： 渲染面
    *
    * mapVSimpleLine：常规线
    *
    * 注：cesium扩展echarts的gl类中的coordinateSystem方法,将二维坐标点转化为三维点，所以需要
    * 增加两个参数配置
    * ** GLMap:{}
    * ** coordinateSystem:"GLMap"
    *
    *
    * */

    var self;
    var source;
    var layer;
    layui.use('layer', function () {
        layer = layui.layer;
    });
    // 调用echarts实例
    var chartView = null;

    var that = this;
    var echartsComponents =null;


    /*main*/
    function Model() {
        if (!MapUtils.viewer) {
            console.log("Model viewer is null");
        }

        if (screen.width <= 1439) {
            $("#echarts-components").css({"height": "347px","width": "920px","top": "-26px"});
        } else {
            $("#echarts-components").css({"height": "430px","width": "1140px"});
        }
        echartsComponents = echarts.init(document.getElementById('echarts-components'), 'macarons');

        chartView = new CompositeView("globalContainer", MapUtils.viewer)
        self = this;
        this._option = {
            color: 'purple',
            slider: 1.0,
            title: "",
            size: 30
        }; // option参数
        this._mapVLyr = []; // mapV图层
        this._center = [];
        /**
         * geojson的数据结构转为mapv需要的数据结构
         * @param feaCol
         */
        this.feaColToMapV = function (feaCol, option) {
            if (!feaCol.type) {
                return feaCol;
            }
            if (feaCol.type.toLowerCase() != "featurecollection") {
                return feaCol;
            }
            var proName = option.prop || "hyda";
            var data = [];
            if ((!feaCol) || (!feaCol.features) || (feaCol.features.length == 0)) {
                layer.msg("渲染数据为空");
                throw new Error("渲染数据为空");
                return data;
            }
            var geoType = "";
            var feaType = feaCol.features[0].geometry.type;
            var dis = option.maxValue - option.minValue;
            $.each(feaCol.features, function (i, e) {
                var val = e.properties[proName] || 1;
                val = val * 1;
                //手动指定分级
                if (option.gradientDic && option.gradientDic.length != 0 && option.dataType == 1) {
                    $(option.gradientDic).each(function (i, e) {
                        if (val >= e.min && val < e.max) {
                            val = (i / option.gradientDic.length) * 100;
                            return;
                        }
                    })
                } else if (option.modelType == "heat" && option.dataType == 1) {//数字且max-min！=0
                    val = (e.properties[proName] - option.minValue) / dis * 100;
                }
                switch (feaType.toLowerCase()) {
                    case "multipoint":
                        if (e.geometry.coordinates.length == 1) {
                            data.push({
                                geometry: {type: "Point", coordinates: e.geometry.coordinates[0]},
                                count: val
                            });
                        }
                        break;
                    case "point":
                        data.push({
                            geometry: {type: "Point", coordinates: e.geometry.coordinates},
                            count: val
                        });
                        break;
                    case "polygon":
                        data.push({
                            geometry: {type: "polygon", coordinates: e.geometry.coordinates},
                            count: e.properties[proName]
                        });
                        break;
                    case "polyline":
                        geoType = "LineString";
                        data.push({
                            geometry: {type: "LineString", coordinates: e.geometry.coordinates},
                            count: e.properties[proName]
                        });
                        break;
                }

            });
            return data;
        }

        /**
         * 执行分级操作
         */
        this.grade = function (feaCol, option) {
            //默认color
            var colorDict = [];
            colorDict[0] = [173, 216, 230, 200];//new Cesium.Color.fromBytes(173,216,230,200);
            colorDict[1] = [100, 149, 237, 200];// new Cesium.Color.fromBytes(100,149,237,200);
            colorDict[2] = [255, 127, 80, 200];//new Cesium.Color.fromBytes(255,127,80,200);
            colorDict[3] = [45, 137, 45, 200];//new Cesium.Color.fromBytes(45,137,45,200);
            colorDict[4] = [255, 0, 255, 200];//new Cesium.Color.fromBytes(255,0,255,200);
            colorDict[5] = [150, 205, 205, 200];//new Cesium.Color.fromBytes(255,0,255,200);
            colorDict[6] = [188, 238, 104, 200];//new Cesium.Color.fromBytes(255,0,255,200);
            colorDict[7] = [154, 205, 50, 200];//new Cesium.Color.fromBytes(255,0,255,200);
            colorDict[8] = [238, 180, 180, 200];//new Cesium.Color.fromBytes(255,0,255,200);
            colorDict[9] = [255, 50, 255, 200];//new Cesium.Color.fromBytes(255,0,255,200);
            colorDict[10] = [200, 50, 255, 200];//new Cesium.Color.fromBytes(255,0,255,200);
            colorDict[11] = [50, 250, 255, 200];//new Cesium.Color.fromBytes(255,0,255,200);

            var max = 0;
            var min = 0;
            var grade = 4;
            var avg = 0;
            //用来文字区别
            var strColorObj = {};

            /**
             * 排序后的要素集合
             * @type {Array}
             */
            var featureSortArr = [];

            var gradeType = option.gradeType || "number";

            /**
             * 从color库中获取color
             * 如果color库不够返回随机颜色
             * @param v
             * @returns {*}
             * @private
             */
            var currentDicIndex = 0;

            function _getRandomColor() {
                if (colorDict.length <= currentDicIndex) {
                    var r = Math.floor(Math.random() * 255);
                    var g = Math.floor(Math.random() * 255);
                    var b = Math.floor(Math.random() * 255);
                    colorDict[currentDicIndex] = [r, g, b, 200];
                }
                currentDicIndex++;
                return colorDict[currentDicIndex - 1];
            }


            /**
             * 按照数值分等级
             * 1、按照默认取值
             * 2、colorbar渐变色取值
             */
            function createColorDicByNumber() {
                //手动分级
                if (option.gradient) {
                    option.gradientDic = JSON.parse(option.gradient);
                } else {
                    option.gradientDic = [];
                }
                //总共默认分四级
                if (option.colorBar) {
                    if (option.colorBar.gradient) {
                        var graArr = option.colorBar.gradient;
                        var ratio = (graArr.length / 5).toFixed(2);
                        for (var i = 0; i < 5; i++) {
                            var count = i * ratio;
                            var startColor = graArr[Math.floor(count)];
                            var endColor = graArr [Math.ceil(count)];
                            var remain = count - Math.floor(count);
                            var startRgbaArr = startColor.split(",");
                            var endRgbaArr = endColor.split(",");
                            var diifr = (endRgbaArr[0] - startRgbaArr[0]) * remain;
                            var diifg = (endRgbaArr[1] - startRgbaArr[1]) * remain;
                            var diifb = (endRgbaArr[2] - startRgbaArr[2]) * remain;
                            var diifa = (endRgbaArr[3] - startRgbaArr[3]) * remain;
                            var tmp = [];
                            tmp[0] = parseFloat(startRgbaArr[0]) + diifr;
                            tmp[1] = parseFloat(startRgbaArr[1]) + diifg;
                            tmp[2] = parseFloat(startRgbaArr[2]) + diifb;
                            tmp[3] = parseFloat(startRgbaArr[3]) + diifa;
                            colorDict[i] = tmp;
                        }
                    } else {
                        var startColor = option.colorBar.startColor;
                        var toColor = option.colorBar.toColor;
                        var startRgbaArr = startColor.split(",");
                        var toRgbaArr = toColor.split(",");
                        var ravg = (toRgbaArr[0] - startRgbaArr[0]) / 4;
                        var gavg = (toRgbaArr[1] - startRgbaArr[1]) / 4;
                        var bavg = (toRgbaArr[2] - startRgbaArr[2]) / 4;
                        var aavg = (toRgbaArr[3] - startRgbaArr[3]) / 4;
                        ravg = parseInt(ravg);
                        gavg = parseInt(gavg);
                        bavg = parseInt(bavg);
                        aavg = parseInt(aavg);
                        for (var i = 0; i < 5; i++) {
                            var tmp = [];
                            tmp[0] = parseFloat(startRgbaArr[0]) + ravg * i;
                            tmp[1] = parseFloat(startRgbaArr[1]) + gavg * i;
                            tmp[2] = parseFloat(startRgbaArr[2]) + bavg * i;
                            tmp[3] = parseFloat(startRgbaArr[3]) + aavg * i;
                            colorDict[i] = tmp;
                        }
                    }
                }
                //手动分级
                //兼容色带分组分级
                if (option.gradientDic && option.gradientDic.length != 0 && option.dataType == 1) {
                    var gradientDic = {};
                    $(option.gradientDic).each(function (i, e) {
                        gradientDic[e.legend] = e;
                    })
                    option.dic = option.dic || gradientDic;
                }
                option.prop = option.prop || "SCORE";
                min = feaCol.features[0].properties[option.prop];
                //判断是否是无指定字段，所有color使用统一图形
                //兼容无此字段
                if (min === undefined) {
                    dataType = 0;
                    min = 0;
                    return;
                }
                $(feaCol.features).each(function (i, e) {
                    var value = e.properties[option.prop];
                    //防止数据格式有误
                    if (value === undefined) {
                        value = 0;
                    }

                    if (isNaN(value)) {
                        layer.msg("请选择数字字段！");
                        throw new Error("请选择数字字段");
                        return;
                    }
                    dataType = 1;
                    value = value * 1;
                    if (value > max) {
                        max = value;
                    }
                    if (value < min) {
                        min = value;
                    }
                })
                //获取4阶段的值
                avg = (max - min) / 4;
                if (avg == 0) {
                    dataType = 2;
                }

                //根据dic进行分组
                if (option.dic) {
                    for (var label in option.dic) {
                        strColorObj[label] = _getRandomColor();
                    }
                }
                option.dataType = dataType;
            }

            var dataType = 0;

            /**
             * 获取数据所需类型
             * 0--undefined
             * 1--字符串
             * 2--数字格式的字符串
             * 3--数字，但是是一样的
             */
            function getDataType() {
                return dataType;
            }

            if (gradeType == "number") {
                createColorDicByNumber();
            } else if (gradeType == "string") {
                createColorByString();
            }

            /**
             * 按照字符串类型区分
             * 每种类型自动赋予颜色
             */
            function createColorByString() {
                //根据dic进行分组
                if (option.dic) {
                    var value = 0;
                    for (var key in option.dic) {
                        value++;
                    }
                    value = 0;
                    for (var key in option.dic) {
                        strColorObj[key] = _getRandomColor();
                        value++;
                    }
                }
            }

            /**
             * 智能分组
             */
            function smartGroup(count) {
                var feaArr = getFeaSort();
                var arr = [];
                var propName = option.prop || "SCORE";
                //获取每个区间的个数
                var speCount = Math.ceil(feaArr.length / count);
                if (feaArr.length == 0) {
                    return [];
                }
                var tmpFeature = feaArr[0];
                if (tmpFeature.properties[propName] == undefined) {
                    return [];
                }
                var lastMax = 0;
                for (var i = 0; i < count; i++) {
                    var obj = {};
                    var startIndex = speCount * i;
                    var endIndex = speCount * (i + 1) - 1;
                    if (endIndex > feaArr.length - 1) {
                        endIndex = feaArr.length - 1;
                    }
                    var min = feaArr[startIndex];
                    var max = feaArr[endIndex];
                    var minValue = min.properties[propName] * 1;
                    var maxValue = max.properties[propName] * 1;
                    if (i == count - 1) {
                        maxValue = maxValue + 1;
                    }
                    if (i == 0) {
                        obj.max = maxValue;
                        obj.min = minValue;
                    } else {
                        obj.min = lastMax;
                        obj.max = maxValue;
                    }

                    obj.count = getCountWithInter(obj.min, obj.max);
                    obj.level = i + 1;
                    lastMax = maxValue;
                    arr.push(obj);
                }
                return arr;
            }

            function getFeaSort() {
                if (featureSortArr.length != 0) {
                    return featureSortArr;
                }

                function compare(v1, v2) {
                    var propName = option.prop || "SCORE";
                    var prop1 = v1.properties[propName];
                    var prop2 = v2.properties[propName];
                    return prop1 - prop2;
                }

                feaCol.features.sort(compare);
                featureSortArr = feaCol.features;
                return featureSortArr;
            }

            /**
             * 获取指定数值区间的个数
             */
            function getCountWithInter(min, max) {
                min = min * 1;
                max = max * 1;
                var count = 0;
                var propName = option.prop || "SCORE";
                $(feaCol.features).each(function (i, v) {
                    var value = v.properties[propName];
                    if (value >= min && value < max) {
                        count++;
                    }
                })
                return count;
            }


            var obj = {};
            //渲染等值图
            obj.getColor = function (v) {
                if (option.useHSLColor) {
                    return this.getHSLColor(v);
                }
                if (option.targetColor) {
                    var color = new Cesium.Color.fromBytes(option.targetColor[0], option.targetColor[1], option.targetColor[2], option.targetColor[3]);
                    return color;
                }
                //回调函数
                if (option.gradeCallBack) {
                    v = option.gradeCallBack(v);
                }
                //初始颜色
                var color = Cesium.Color.ORANGERED;

                if (gradeType == "string") {
                    var colorArr = strColorObj[v];
                    if (!colorArr) {
                        colorArr = strColorObj[v] = _getRandomColor();
                    }
                    var color = new Cesium.Color.fromBytes(colorArr[0], colorArr[1], colorArr[2], 200);
                    return color;
                }
                var grade = -1;
                var label = "";
                if (gradeType == "number" || !gradeType) {
                    if (option.dic) {
                        var index = 0;
                        //使用数值进行分级渲染
                        for (var i in option.dic) {
                            var obj = option.dic[i];
                            var minNumber = obj["min"];
                            var maxNumber = obj["max"];
                            if (v >= minNumber && v < maxNumber) {
                                label = i;
                                grade = index;
                                break
                            }
                            index++;
                        }
                        color = new Cesium.Color.fromBytes(colorDict[grade][0], colorDict[grade][1], colorDict[grade][2], option.opcity || 200);
                        if (grade == -1) {
                            color = new Cesium.Color.fromBytes(240, 230, 140, option.opcity || 200);
                        }
                        return color;
                    } else {
                        //获取等级
                        var dis = v - min;
                        if (dis != 0 && avg != 0) {
                            grade = Math.floor(dis / avg);
                        } else {
                            grade = 0;
                        }
                        color = new Cesium.Color.fromBytes(colorDict[grade][0], colorDict[grade][1], colorDict[grade][2], option.opcity || 200);
                        return color;
                    }

                }
            };
            /**
             * 获取max
             * @returns {number}
             */
            obj.getMax = function () {
                return max;
            };

            obj.getMin = function () {
                return min;
            }

            /**
             * 获取hsl颜色
             * @param v
             */
            obj.getHSLColor = function (v) {
                var h = 0.3;
                //获取等级
                var dis = max - min;
                var zb = (v - min) / dis * 0.3.toFixed(5);
                if (dis == 0) {
                    zb = 0;
                }
                var color = Cesium.Color.fromHsl(h + zb, 1.0, 0.5);
                return color;
            };

            /**
             * 返回颜色分组
             */
            obj.getColorDic = function () {
                return colorDict;
            }

            obj.getLegendColorDic = function () {
                var obj = {};
                var dic = option.dic;
                if (!dic) {
                    var colorDic = this.getColorDic();
                    //设置图例
                    obj["一级:"] = colorDic[0];
                    obj["二级:"] = colorDic[1];
                    obj["三级:"] = colorDic[2];
                    obj["四级:"] = colorDic[3];
                    obj["五级:"] = colorDic[4];
                    return obj;
                }
                if (gradeType == "string") {
                    for (var key in strColorObj) {
                        obj[dic[key]] = strColorObj[key];
                    }
                }
                //dic["我是第一级"]={min:20,max:30}
                if (gradeType == "number") {
                    for (var key in strColorObj) {
                        obj[key] = strColorObj[key];//颜色
                    }
                }

                obj["其他"] = [240, 230, 140, 200];
                return obj;
            }

            /**
             * 渲染热力图
             * 从0到1
             */
            obj.getHeatColor = function (dic) {
                var _colorDic = colorDict || dic;
                var gradient = {};
                for (var i = 0; i < 5; i++) {
                    var tmpRGB = _colorDic[i][0] + "," + _colorDic[i][1] + "," + _colorDic[i][2];
                    gradient[(i + 1) / 5] = "rgb(" + tmpRGB + ")";
                }
                return gradient;
            }

            obj.getCountWithInter = getCountWithInter;

            obj.smartGroup = smartGroup;

            obj.getDataType = getDataType;

            return obj;
        }

        // 支持增量更新
        this.choroplethByArr = function (result, showLoading) {
            var model = this;
            var viewer = MapUtils.viewer;
            // var legend = {"一级:":"#add8e6","二级:":"#6495ed","三级:":"#ff7f50","四级:":"#2dad2d","五级:":"#ff00ff"};
            var temp = {};
            temp.type = result.type;
            temp.features = result.features;
            /**
             * 渲染字段
             * @type {*|string}
             */
            var propName = model._option.prop || "SCORE";
            if (model._option.heightByArea) {
                $(temp.features).each(function (i, e) {
                    var area = 0;
                    try {
                        var coor = e.geometry.coordinates;
                        var polygon = turf.polygon(coor);
                        area = turf.area(polygon);
                    } catch (e) {
                        console.log(e);
                    }
                    if (area > 50000000) {
                        area = 300;
                    } else {
                        area = 50;
                    }
                    e.properties["heightByArea"] = area;
                });
            }
            var propHeight = model._option.propHeight;
            var promise;
            var grade = model.grade(result, model._option);
            if (showLoading) {
                layer.msg(model._option.msg || '等值面图生成中...', {
                    time: 2000, //自动关闭
                    btn: false
                });
            }
            if (model._option.clampToGround) {
                promise = Cesium.GeoJsonDataSource.load(temp, {
                    stroke: Cesium.Color.HOTPINK,
                    camera: viewer.scene.camera,
                    canvas: viewer.scene.canvas,
                    clampToGround: true
                });
            } else {
                promise = Cesium.GeoJsonDataSource.load(temp);
            }
            promise.then(function (dataSource) {
                viewer.dataSources.add(dataSource);
                //如果无指定字段
                var targetEntity = dataSource.entities.values[0];

                //隐藏图例
                if (targetEntity.properties[propName] === undefined) {
                    model._option.hideLegend = true;
                    // $("#choroplethLegend").hide();
                }
                var entities = dataSource.entities.values;
                // 分级
                for (var i = 0; i < entities.length; i++) {
                    try {
                        var entity = entities[i];
                        var score;
                        var value = entity.properties[propName];
                        var heightValue = entity.properties[propHeight];
                        if (model._option.heightValueCallback) {
                            heightValue = model._option.heightValueCallback(heightValue);
                        }
                        if (model._option.heightMultiple && heightValue) {
                            heightValue = heightValue * model._option.heightMultiple;
                        }
                        if (model._option.heightByArea) {
                            heightValue = entity.properties["heightByArea"];
                            if (heightValue < 150) {
                                heightValue = 150;
                            }
                        }
                        if (!value) {
                            score = 1;
                        } else {
                            value = value._value;
                        }
                        if (model._option.gradeType == "string") {
                            score = value;
                        } else {
                            score = value * 1;
                        }
                        var color = grade.getColor(score);

                        entity.polygon.material = color;
                        if (!model._option.clampToGround) {
                            entity.polygon.extrudedHeight = heightValue || model._option.polygonHeight || 150;
                        }
                        entity.polygon.outline = true;
                        entity.polygon.outlineWidth = model._option.lineWidth || 800;
                        entity.polygon.outlineColor = new Cesium.Color.fromBytes(0, 0, 200, model._option.lineOpcity || 30);
                    } catch (e) {
                    }
                }

            });
            if (showLoading) {
                if (!model._option.hideLegend) {
                    var legend = grade.getLegendColorDic();
                    // 标注
                    var html = "<div id='choroplethLegend' class='choropleth-legend'>";
                    for (var key in legend) {
                        var corlorHtml = "rgba(" + legend[key] + ")";
                        html += "<div class='view_legend' style='margin-top: 3px;display: inline-block';>" + "<div class='legend_title' title='" + key + "'>" + key + " </div>" + "<span style='display: inline-block;margin-left: 10px;width: 30px;height: 10px;background:" + corlorHtml + "'></span></div>"
                    }
                    html += "</div>";
                    $("#globalContainer").append(html);
                }
                viewer.flyTo(promise);
            }
            return source;
        };
    }


    Model.prototype = {
        //兼容index.js
        setViewer: function (viewer) {
            chartView = new CompositeView("globalContainer", viewer);
        },
        /**
         * 渲染接口
         * @param result 数据
         * @param type 渲染类型
         * @param large 是否大数据量
         */
        addModel: function (result, type, large) {
            try {
                this.clear();
                this._viewer = MapUtils.viewer;
                switch (type) {
                    case("geojson"):
                        //viewer
                        choropleth(result, this);
                        break;
                    case("jsonArr"):
                        //viewer
                        choroplethByArr(result, this);
                        break;
                    case("points"):
                        //chart
                        scatter(result, this);
                        break;
                    case("lines"):
                        lines(result);
                        break;
                    case("flow"):
                        //chart
                        flow(result, large);
                        break;
                    case("heat"):
                        //mapV point value
                        heat(result, this);
                        break;
                    case("honeyComb"):
                        //mapV Point value
                        honeyComb(result, this);
                        break;
                    case("honeyCombMapV"):
                        honeyCombMapV(result);
                        break;
                    case ("migration"):
                        migration(result, this);
                        break;
                    case ("highHeat"):
                        this._option.useHSLColor = true;
                        highHeat(result, this);
                        break;
                    case ("scatter"):
                        scatterMapV(result, this);
                        break;
                    case ("line_model"):
                        lineEcharts(result, this);
                        break;
                    case ("bar_model"):
                        barEcharts(result, this);
                        break;
                    case ("pie_model"):
                        pieEcharts(result, this);
                        break;
                    case ("migration_model"):
                        migrationModel(result, this);
                        break;
                    case ("radar_model"):
                        radarModel(result, this);
                        break;
                    case ("dashboard_model"):
                        dashboardModel(result, this);
                        break;

                }
                if (this._option.setMapView) {
                    try {
                        MapUtils.setMapView(this._option.setMapView);
                    } catch (e) {

                    }
                }
                //图形加载结束
                EventBus.trigger("themeLoaded");
            } catch (e) {
                console.log(e);
                layer.msg(e.message || '渲染出错。', {
                    time: 2000, //自动关闭
                    btn: false
                });
            }

        },

        /**
         * 清除接口
         */
        clear: function () {
            if (echartsComponents) {
                try {
                    echartsComponents.clear();
                }catch (e) {

                }
            }

            // 移除mapV图层
            if (this._mapVLyr.length > 0) {
                this._mapVLyr[0].destroy();
                this._mapVLyr = [];
            }
            MapUtils.viewer.entities.removeAll();
            // 移除cesium渲染的geojson
            if (source) {
                $("#choroplethLegend").remove();
                self._viewer.dataSources.remove(source);
            }
            MapUtils.viewer.dataSources.removeAll();
            $("#choroplethLegend").remove();

            // 移除echarts
            if (chartView) {
                chartView.destroy();
            }

        },

        /**
         * set参数
         * @param option
         */
        setOption: function (option) {
            $.extend(this._option, option);
        },

        /**
         * 时间轴渲染热力图接口
         * @param result
         * @returns {*}
         */
        addHeat: function (result) {
            this._viewer = MapUtils.viewer;
            self.clear();
            var data = [];
            $.each(result, function (i, e) {
                data.push({
                    geometry: {type: "Point", coordinates: e.coords},
                    count: e.value
                });
            });
            self._center = data[0].geometry.coordinates;
            var option = {
                size: self._option.size,
                draw: "heatmap",
                gradient: {0.25: "rgb(0,0,255)", 0.55: "rgb(0,255,0)", 0.85: "yellow", 1.0: "rgb(255,0,0)"},
                max: 100,
                lineWidth: 10,
                shadowBlur: 50,
            };
            var dataSet = new mapV.DataSet(data);
            var lyr = new MapVLayer(self._viewer, dataSet, option);
            self._mapVLyr.push(lyr);
            return lyr;
        }


    };

    /********************************************************【模版方法】************/

    /**
     * 等值面 1-5级 无级别默认颜色
     * @param result
     * @returns {*}
     */
    function choropleth(result, model) {
        var viewer = self._viewer;
        // var legend = {"一级:":"#add8e6","二级:":"#6495ed","三级:":"#ff7f50","四级:":"#2dad2d","五级:":"#ff00ff"};
        var temp = {};
        temp.type = result.type;
        temp.features = result.features;
        // var lineGeojson = MapUtils.polygonToLine(temp);
        // line(lineGeojson,{
        //     clampToGround:true
        // });


        /**
         * 渲染字段
         * @type {*|string}
         */
        var propName = model._option.prop || "SCORE";
        if (model._option.heightByArea) {
            $(temp.features).each(function (i, e) {
                var area = 0;
                try {
                    var coor = e.geometry.coordinates;
                    var polygon = turf.polygon(coor);
                    area = turf.area(polygon);
                } catch (e) {
                    console.log(e);
                }
                if (area > 50000000) {
                    area = 300;
                } else {
                    area = 50;
                }
                e.properties["heightByArea"] = area;
            });
        }
        var propHeight = model._option.propHeight;
        var promise;
        var grade = model.grade(result, model._option);
        layer.msg(model._option.msg || '等值面图生成中...', {
            time: 2000, //自动关闭
            btn: false
        });
        if (model._option.clampToGround) {
            promise = Cesium.GeoJsonDataSource.load(temp, {
                stroke: Cesium.Color.HOTPINK,
                camera: viewer.scene.camera,
                canvas: viewer.scene.canvas,
                clampToGround: true
            });
        } else {
            promise = Cesium.GeoJsonDataSource.load(temp);
        }
        promise.then(function (dataSource) {
            self._viewer.dataSources.add(dataSource);
            source = dataSource;

            //如果无指定字段
            var targetEntity = dataSource.entities.values[0];

            //隐藏图例
            if (targetEntity.properties[propName] === undefined) {
                model._option.hideLegend = true;
                // $("#choroplethLegend").hide();
            }
            var entities = dataSource.entities.values;
            // 分级
            for (var i = 0; i < entities.length; i++) {
                try {
                    var entity = entities[i];
                    var score;
                    var value = entity.properties[propName];
                    var heightValue = entity.properties[propHeight];
                    if (model._option.heightValueCallback) {
                        heightValue = model._option.heightValueCallback(heightValue);
                    }
                    if (model._option.heightMultiple && heightValue) {
                        heightValue = heightValue * model._option.heightMultiple;
                    }
                    if (model._option.heightByArea) {
                        heightValue = entity.properties["heightByArea"];
                        if (heightValue < 150) {
                            heightValue = 150;
                        }
                    }
                    if (!value) {
                        score = 1;
                    } else {
                        value = value._value;
                    }
                    if (model._option.gradeType == "string") {
                        score = value;
                    } else {
                        score = value * 1;
                    }
                    var color = grade.getColor(score);

                    entity.polygon.material = color;
                    if (!model._option.clampToGround) {
                        entity.polygon.extrudedHeight = heightValue || model._option.polygonHeight || 150;
                    }
                    entity.polygon.outline = true;
                    entity.polygon.outlineWidth = model._option.lineWidth || 800;
                    entity.polygon.outlineColor = new Cesium.Color.fromBytes(0, 0, 200, model._option.lineOpcity || 30);
                } catch (e) {
                }
            }

        });

        if (!model._option.hideLegend) {
            var legend = grade.getLegendColorDic();
            // 标注
            var html = "<div id='choroplethLegend' class='choropleth-legend'>";
            for (var key in legend) {
                var corlorHtml = "rgba(" + legend[key] + ")";
                html += "<div class='view_legend' style='margin-top: 3px;display: inline-block';>" + "<div class='legend_title' title='" + key + "'>" + key + " </div>" + "<span style='display: inline-block;margin-left: 10px;width: 30px;height: 10px;background:" + corlorHtml + "'></span></div>"
            }
            html += "</div>";
            $("#globalContainer").append(html);
        }
        self._viewer.flyTo(promise);
        return source;
    }

    /**
     * 渲染二维线模板
     */
    function line(geojson, option) {
        var _option = option || this._option;
        promise = Cesium.GeoJsonDataSource.load(geojson);
        self._viewer.flyTo(promise);
    }

    /*渐变面*/
    function intensity(result) {
        var aa = function (i, j) {
            return i - j;  //i-j为升序，
        };
        var bb = function (i, j) {
            return j - i; //j-i为降序
        };
        var score = [];
        self.clear();
        var data = [], polygon = [];
        data = result.features;
        $.each(data, function (i, e) {
            e.count = e.properties.SCORE;
            score.push(e.properties.SCORE * 1);
            polygon.push(e);
        });
        var max = score.sort(bb)[0];
        var option = {
            gradient: {
                0: 'rgba(50, 50, 255, 0.8)',
                0.5: 'rgba(255, 255, 50, 0.8)',
                1: 'rgba(255, 55, 50, 0.8)'
            },
            max: max,
            draw: 'intensity'
        };
        var dataSet = new mapV.DataSet(polygon);
        var lyr = new MapVLayer(self._viewer, dataSet, option);
        self._mapVLyr.push(lyr);
        return lyr;
    }

    /*流向图*/
    function flow(line, large) {
        var points = [];
        $.each(line, function (i, e) {
            var temp = {level: 30};
            if (i === 0) {
                temp.name = e.fromName;
                temp.value = e.coords[0];
                points.push(JSON.parse(JSON.stringify(temp)));
            }
            temp.name = e.toName;
            temp.value = e.coords[1];
            points.push(temp);
        });
        if (large) {
            var option = {
                GLMap: {},
                series: [{
                    blendMode: "lighter",
                    coordinateSystem: "GLMap",
                    data: line,
                    effect: {
                        show: true,
                        trailWidth: 2,
                        trailLength: 0.15,
                        trailOpacity: 1,
                        trailColor: "rgb(30, 30, 60)"
                    },
                    lineStyle: {width: 1, color: "rgb(50, 50, 150)", opacity: 0.1},
                    type: "lines",
                    zlevel: 0
                }]
            }
        } else {
            var option = {
                animation: false,
                backgroundColor: "rgba(116, 112, 124, 0.2)",
                title: {
                    text: self._option.title,
                    left: 'center'
                },
                GLMap: {},
                tooltip: {trigger: "item"},
                series: [
                    {
                        type: "effectScatter",
                        coordinateSystem: "GLMap",
                        data: points,
                        symbolSize: function (e) {
                            return (20);
                        },
                        showEffectOn: "render",
                        rippleEffect: {brushType: "stroke"},
                        hoverAnimation: !0,
                        label: {normal: {formatter: "{b}", position: "right", show: true, color: "#333"}},
                        itemStyle: {normal: {color: 'orange', shadowBlur: 60, shadowColor: "#cccccc"}},
                        zlevel: 1
                    },
                    {
                        type: "lines",
                        coordinateSystem: "GLMap",
                        lineStyle: {
                            normal: {
                                color: self._option.color,
                                curveness: 0.2,
                                opacity: 0.8,
                                width: self._option.size / 10
                            }
                        },
                        effect: {
                            period: 6,
                            show: true,
                            symbol: "arrow",
                            color: "red",
                            symbolSize: 10,
                            trailLength: 0.1
                        },
                        data: line,
                        zlevel: 2
                    },
                    {
                        type: "lines",
                        coordinateSystem: "GLMap",
                        lineStyle: {
                            normal: {
                                color: self._option.color,
                                curveness: 0.2,
                                width: self._option.size / 10
                            }
                        },
                        effect: {
                            color: "#fff",
                            period: 6,
                            show: true,
                            symbolSize: 10,
                            trailLength: 0.7
                        },
                        data: line,
                        zlevel: 3
                    }
                ]
            };
        }
        chartView.updateOverlay(option);
        flyToP(self._center, 20000);
    }

    /*热力图*/
    function heat(result, model) {
        model._option.hideLegend = true;
        var grade = model.grade(result, model._option);
        model._option.maxValue = grade.getMax();
        model._option.minValue = grade.getMin();
        model._option.dataType = grade.getDataType();
        model._option.modelType = "heat";
        var data = model.feaColToMapV(result, model._option);
        self._center = data[0].geometry.coordinates;
        var option = {
            size: self._option.size,
            draw: "heatmap",
            //gradient: { 0.25: "rgb(0,0,255)", 0.55: "rgb(0,255,0)", 0.85: "yellow", 1.0: "rgb(255,0,0)"},
            // gradient: { 0.25: "rgb(0,0,255)", 0.55: "rgb(0,255,0)", 0.85: "yellow", 1.0: "rgb(255,0,0)"},
            //
            // max:100
        };
        option["gradient"] = grade.getHeatColor([[0, 0, 255], [0, 255, 0], [255, 255, 0], [0, 255, 255], [255, 0, 0]]);
        // option.animation={
        //     type: 'time',
        //         stepsRange: {
        //         start: 0,
        //             end: 80
        //     },
        //     trails: 15,
        //         duration: 5
        // }
        var dataSet = new mapV.DataSet(data);
        var lyr = new MapVLayer(self._viewer, dataSet, option);
        self._mapVLyr.push(lyr);
        flyToP(self._center, 2000000);

        if (!model._option.hideLegend) {
            var legend = grade.getLegendColorDic();
            // 标注
            var html = "<div id='choroplethLegend' style='width: 100px' class='choropleth-legend'>";
            for (var key in legend) {
                var corlorHtml = "rgba(" + legend[key] + ")";
                html += "<div class='view_legend' style='margin-top: 3px;display: inline-block';>" + "<div style='width: 40px;display: inline-block;overflow: hidden;'>" + key.substr(0, 4) + " </div>" + "<span style='display: inline-block;margin-left: 10px;width: 30px;height: 10px;background:" + corlorHtml + "'></span></div>"
            }
            html += "</div>";
            $("#globalContainer").append(html);
        }
        return lyr;
    }

    /*折线图*/
    function lineEcharts(result, model) {
        $("#div-bottom-pane").slideLeftShow(500);
        /*$("#echarts-components").css("height", "430px");
        $("#echarts-components").css("width", "1140px");
        $("#echarts-components").empty();*/
        var name = model._option.name;
        var value = model._option.value;

        var dataName = [];
        var dataVale = [];
        if (result.features.length > 0) {
            $.each(result.features, function (i, e) {
                if (e.properties.hasOwnProperty(name)) {
                    dataName.push(e.properties[name]);
                } else {
                    dataName.push("");
                }
                if (e.properties.hasOwnProperty(value)) {
                    dataVale.push(e.properties[value]);
                } else {
                    dataVale.push(0);
                }
            });
        }
        //echartsComponents = echarts.init(document.getElementById('echarts-components'), 'macarons');
        var option = {
            xAxis: {
                type: 'category',
                data: []
            },
            yAxis: {
                type: 'value'
            },
            series: [{
                data: [],
                type: 'line'
            }]
        };
        option.xAxis.data = dataName;
        option.series[0].data = dataVale;
        echartsComponents.setOption(option);

    }


    /*雷达图模版*/
    function radarModel(result, model) {
        $("#div-bottom-pane").slideLeftShow(500);
        /*$("#echarts-components").css("height", "430px");
        $("#echarts-components").css("width", "1140px");
        $("#echarts-components").empty();*/
        var name = model._option.name;
        var value = model._option.value;
        var max = model._option.max;

        var dataName = [];
        var dataVale = [];
        if (result.features.length > 0) {
            $.each(result.features, function (i, e) {
                var dataMap = {};
                if (e.properties.hasOwnProperty(name)) {
                    dataMap.name = e.properties[name];
                } else {
                    dataMap.name = "";
                }
                if (e.properties.hasOwnProperty(max)) {
                    dataMap.max = e.properties[max];
                } else {
                    dataMap.max = "";
                }
                dataName.push(dataMap);
                if (e.properties.hasOwnProperty(value)) {
                    dataVale.push(e.properties[value]);
                } else {
                    dataVale.push(0);
                }
            });
        }
        //echartsComponents = echarts.init(document.getElementById('echarts-components'), 'macarons');
        var option = {
            radar: {
                indicator: []
            },
            series: [{
                type: 'radar',
                data: [
                    {
                        value: []
                    }
                ]
            }]
        };
        option.radar.indicator = dataName;
        option.series[0].data = dataVale;
        echartsComponents.setOption(option);
    }

    /*仪表盘模版*/
    function dashboardModel(result, model) {
        $("#div-bottom-pane").slideLeftShow(500);
        /*$("#echarts-components").css("height", "430px");
        $("#echarts-components").css("width", "1140px");
        $("#echarts-components").empty();*/
        var value = model._option.value;
        var max = model._option.max;

        var dataMax = 0;
        var dataVale = 0;
        if (result.features.length > 0) {
            var e = result.features[0];

            if (e.properties.hasOwnProperty(max)) {
                dataMax = e.properties[max];
            } else {
                dataMax = 0;
            }

            if (e.properties.hasOwnProperty(value)) {
                dataVale = e.properties[value];
            } else {
                dataVale = 0;
            }
        }
        //echartsComponents = echarts.init(document.getElementById('echarts-components'), 'macarons');
        var option = {
            series: [
                {
                    type: 'gauge',
                    data: [{value: 50}]
                }
            ]
        };
        option.series[0].max = dataMax;
        option.series[0].data[0].value = dataVale;
        echartsComponents.setOption(option);
    }

    /*条形图*/
    function barEcharts(result, model) {
        $("#div-bottom-pane").slideLeftShow(500);
        var title = model._option.title;
        var name = model._option.name;
        var value = model._option.value;

        var dataName = [];
        var dataVale = [];
        if (result.features.length > 0) {
            $.each(result.features, function (i, e) {
                if (e.properties.hasOwnProperty(name)) {
                    dataName.push(e.properties[name]);
                } else {
                    dataName.push("");
                }
                if (e.properties.hasOwnProperty(value)) {
                    dataVale.push(e.properties[value]);
                } else {
                    dataVale.push(0);
                }
            });
        }

        var option = {
            title:{left :'center'},
            color: ['#3398DB'],
            tooltip: {
                trigger: 'axis',
                axisPointer: {            // 坐标轴指示器，坐标轴触发有效
                    type: 'shadow'        // 默认为直线，可选为：'line' | 'shadow'
                }
            },
            xAxis: [
                {
                    type: 'category',
                    data: [],
                    show:true,
                    axisTick: {
                        alignWithLabel: true
                    }
                }
            ],
            yAxis: [
                {
                    type: 'value'
                }
            ],
            series: [
                {
                    type: 'bar',
                    barWidth: '60%',
                    data: []
                }
            ]
        };
        option.title.text=title;
        option.xAxis[0].data = dataName;
        option.series[0].data = dataVale;
        echartsComponents.setOption(option);

    }

    /*饼图*/
    function pieEcharts(result, model) {
        $("#div-bottom-pane").slideLeftShow(500);
        /*$("#echarts-components").css("height", "430px");
        $("#echarts-components").css("width", "1140px");
        $("#echarts-components").empty();*/
        var name = model._option.name;
        var value = model._option.value;
        var title = model._option.title;

        var dataName = [];
        var dataVale = [];
        if (result.features.length > 0) {
            $.each(result.features, function (i, e) {
                var dataMap = {};
                if (e.properties.hasOwnProperty(name)) {
                    dataName.push(e.properties[name]);
                    dataMap.name = e.properties[name];
                } else {
                    dataName.push("");
                    dataMap.name = "";
                }
                if (e.properties.hasOwnProperty(value)) {
                    dataMap.value = e.properties[value];
                } else {
                    dataMap.value = 0;
                }
                dataVale.push(dataMap);
            });
        }
        //echartsComponents = echarts.init(document.getElementById('echarts-components'), 'macarons');
        var option = {
            title:{left :'center'},
            tooltip: {
                trigger: 'item',
                formatter: "{b} : {c} ({d}%)"
            },
            legend: {
                orient: 'vertical',
                left: 'left',
                data: []
            },
            series: [
                {
                    type: 'pie',
                    radius: '55%',
                    center: ['50%', '60%'],
                    data: [],
                    itemStyle: {
                        emphasis: {
                            shadowBlur: 10,
                            shadowOffsetX: 0,
                            shadowColor: 'rgba(0, 0, 0, 0.5)'
                        }
                    }
                }
            ]
        };
        option.title.text=title;
        option.legend.data=dataName;
        option.series[0].data = dataVale;
        echartsComponents.setOption(option);

    }

    function migrationModel(result, model) {

        var startNameP = model._option.startName;
        var destinationNameP = model._option.destinationName;
        var qdXP = model._option.qdX;
        var qdYP = model._option.qdY;
        var zdXP = model._option.zdX;
        var zdYP = model._option.zdY;
        if (result.features.length > 0) {
            $.each(result.features, function (i, e) {

                var startName = e.properties[startNameP];
                var destinationName = e.properties[destinationNameP];
                var qdX = e.properties[qdXP];
                var qdY = e.properties[qdYP];
                var zdX = e.properties[zdXP];
                var zdY = e.properties[zdYP];

                if (i == 0) {
                    self._center = [qdX, qdY];
                }

                // 用于拟合当前曲线的笛卡尔坐标点数组
                var startPt = Cesium.Cartesian3.fromDegrees(qdX, qdY, 0);
                var endPt = Cesium.Cartesian3.fromDegrees(zdX, zdY, 0);
                //根据两点拟合曲线
                var curLinePointsArr = generateCurve(startPt, endPt);
                self._viewer.entities.add({ // 背景线
                    description: 'background-line',
                    name: startName + ' -> ' + destinationName,
                    polyline: {
                        width: 1.5,
                        positions: curLinePointsArr,
                        material: new Cesium.PolylineDashMaterialProperty({
                            color: new Cesium.Color(118 / 255, 233 / 255, 241 / 255, 0.8)
                        })
                    }
                });

                self._viewer.entities.add({ // 尾迹线
                    description: 'trail-line',
                    name: startName + ' -> ' + destinationName,
                    polyline: {
                        width: 1,
                        positions: curLinePointsArr,
                        material: new Cesium.PolylineTrailMaterialProperty({ // 尾迹线材质
                            color: Cesium.Color.fromCssColorString("rgba(72, 61, 139, 1.0)"),
                            trailLength: 1,
                            period: 5.0
                        })
                    }
                });

                self._viewer.entities.add({ // 起点
                    description: 'start-point',
                    name: startName,
                    position: Cesium.Cartesian3.fromDegrees(qdX, qdY),
                    point: {
                        color: new Cesium.Color(255 / 255, 249 / 255, 0, 1),
                        pixelSize: 5
                    }
                });

                self._viewer.entities.add({ // 终点
                    description: 'end-point',
                    name: destinationName,
                    position: Cesium.Cartesian3.fromDegrees(zdX, zdY),
                    point: {
                        color: new Cesium.Color(251 / 255, 7 / 255, 0, 0.7),
                        pixelSize: 10
                    }
                });
            });
        }
        flyToP(self._center, 20000);

    }

    /**
     *
     * @param startPoint 起点
     * @param endPoint 终点
     * @returns {Array}
     */
    function generateCurve(startPoint, endPoint) {
        var addPointCartesian = new Cesium.Cartesian3();
        Cesium.Cartesian3.add(startPoint, endPoint, addPointCartesian);//计算两个笛卡尔分量和
        var midPointCartesian = new Cesium.Cartesian3();
        Cesium.Cartesian3.divideByScalar(addPointCartesian, 2, midPointCartesian);//通过比例除以分量
        var midPointCartographic = Cesium.Cartographic.fromCartesian(midPointCartesian);//从笛卡尔3d坐标创建新的椭球坐标
        midPointCartographic.height = Cesium.Cartesian3.distance(startPoint, endPoint) / 10;
        var midPoint = new Cesium.Cartesian3();
        Cesium.Ellipsoid.WGS84.cartographicToCartesian(midPointCartographic, midPoint);
        var spline = new Cesium.CatmullRomSpline({
            times: [0.0, 0.5, 1.0],
            points: [startPoint, midPoint, endPoint]
        });
        var curvePointsArr = [];
        for (var i = 0, len = 300; i < len; i++) {
            curvePointsArr.push(spline.evaluate(i / len));
        }
        return curvePointsArr;
    }


    $('#showtable-close').click(function () {
        echartsComponents.clear();
        $("#div-bottom-pane").slideLeftHide(500);
        $("#echarts-components").css("height", "0px");
        $("#echarts-components").css("width", "0px");
    });

    jQuery.fn.slideLeftHide = function (speed, callback) {
        this.animate({
            width: "hide",
            paddingLeft: "hide",
            paddingRight: "hide",
            marginLeft: "hide",
            marginRight: "hide"
        }, speed, callback);
    };
    jQuery.fn.slideLeftShow = function (speed, callback) {
        this.animate({
            width: "show",
            paddingLeft: "show",
            paddingRight: "show",
            marginLeft: "show",
            marginRight: "show"
        }, speed, callback);
    };

    /**
     * 柱状热力图
     * param:渲染字段、渲染颜色、高度倍数拉伸
     * 渲染颜色使用hsl(不可配置)
     */
    function highHeat(result, model) {
        //渲染字段
        var propName = model._option.prop || "SCORE";
        source = new Cesium.CustomDataSource('geoJsonDatasource');
        //添加阴影
        MapUtils.viewer.shadows = true;
        MapUtils.viewer.terrainShadows = Cesium.ShadowMode.ENABLED;
        var grade = model.grade(result, model._option);
        $.each(result.features, function (i, e) {
            var coors = e.geometry.coordinates;
            var properties = e.properties;
            var value = properties[propName];
            value = value || 50;
            if (model._option.heightMultiple) {
                value = value * model._option.heightMultiple;
            }
            var surfacePosition = Cesium.Cartesian3.fromDegrees(coors[0], coors[1], 0);
            var heightPosition = Cesium.Cartesian3.fromDegrees(coors[0], coors[1], value);
            var polyline = new Cesium.PolylineGraphics();
            polyline.width = new Cesium.ConstantProperty(model._option.pointWidth || 10);
            polyline.arcType = new Cesium.ConstantProperty(Cesium.ArcType.NONE);
            polyline.positions = new Cesium.ConstantProperty([surfacePosition, heightPosition]);
            polyline.material = grade.getColor(properties[propName] * 1);
            var entity = new Cesium.Entity({
                polyline: polyline,
                // seriesName : seriesName //Custom property to indicate series name
            });
            source.entities.add(entity);
        });
        promise = self._viewer.dataSources.add(source);
        self._viewer.flyTo(promise);
    }

    /* 蜂窝图*/
    function honeyComb(result, model) {
        var data = model.feaColToMapV(result, model._option);
        // var data = [];
        // $.each(result,function (i,e) {
        //     data.push({
        //         geometry: {type: "Point", coordinates: e.coords},
        //         count: e.value
        //     });
        // });
        self._center = data[0].geometry.coordinates;
        var option = {
            size: self._option.size,
            fillStyle: 'rgba(55, 50, 250, 0.8)',
            shadowColor: 'rgba(255, 250, 50, 1)',
            shadowBlur: 20,
            max: 100,
            label: { // 网格中显示累加的值总和
                show: true,
                fillStyle: 'white',
                shadowColor: 'yellow',
                font: '20px Arial',
                shadowBlur: 10
            },
            gradient: {0.25: "rgb(0,0,255)", 0.55: "rgb(0,255,0)", 0.85: "yellow", 1.0: "rgb(255,0,0)"},
            draw: 'honeycomb'
        };
        var dataSet = new mapV.DataSet(data);
        var lyr = new MapVLayer(self._viewer, dataSet, option);
        self._mapVLyr.push(lyr);
        flyToP(self._center, 20000);
        return lyr;
    }

    /**
     * 蜂窝图
     * geojson
     * @param result
     */
    function honeyCombMapV(result) {
        var data = [];
        $.each(result, function (i, e) {
            data.push({
                geometry: {type: "Point", coordinates: e.coords},
                count: e.value
            });
        });
        self._center = data[0].geometry.coordinates;
        var option = {
            size: self._option.size,
            fillStyle: 'rgba(55, 50, 250, 0.8)',
            shadowColor: 'rgba(255, 250, 50, 1)',
            shadowBlur: 20,
            max: 100,
            label: { // 网格中显示累加的值总和
                show: true,
                fillStyle: 'white',
                shadowColor: 'yellow',
                font: '20px Arial',
                shadowBlur: 10
            },
            gradient: {0.25: "rgb(0,0,255)", 0.55: "rgb(0,255,0)", 0.85: "yellow", 1.0: "rgb(255,0,0)"},
            draw: 'honeycomb'
        };
        var dataSet = new mapV.DataSet(data);
        var lyr = new MapVLayer(self._viewer, dataSet, option);
        self._mapVLyr.push(lyr);
        flyToP(self._center, 20000);
        return lyr;
    }


    /*散点图*/
    function scatterMapV(result, model) {
        var data = model.feaColToMapV(result, model._option);

        self._center = data[0].geometry.coordinates;
        var option = {
            fillStyle: 'rgba(255, 50, 50, 0.6)',
            maxSize: self._option.size,
            max: self._option.size * 2,
            draw: 'bubble'
        };
        var dataSet = new mapV.DataSet(data);
        var lyr = new MapVLayer(self._viewer, dataSet, option);
        self._mapVLyr.push(lyr);
        flyToP(self._center, 20000);
    }

    /*常规线*/
    function lines(result) {
        var data = [];
        $.each(result, function (i, e) {
            data.push({
                geometry: {type: "LineString", coordinates: e.coords},
                count: e.value,
                text: e.name
            });
        });
        self._center = data[0].geometry.coordinates;
        var option = {
            strokeStyle: self._option.color,
            shadowColor: 'rgba(255, 250, 50, 1)',
            shadowBlur: 20,
            lineWidth: self._option.size / 100,
            draw: 'simple'
        };
        var dataSet = new mapV.DataSet(data);
        var lyr = new MapVLayer(self._viewer, dataSet, option);
        self._mapVLyr.push(lyr);
        flyToP(self._center, 20000);
        return lyr;
    }

    /***************************************************************************/

    function flyToP(center, height) {
        var h = height ? height : 2000000;
        MapUtils.viewer.scene.camera.setView({ // 图层加载完成,设置相机位置
            destination: Cesium.Cartesian3.fromDegrees(center[0], center[1], h)
        });
    }

    return new Model;
});