define([
    'Cesium'
], function (
    Cesium
) {
    'use strict';

    // dpi 和像素之间转换
    //  Scale = 1 : (96 * 2 * Math.PI * 6378137 * resolution / 360 / 0.0254);
    var defaultTileInfo = {
        "rows": 256,
        "cols": 256,
        "dpi": 96,
        "format": "PNG24",
        "compressionQuality": 0,
        "origin": {
            "x": -180,
            "y": 90
        },
        "spatialReference": {
            "wkid": 4490,
            "latestWkid": 4490
        },
        "lods": [
            {
                "level": 0,
                "resolution": 1.406250026231578,
                "scale": 590995197.141668
            },
            {
                "level": 1,
                "resolution": 0.703125013115789,
                "scale": 295497598.570834
            },
            {
                "level": 2,
                "resolution": 0.3515625065578945,
                "scale": 147748799.285417
            },
            {
                "level": 3,
                "resolution": 0.17578125327894775,
                "scale": 73874399.6427087
            },
            {
                "level": 4,
                "resolution": 0.08789062663947399,
                "scale": 36937199.8213544
            },
            {
                "level": 5,
                "resolution": 0.043945313319736994,
                "scale": 18468599.9106772
            },
            {
                "level": 6,
                "resolution": 0.021972656659868472,
                "scale": 9234299.95533859
            },
            {
                "level": 7,
                "resolution": 0.010986328329934226,
                "scale": 4617149.97766929
            },
            {
                "level": 8,
                "resolution": 0.005493164164967124,
                "scale": 2308574.98883465
            },
            {
                "level": 9,
                "resolution": 0.0027465820824835504,
                "scale": 1154287.49441732
            },
            {
                "level": 10,
                "resolution": 0.0013732910412417797,
                "scale": 577143.747208662
            },
            {
                "level": 11,
                "resolution": 0.0006866455206208899,
                "scale": 288571.873604331
            },
            {
                "level": 12,
                "resolution": 0.0003433227603104438,
                "scale": 144285.936802165
            },
            {
                "level": 13,
                "resolution": 0.0001716613801552224,
                "scale": 72142.9684010827
            },
            {
                "level": 14,
                "resolution": 0.00008583069007761132,
                "scale": 36071.4842005414
            },
            {
                "level": 15,
                "resolution": 0.00004291534503880566,
                "scale": 18035.7421002707
            },
            {
                "level": 16,
                "resolution": 0.000021457672519402802,
                "scale": 9017.87105013534
            },
            {
                "level": 17,
                "resolution": 0.000010728836259701401,
                "scale": 4508.93552506767
            },
            {
                "level": 18,
                "resolution": 0.000005364418129850712,
                "scale": 2254.46776253384
            },
            {
                "level": 19,
                "resolution": 0.000002682209064925356,
                "scale": 1127.23388126692
            },
            {
                "level": 20,
                "resolution": 0.000001341104532462678,
                "scale": 563.61694063346
            }
        ]
    };

    /**
     * A tiling scheme for geometry referenced to a simple {@link GeographicProjection} where
     * longitude and latitude are directly mapped to X and Y.  This projection is commonly
     * known as geographic, equirectangular, equidistant cylindrical, or plate carrée.
     *
     * @alias GeographicTilingScheme
     * @constructor
     *
     * @param {Object} [options] Object with the following properties:
     * @param {Ellipsoid} [options.ellipsoid=Ellipsoid.WGS84] The ellipsoid whose surface is being tiled. Defaults to
     * the WGS84 ellipsoid.
     * @param {Rectangle} [options.rectangle=Rectangle.MAX_VALUE] The rectangle, in radians, covered by the tiling scheme.
     * @param {Number} [options.numberOfLevelZeroTilesX=2] The number of tiles in the X direction at level zero of
     * the tile tree.
     * @param {Number} [options.numberOfLevelZeroTilesY=1] The number of tiles in the Y direction at level zero of
     * the tile tree.
     */
    function GeographicTilingScheme(options) {
        options = Cesium.defaultValue(options, {});

        this._ellipsoid = Cesium.defaultValue(options.ellipsoid, Cesium.Ellipsoid.WGS84);
        this._rectangle = Cesium.defaultValue(options.rectangle, Cesium.Rectangle.MAX_VALUE);
        this._zoomOffset = Cesium.defaultValue(options.zoomOffset, 0);
        this._projection = new Cesium.GeographicProjection(this._ellipsoid);
        this._numberOfLevelZeroTilesX = Cesium.defaultValue(options.numberOfLevelZeroTilesX, 2);
        this._numberOfLevelZeroTilesY = Cesium.defaultValue(options.numberOfLevelZeroTilesY, 1);
        if (options.hasOwnProperty('tileInfo')) {
            this._tileInfo = options.tileInfo;
        }
        this._tileWidth = Cesium.defaultValue(options.tileWidth, 256);
        this._tileHeight = Cesium.defaultValue(options.tileHeight, 256);
    }

    Cesium.defineProperties(GeographicTilingScheme.prototype, {
        /**
         * Gets the ellipsoid that is tiled by this tiling scheme.
         * @memberof GeographicTilingScheme.prototype
         * @type {Ellipsoid}
         */
        ellipsoid: {
            get: function () {
                return this._ellipsoid;
            }
        },

        /**
         * Gets the rectangle, in radians, covered by this tiling scheme.
         * @memberof GeographicTilingScheme.prototype
         * @type {Rectangle}
         */
        rectangle: {
            get: function () {
                return this._rectangle;
            }
        },

        /**
         * Gets the map projection used by this tiling scheme.
         * @memberof GeographicTilingScheme.prototype
         * @type {MapProjection}
         */
        projection: {
            get: function () {
                return this._projection;
            }
        },


        zoomOffset: {
            get: function () {
                return this._zoomOffset;
            }
        }
    });

    /**
     * Gets the total number of tiles in the X direction at a specified level-of-detail.
     *
     * @param {Number} level The level-of-detail.
     * @returns {Number} The number of tiles in the X direction at the given level.
     */
    GeographicTilingScheme.prototype.getNumberOfXTilesAtLevel = function (level) {

        return this._numberOfLevelZeroTilesX << (level + this._zoomOffset);
    };

    /**
     * Gets the total number of tiles in the Y direction at a specified level-of-detail.
     *
     * @param {Number} level The level-of-detail.
     * @returns {Number} The number of tiles in the Y direction at the given level.
     */
    GeographicTilingScheme.prototype.getNumberOfYTilesAtLevel = function (level) {

        return this._numberOfLevelZeroTilesY << (level + this._zoomOffset);
    };

    /**
     * Transforms a rectangle specified in geodetic radians to the native coordinate system
     * of this tiling scheme.
     *
     * @param {Rectangle} rectangle The rectangle to transform.
     * @param {Rectangle} [result] The instance to which to copy the result, or undefined if a new instance
     *        should be created.
     * @returns {Rectangle} The specified 'result', or a new object containing the native rectangle if 'result'
     *          is undefined.
     */
    GeographicTilingScheme.prototype.rectangleToNativeRectangle = function (rectangle, result) {
        //>>includeStart('debug', pragmas.debug);
        Cesium.Check.Cesium.defined('rectangle', rectangle);
        //>>includeEnd('debug');

        var west = Cesium.Math.toDegrees(rectangle.west);
        var south = Cesium.Math.toDegrees(rectangle.south);
        var east = Cesium.Math.toDegrees(rectangle.east);
        var north = Cesium.Math.toDegrees(rectangle.north);

        if (!Cesium.defined(result)) {
            return new Cesium.Rectangle(west, south, east, north);
        }

        result.west = west;
        result.south = south;
        result.east = east;
        result.north = north;
        return result;
    };

    /**
     * Converts tile x, y coordinates and level to a rectangle expressed in the native coordinates
     * of the tiling scheme.
     *
     * @param {Number} x The integer x coordinate of the tile.
     * @param {Number} y The integer y coordinate of the tile.
     * @param {Number} level The tile level-of-detail.  Zero is the least detailed.
     * @param {Object} [result] The instance to which to copy the result, or undefined if a new instance
     *        should be created.
     * @returns {Rectangle} The specified 'result', or a new object containing the rectangle
     *          if 'result' is undefined.
     */
    GeographicTilingScheme.prototype.tileXYToNativeRectangle = function (x, y, level, result) {
        var rectangleRadians = this.tileXYToRectangle(x, y, level, result);
        rectangleRadians.west = Cesium.Math.toDegrees(rectangleRadians.west);
        rectangleRadians.south = Cesium.Math.toDegrees(rectangleRadians.south);
        rectangleRadians.east = Cesium.Math.toDegrees(rectangleRadians.east);
        rectangleRadians.north = Cesium.Math.toDegrees(rectangleRadians.north);
        return rectangleRadians;
    };

    /**
     * Converts tile x, y coordinates and level to a cartographic rectangle in radians.
     *
     * @param {Number} x The integer x coordinate of the tile.
     * @param {Number} y The integer y coordinate of the tile.
     * @param {Number} level The tile level-of-detail.  Zero is the least detailed.
     * @param {Object} [result] The instance to which to copy the result, or undefined if a new instance
     *        should be created.
     * @returns {Rectangle} The specified 'result', or a new object containing the rectangle
     *          if 'result' is undefined.
     */
    GeographicTilingScheme.prototype.tileXYToRectangle = function (x, y, level, result) {
        var rectangle = this._rectangle;


        var xTiles = this.getNumberOfXTilesAtLevel(level);
        var yTiles = this.getNumberOfYTilesAtLevel(level);

        var xTileWidth = rectangle.width / xTiles;
        var yTileHeight = rectangle.height / yTiles;

        var west = x * xTileWidth + rectangle.west,
            east = (x + 1) * xTileWidth + rectangle.west,
            north = rectangle.north - y * yTileHeight,
            south = rectangle.north - (y + 1) * yTileHeight;

        if (Cesium.defined(this._tileInfo) && this._tileInfo.lods.length > level) {
            var resolution = this._tileInfo.lods[level].resolution * Cesium.Math.TWO_PI / 360;
            west = -Cesium.Math.PI + x * this._tileInfo.cols * resolution;
            east = -Cesium.Math.PI + (x + 1) * this._tileInfo.cols * resolution;

            north = Cesium.Math.PI_OVER_TWO - y * this._tileInfo.rows * resolution;
            south = Cesium.Math.PI_OVER_TWO - (y + 1) * this._tileInfo.rows * resolution;
        }


        if (!Cesium.defined(result)) {
            result = new Cesium.Rectangle(west, south, east, north);
        }

        result.west = west;
        result.south = south;
        result.east = east;
        result.north = north;
        return result;
    };

    /**
     * Calculates the tile x, y coordinates of the tile containing
     * a given cartographic position.
     *
     * @param {Cartographic} position The position.
     * @param {Number} level The tile level-of-detail.  Zero is the least detailed.
     * @param {Cartesian2} [result] The instance to which to copy the result, or undefined if a new instance
     *        should be created.
     * @returns {Cartesian2} The specified 'result', or a new object containing the tile x, y coordinates
     *          if 'result' is undefined.
     */
    GeographicTilingScheme.prototype.positionToTileXY = function (position, level, result) {
        var rectangle = this._rectangle;
        if (!Cesium.Rectangle.contains(rectangle, position)) {
            // outside the bounds of the tiling scheme
            return undefined;
        }

        var xTiles = this.getNumberOfXTilesAtLevel(level);
        var yTiles = this.getNumberOfYTilesAtLevel(level);

        var xTileWidth = rectangle.width / xTiles;
        var yTileHeight = rectangle.height / yTiles;

        if (Cesium.defined(this._tileInfo) && this._tileInfo.lods.length > level) {
            var resolution = this._tileInfo.lods[level].resolution * Cesium.Math.TWO_PI / 360;
            xTileWidth = this._tileInfo.cols * resolution;
            yTileHeight = this._tileInfo.rows * resolution;
        }

        var longitude = position.longitude;
        if (rectangle.east < rectangle.west) {
            longitude += Cesium.Math.TWO_PI;
        }

        var xTileCoordinate = (longitude - rectangle.west) / xTileWidth | 0;
        if (xTileCoordinate >= xTiles) {
            xTileCoordinate = xTiles - 1;
        }

        var yTileCoordinate = (rectangle.north - position.latitude) / yTileHeight | 0;
        if (yTileCoordinate >= yTiles) {
            yTileCoordinate = yTiles - 1;
        }

        if (!Cesium.defined(result)) {
            return new Cesium.Cartesian2(xTileCoordinate, yTileCoordinate);
        }

        result.x = xTileCoordinate;
        result.y = yTileCoordinate;
        return result;
    };

    return GeographicTilingScheme;
});
