/**
 * 关于cesium相机计算的一些方法和拟合地图层级函数
 * @author by <a>cyj</a>
 * @version v1.0 2019/9/10/14:56  Copyright gtmap Corp
 */

define([
    "global/utils/MapUtils",
    "Cesium"
        ],function (MapUtils,Cesium) {

    var that;
    function CameraAndScale() {
        try {
            // 赋值
            that = this;
            this._viewer = MapUtils.viewer;
        }catch (e){console.log(e)}
    }

    /*类接口*/
    CameraAndScale.prototype = {

        /**
         * 获取屏幕范围
         */
        getCameraExtent: function () {
            this._viewer = MapUtils.viewer;
            return computeCameraExtent();
        },

        /**
         * 获取屏幕中心点
         */
        getCenterPoint: function () {
            this._viewer = MapUtils.viewer;
            return getCenterPosition();
        },

        /**
         * 获取层级
         * @param height
         */
        getLevel: function (height) {
            this._viewer = MapUtils.viewer;
            return altitudeToZoom(height);
        },

        /**
         * 获取比例尺下标准层级
         * @param sr
         * @returns {*[]}
         */
        getResolution: function (sr) {
            this._viewer = MapUtils.viewer;
            return getScale(sr);
        }

    };


/*****************************************************方法函数 ***/

    /**
     * 根据canvas宽高计算屏幕两点，转换为extent
     * @returns {{}}
     */
    function computeCameraExtent(){
        // 范围对象
        var sceneExtent = {};
        try {

            // 得到当前三维场景
            var scene = that._viewer.scene;

            // 得到当前三维场景的椭球体
            var ellipsoid = scene.globe.ellipsoid;
            var canvas = scene.canvas;

            // canvas左上角
            var car3_lt = that._viewer.camera.pickEllipsoid(new Cesium.Cartesian2(0, 0), ellipsoid);

            // canvas右下角
            var car3_rb = that._viewer.camera.pickEllipsoid(new Cesium.Cartesian2(canvas.width, canvas.height), ellipsoid);

            // 当canvas左上角和右下角全部在椭球体上
            if (car3_lt && car3_rb) {
                var carto_lt = ellipsoid.cartesianToCartographic(car3_lt);
                var carto_rb = ellipsoid.cartesianToCartographic(car3_rb);
                sceneExtent.xmin = Cesium.Math.toDegrees(carto_lt.longitude);
                sceneExtent.ymax = Cesium.Math.toDegrees(carto_lt.latitude);
                sceneExtent.xmax = Cesium.Math.toDegrees(carto_rb.longitude);
                sceneExtent.ymin = Cesium.Math.toDegrees(carto_rb.latitude);
            }

            // 当canvas左上角不在但右下角在椭球体上
            else if (!car3_lt && car3_rb) {
                var car3_lt2 = null;
                var yIndex = 0;
                do {
                    // 这里每次10像素递加，一是10像素相差不大，二是为了提高程序运行效率
                    yIndex <= canvas.height ? yIndex += 10 : canvas.height;
                    car3_lt2 = that._viewer.camera.pickEllipsoid(new Cesium.Cartesian2(0, yIndex), ellipsoid);
                } while (!car3_lt2);
                var carto_lt2 = ellipsoid.cartesianToCartographic(car3_lt2);
                var carto_rb2 = ellipsoid.cartesianToCartographic(car3_rb);
                sceneExtent.xmin = Cesium.Math.toDegrees(carto_lt2.longitude);
                sceneExtent.ymax = Cesium.Math.toDegrees(carto_lt2.latitude);
                sceneExtent.xmax = Cesium.Math.toDegrees(carto_rb2.longitude);
                sceneExtent.ymin = Cesium.Math.toDegrees(carto_rb2.latitude);
            }

            // 获取rectangle
            sceneExtent.rectangle =
                new Cesium.Rectangle.fromDegrees(sceneExtent.xmin, sceneExtent.ymin, sceneExtent.xmax, sceneExtent.ymax);

        }catch (error){console.log(error)}
        // return
        return sceneExtent;
    }

    /**
     * 获取屏幕中心点
     * @returns {{lon: number, lat: number, height, zoom: number}}
     */
    function getCenterPosition() {
        // 屏幕坐标转化
        var result =
            that._viewer.camera.pickEllipsoid(new Cesium.Cartesian2(that._viewer.canvas.clientWidth / 2, that._viewer.canvas
            .clientHeight / 2));
        var curPosition = Cesium.Ellipsoid.WGS84.cartesianToCartographic(result);
        var lon = curPosition.longitude * 180 / Math.PI;
        var lat = curPosition.latitude * 180 / Math.PI;
        var height = getHeight();
        return {
            lon: lon,
            lat: lat,
            height: height,
            zoom: altitudeToZoom(height)
        };
    }

    /**
     * 获取高度
     */
    function getHeight() {
        if (that._viewer) {
            var scene = that._viewer.scene;
            var ellipsoid = scene.globe.ellipsoid;
            return ellipsoid.cartesianToCartographic(that._viewer.camera.position).height;
        }
    }


    /**
     * 拟合函数
     * @param altitude
     * @returns {number}
     */
    function altitudeToZoom(altitude) {
        var A = 40487.57;
        var B = 0.00007096758;
        var C = 91610.74;
        var D = -40467.74;

        return Math.round(D+(A-D)/(1+ Math.pow(altitude/C, B)));
    }


    /**
     * 比例尺层级
     * @returns {*[]}
     */
    function getScale(sr) {
         return resolution[sr + "-resolution"];
    }

    /***********************************************【定义坐标系】****/

    var resolution = {
        "4490-resolution":[
            {
                "level": 0,
                "resolution": 1.406250026231578,
                "scale": 590995197.141668
            },
            {
                "level": 1,
                "resolution": 0.703125013115789,
                "scale": 295497598.570834
            },
            {
                "level": 2,
                "resolution": 0.3515625065578945,
                "scale": 147748799.285417
            },
            {
                "level": 3,
                "resolution": 0.17578125327894775,
                "scale": 73874399.6427087
            },
            {
                "level": 4,
                "resolution": 0.08789062663947399,
                "scale": 36937199.8213544
            },
            {
                "level": 5,
                "resolution": 0.043945313319736994,
                "scale": 18468599.9106772
            },
            {
                "level": 6,
                "resolution": 0.021972656659868472,
                "scale": 9234299.95533859
            },
            {
                "level": 7,
                "resolution": 0.010986328329934226,
                "scale": 4617149.97766929
            },
            {
                "level": 8,
                "resolution": 0.005493164164967124,
                "scale": 2308574.98883465
            },
            {
                "level": 9,
                "resolution": 0.0027465820824835504,
                "scale": 1154287.49441732
            },
            {
                "level": 10,
                "resolution": 0.0013732910412417797,
                "scale": 577143.747208662
            },
            {
                "level": 11,
                "resolution": 0.0006866455206208899,
                "scale": 288571.873604331
            },
            {
                "level": 12,
                "resolution": 0.0003433227603104438,
                "scale": 144285.936802165
            },
            {
                "level": 13,
                "resolution": 0.0001716613801552224,
                "scale": 72142.9684010827
            },
            {
                "level": 14,
                "resolution": 0.00008583069007761132,
                "scale": 36071.4842005414
            },
            {
                "level": 15,
                "resolution": 0.00004291534503880566,
                "scale": 18035.7421002707
            },
            {
                "level": 16,
                "resolution": 0.000021457672519402802,
                "scale": 9017.87105013534
            },
            {
                "level": 17,
                "resolution": 0.000010728836259701401,
                "scale": 4508.93552506767
            },
            {
                "level": 18,
                "resolution": 0.000005364418129850712,
                "scale": 2254.46776253384
            },
            {
                "level": 19,
                "resolution": 0.000002682209064925356,
                "scale": 1127.23388126692
            },
            {
                "level": 20,
                "resolution": 0.000001341104532462678,
                "scale": 563.61694063346
            }
        ],
        "4326-resolution":[]
    };

    return new CameraAndScale;
});