/**
 * ArcGisMapServerVectorTileImageryProvider
 *
 *
 *
 * @example var layer = new ArcGisMapServerVectorTileImageryProvider({
            url: url,
            tilingScheme: geographicTilingScheme // 2000切片方案
        });
 var geographicTilingScheme = new GeographicTilingScheme({
            ellipsoid: Utils.config.ellipsolid['EPSG:4490'],
            rectangle: Cesium.Rectangle.fromDegrees(-180, -90, 180, 90),
            numberOfLevelZeroTilesX: 4,
            numberOfLevelZeroTilesY: 2
        });
 *
 * @author by <a href="mailto:chaomashi@gmail.com">Du</a>
 * @version v1.0 2019/6/25/15:32  Copyright gtmap Corp
 */

define(['Cesium',
        'ol',
        './GeographicTilingScheme'
    ], function (
    Cesium,
    ol,
    GeographicTilingScheme
    ) {
        /**
         * 矢量地图加载
         * @param options
         * @constructor
         */

        var that;

        function ArcGisMapServerVectorTileImageryProvider(options) {

            options = Cesium.defaultValue(options, Cesium.defaultValue.EMPTY_OBJECT);

            this._readyPromise = Cesium.when.resolve(true);
            this._mvtParser = new ol.format.MVT();
            this._key = Cesium.defaultValue(options.key, "");
            if (!Cesium.defined(options.url)) {
                throw new Cesium.DeveloperError('url is required!');
            }

            this._tilingScheme = Cesium.defaultValue(options.tilingScheme, new GeographicTilingScheme({ellipsoid: options.ellipsoid}));
            var resource = Cesium.Resource.createIfNeeded(options.url);
            resource.appendForwardSlash();
            if (Cesium.defined(options.token)) {
                resource.setQueryParameters({
                    token: options.token
                });
            }
            this._maximumLevel = options.maximumLevel;
            this._resource = resource;

            this._url = Cesium.defaultValue(options.url, "");

            this._useTiles = Cesium.defaultValue(options.usePreCachedTilesIfAvailable, true);
            this._pixelRatio = 1;
            this._transform = [.125, 0, 0, .125, 0, 0];
            this._replays = ["Default", "Image", "Polygon", "LineString", "Text"];
            this._tileQueue = new Cesium.TileReplacementQueue;
            this._cacheSize = Cesium.defaultValue(options.cacheSize, 1000);
            this._ready = false;
            this._tileInfo = [];
            that = this;
            var metadataError;

            function metadataSuccess(data) {
                var tileInfo = data.tileInfo;
                that._tileInfo = tileInfo;
                if (!Cesium.defined(tileInfo)) {
                    that._useTiles = true;
                } else {
                    that._tileWidth = tileInfo.rows;
                    that._tileHeight = tileInfo.cols;
                    if (tileInfo.spatialReference.wkid === 102100 || tileInfo.spatialReference.wkid === 102113) {
                        that._tilingScheme = new Cesium.WebMercatorTilingScheme({ellipsoid: options.ellipsoid});
                    } else if (data.tileInfo.spatialReference.wkid === 4326) {
                        that._tilingScheme = new Cesium.GeographicTilingScheme({ellipsoid: options.ellipsoid});
                    } else if (data.fullExtent.spatialReference.wkid === 4490) {
                        that._tilingScheme = new GeographicTilingScheme({
                            rectangle: Cesium.Rectangle.fromDegrees(data.fullExtent.xmin, data.fullExtent.ymin, data.fullExtent.xmax, data.fullExtent.ymax),
                            ellipsoid: options.ellipsoid,
                            tileInfo: data.tileInfo
                        });
                    } else {
                        var message = 'Tile spatial reference WKID ' + data.tileInfo.spatialReference.wkid + ' is not supported.';
                        metadataError = Cesium.TileProviderError.handleError(metadataError, that, that._errorEvent, message, undefined, undefined, undefined, requestMetadata);
                        return;
                    }

                    if (data.hasOwnProperty('maxzoom')) {
                        that._maximumLevel = data.maxzoom;
                    }

                    if (Cesium.defined(data.fullExtent)) {
                        if (Cesium.defined(data.fullExtent.spatialReference) && Cesium.defined(data.fullExtent.spatialReference.wkid)) {
                            if (data.fullExtent.spatialReference.wkid === 102100 || data.fullExtent.spatialReference.wkid === 102113) {
                                var projection = new Cesium.WebMercatorProjection();
                                var extent = data.fullExtent;
                                var sw = projection.unproject(new Cesium.Cartesian3(Math.max(extent.xmin, -that._tilingScheme.ellipsoid.maximumRadius * Math.PI), Math.max(extent.ymin, -that._tilingScheme.ellipsoid.maximumRadius * Math.PI), 0.0));
                                var ne = projection.unproject(new Cesium.Cartesian3(Math.min(extent.xmax, that._tilingScheme.ellipsoid.maximumRadius * Math.PI), Math.min(extent.ymax, that._tilingScheme.ellipsoid.maximumRadius * Math.PI), 0.0));
                                that._rectangle = new Cesium.Rectangle(sw.longitude, sw.latitude, ne.longitude, ne.latitude);
                            } else if (data.fullExtent.spatialReference.wkid === 4326 || data.fullExtent.spatialReference.wkid === 4490) {
                                that._rectangle = Cesium.Rectangle.fromDegrees(data.fullExtent.xmin, data.fullExtent.ymin, data.fullExtent.xmax, data.fullExtent.ymax);
                            } else {
                                var extentMessage = 'fullExtent.spatialReference WKID ' + data.fullExtent.spatialReference.wkid + ' is not supported.';
                                metadataError = Cesium.TileProviderError.handleError(metadataError, that, that._errorEvent, extentMessage, undefined, undefined, undefined, requestMetadata);
                                return;
                            }
                        }
                    } else {
                        that._rectangle = that._tilingScheme.rectangle;
                    }

                    // Install the default tile discard policy if none has been supplied.
                    // if (!Cesium.defined(that._tileDiscardPolicy)) {
                    //     that._tileDiscardPolicy = new Cesium.DiscardMissingTileImagePolicy({
                    //         missingImageUrl: buildImageResource(that, 0, 0, that._maximumLevel).url,
                    //         pixelsToCheck: [new Cesium.Cartesian2(0, 0), new Cesium.Cartesian2(200, 20), new Cesium.Cartesian2(20, 200), new Cesium.Cartesian2(80, 110), new Cesium.Cartesian2(160, 130)],
                    //         disableCheckIfAllPixelsAreTransparent: true
                    //     });
                    // }

                    that._useTiles = true;
                }

                if (Cesium.defined(data.copyrightText) && data.copyrightText.length > 0) {
                    that._credit = new Cesium.Credit(data.copyrightText);
                }

                that._ready = true;
                that._readyPromise.resolve(true);
                Cesium.TileProviderError.handleSuccess(metadataError);
            }


            function styleMetadataSuccess(data) {

                if (!Cesium.defined(data)) {
                    throw new Cesium.DeveloperError('arcgis vector tile is required!');
                }

                that._layerStyles = data;
            }

            function metadataFailure(e) {
                var message = 'An error occurred while accessing ' + that._resource.url + '.';
                metadataError = Cesium.TileProviderError.handleError(metadataError, that, that._errorEvent, message, undefined, undefined, undefined, requestMetadata);
                that._readyPromise.reject(new Cesium.RuntimeError(message));
            }

            /**
             * requestMetadata
             */
            function requestMetadata() {
                // arcgis service request
                var resource = that._resource.getDerivedResource({
                    queryParameters: {
                        f: 'json'
                    }
                });
                var metadata = resource.fetchJsonp();
                Cesium.when(metadata, metadataSuccess, metadataFailure);

                // arcgis service styles request
                var _resource = Cesium.Resource.createIfNeeded(that._url + '/resources/styles/root.json');
                _resource.appendForwardSlash();
                var styleResource = _resource.getDerivedResource({queryParameters: {f: 'json'}});

                Cesium.when(styleResource.fetchJsonp(), styleMetadataSuccess, metadataFailure);
            }

            if (this._useTiles) {
                requestMetadata();
            }
        }


        function _findTileInQueue(x, y, level, tileQueue) {
            var head = tileQueue.head;
            while (head !== undefined && !(head.xMvt === x && head.yMvt === y && head.zMvt === level)) {
                head = head.replacementNext;
            }
            return head;
        }

        function buildImageResource(imageryProvider, x, y, level, request) {
            if (!request) {
                return
            }

            // return that._url;
        }


        Cesium.defineProperties(ArcGisMapServerVectorTileImageryProvider.prototype, {
            proxy: {
                get: function () {
                }
            },
            tileWidth: {
                get: function () {
                    return this._tileWidth
                }
            },
            type: {
                get: function () {
                    return this._type;
                }
            },
            tileHeight: {
                get: function () {
                    return this._tileHeight
                }
            },
            maximumLevel: {
                get: function () {
                }
            },
            minimumLevel: {
                get: function () {
                }
            },
            tilingScheme: {
                get: function () {
                    return this._tilingScheme
                }
            },
            rectangle: {
                get: function () {
                    return this._tilingScheme.rectangle
                }
            },
            tileDiscardPolicy: {
                get: function () {
                }
            },
            errorEvent: {
                get: function () {
                    return this._errorEvent
                }
            },
            ready: {
                get: function () {
                    return true
                }
            },
            readyPromise: {
                get: function () {
                    return this._readyPromise
                }
            },
            credit: {
                get: function () {
                }
            },
            hasAlphaChannel: {
                get: function () {
                    return true
                }
            }
        });


        /**
         * 创建要素样式
         * @param geometry
         * @param layerStyle
         * @param resolutions
         * @private
         */
        function _createGeometryStyle(geometry, layerStyle, resolutions) {

            var fill = '',
                lods = that._tileInfo.lods,
                paint = layerStyle.paint,
                layout = layerStyle.layout,
                stroke = new ol.style.Stroke({color: '#ffffff', width: 1});


            var geoLod = lods.find(function (lod) {
                return lod.resolution === resolutions;
            });

            if (layerStyle['source-layer'] === '国普河流_工作:1') {
                debugger;
            }

            // maxZoom
            if (layerStyle.maxzoom > geoLod.level || layerStyle.minzoom < geoLod.level) {

                if (paint.hasOwnProperty('fill-color')) {
                    fill = new ol.style.Fill({color: paint["fill-color"]});
                }

                // if (paint.hasOwnProperty('fill-outline-color')) {
                //     stroke = new ol.style.Stroke({color: paint["fill-outline-color"], width: 1})
                // }

                if (paint.hasOwnProperty('line-color')) {
                    stroke.setColor(paint['line-color']);
                }

                if (paint.hasOwnProperty('line-width')) {
                    stroke.setWidth(paint['line-width']);
                }

                if (layout.hasOwnProperty('line-cap')) {
                    stroke.setLineCap(layout['line-cap'])
                }

                if (layout.hasOwnProperty('line-join')) {
                    stroke.setLineJoin(layout['line-join'])
                }

                if (layout.hasOwnProperty('line-dash')) {
                    stroke.setLineDash(layout['line-dash'])
                }
            }


            switch (geometry) {
                case 'Polygon':
                    return new ol.style.Style({fill: fill, stroke: stroke});
                case 'Line' :
                case 'LineString':
                case 'MultiLineString':
                    return new ol.style.Style({stroke: stroke});
                case 'Point':
                    break;
            }
        }


        function _remove(tileReplacementQueue, item) {
            var previous = item.replacementPrevious;
            var next = item.replacementNext;

            if (item === tileReplacementQueue._lastBeforeStartOfFrame) {
                tileReplacementQueue._lastBeforeStartOfFrame = next;
            }

            if (item === tileReplacementQueue.head) {
                tileReplacementQueue.head = next;
            } else {
                previous.replacementNext = next;
            }

            if (item === tileReplacementQueue.tail) {
                tileReplacementQueue.tail = previous;
            } else {
                next.replacementPrevious = previous;
            }

            item.replacementPrevious = undefined;
            item.replacementNext = undefined;

            --tileReplacementQueue.count;
        }

        function _trimTiles(tileQueue, maximumTiles) {
            var tileToTrim = tileQueue.tail;
            while (tileQueue.count > maximumTiles && Cesium.defined(tileToTrim)) {
                var previous = tileToTrim.replacementPrevious;

                _remove(tileQueue, tileToTrim);
                tileToTrim = null;

                tileToTrim = previous;
            }
        }

        function _getFeatureStyle(feature, resolutions) {
            var layerStyles = that._layerStyles;
            var layer = feature.get('layer');
            var symbol = feature.get('_symbol');
            var geometry = feature.getGeometry().getType();
            // set feature style
            var layerStyle = layerStyles.layers.filter(function (lyr) {
                if (lyr.hasOwnProperty('filter')) {
                    return layer === lyr['source-layer'] && symbol === lyr.filter[2]
                } else {
                    return layer === lyr['source-layer']
                }
            });

            return _createGeometryStyle(geometry, layerStyle[0], resolutions)
        }

        /**
         * 解析vectorTile
         * @param {Number} x The tile X coordinate.
         * @param {Number} y The tile Y coordinate.
         * @param {Number} z The tile level;
         * @param url
         * @private
         */
        function _analyzeMbtile(x, y, z, url) {
            return Cesium.Resource.createIfNeeded(url).fetchArrayBuffer().then(function (u) {
                var canvas = document.createElement("canvas");
                canvas.width = that._tileWidth;
                canvas.height = that._tileHeight;
                var context = canvas.getContext("2d"),
                    features = that._mvtParser.readFeatures(u),
                    replayGroup = new ol.render.canvas.ReplayGroup(0, [0, 0, 4096, 4096], 8, true, 100),
                    squaredTolerance = ol.renderer.vector.getSquaredTolerance(8, window.devicePixelRatio);

                var lods = that._tileInfo.lods.find(function (value) {
                    return value.level === z
                });

                features.forEach(function (feature) {
                    var style = _getFeatureStyle(feature, lods.resolution);
                    if (Cesium.defined(style)) {
                        ol.renderer.vector.renderFeature_(replayGroup, feature, style, squaredTolerance)
                    }
                });

                replayGroup.finish();
                // params: context,center, resolution, rotation, size, pixelRatio,opacity, skippedFeaturesHash
                replayGroup.replay(context, that._pixelRatio, that._transform, 0, {}, that._replays, true);

                if (that._tileQueue.count > that._cacheSize) {
                    _trimTiles(that._tileQueue, that._cacheSize / 2);
                }
                canvas.xMvt = x;
                canvas.yMvt = y;
                canvas.zMvt = z;
                that._tileQueue.markTileRendered(canvas);
                replayGroup = null;
                return canvas;
            }).otherwise(function (error) {

            })
        }

        /**
         * Requests the image for a given tile.  This function should
         * not be called before {@link ArcGisMapServerVectorTileImageryProvider#ready} returns true.
         *
         * @param {Number} x The tile X coordinate.
         * @param {Number} y The tile Y coordinate.
         * @param {Number} level The tile level.
         * @param {Request} [request] The request object. Intended for internal use only.
         * @returns {Promise.<Image|Canvas>|undefined} A promise for the image that will resolve when the image is available, or
         *          undefined if there are too many active requests to the server, and the request
         *          should be retried later.  The resolved image may be either an
         *          Image or a Canvas DOM object.
         *
         * @exception {DeveloperError} <code>requestImage</code> must not be called before the imagery provider is ready.
         */
        ArcGisMapServerVectorTileImageryProvider.prototype.requestImage = function (x, y, level, request) {
            if (this._ready) {
                // throw new Cesium.DeveloperError('requestImage must not be called before the imagery provider is ready.');
                //>>includeEnd('debug');
                var cacheTile = _findTileInQueue(x, y, level, that._tileQueue);

                if (cacheTile) {
                    return cacheTile;
                }
                // 获取url
                return _analyzeMbtile(x, y, level, that._url + '/tile/' + level + '/' + y + '/' + x + '.pbf');
            }
        };

        /**
         * Gets the credits to be displayed when a given tile is displayed.
         *
         * @param {Number} x The tile X coordinate.
         * @param {Number} y The tile Y coordinate.
         * @param {Number} level The tile level;
         * @returns {Credit[]} The credits to be displayed when the tile is displayed.
         *
         * @exception {DeveloperError} <code>getTileCredits</code> must not be called before the imagery provider is ready.
         */
        ArcGisMapServerVectorTileImageryProvider.prototype.getTileCredits = function (x, y, level) {

        };

        /**
         /**
         * Asynchronously determines what features, if any, are located at a given longitude and latitude within
         * a tile.  This function should not be called before {@link ImageryProvider#ready} returns true.
         *
         * @param {Number} x The tile X coordinate.
         * @param {Number} y The tile Y coordinate.
         * @param {Number} level The tile level.
         * @param {Number} longitude The longitude at which to pick features.
         * @param {Number} latitude  The latitude at which to pick features.
         * @return {Promise.<ImageryLayerFeatureInfo[]>|undefined} A promise for the picked features that will resolve when the asynchronous
         *                   picking completes.  The resolved value is an array of {@link ImageryLayerFeatureInfo}
         *                   instances.  The array may be empty if no features are found at the given location.
         *
         * @exception {DeveloperError} <code>pickFeatures</code> must not be called before the imagery provider is ready.
         */
        ArcGisMapServerVectorTileImageryProvider.prototype.pickFeatures = function (x, y, level, longitude, latitude) {

        };

        /**
         * 获取vector tile style
         * @param feature
         * @param resolutions
         */
        ArcGisMapServerVectorTileImageryProvider.prototype.getFeatureStyle = function (feature, resolutions) {


        };

        return ArcGisMapServerVectorTileImageryProvider;
    }
);