define([
    'Cesium'
], function (
    Cesium
) {
    'use strict';
    var that;

    function ArcGisMapServerDynamicImageryProvider(options) {
        options = Cesium.defaultValue(options, {});

        //>>includeStart('debug', pragmas.debug);
        if (!Cesium.defined(options.url)) {
            throw new Cesium.DeveloperError('options.url is required.');
        }
        //>>includeEnd('debug');

        var resource = Cesium.Resource.createIfNeeded(options.url);
        resource.appendForwardSlash();

        if (Cesium.defined(options.token)) {
            resource.setQueryParameters({
                token: options.token
            });
        }


        this._resource = resource;

        this._dpi = Cesium.defaultValue(options.dpi, 96);
        this._format = Cesium.defaultValue(options.format, 'png24');
        this._rectangle = undefined;
        this._singleTile = undefined;

        this._layers = options.layers;
        this._layerDefs = options.layerDefs;
        this._time = options.time;
        this._errorEvent = new Cesium.Event();
        this._minimumLevel = Cesium.defaultValue(options.tileHeight, 0);

        this._ready = false;
        this._readyPromise = Cesium.when.defer();
        this._viewer = options.viewer;
        this._isExport = false;
        // Grab the details of this MapServer.
        that = this;
        var metadataError;

        function metadataSuccess(data) {
            that._wkid = data.spatialReference.wkid;

            if (data.hasOwnProperty('tileInfo')) {
                console.log('config layer type error!')
            } else {
                if (Cesium.defined(data.fullExtent)) {
                    if (Cesium.defined(data.fullExtent.spatialReference) && Cesium.defined(data.fullExtent.spatialReference.wkid)) {
                        if (data.fullExtent.spatialReference.wkid === 102100 ||
                            data.fullExtent.spatialReference.wkid === 102113) {

                            var projection = new Cesium.WebMercatorProjection();
                            var extent = data.fullExtent;
                            var sw = projection.unproject(new Cesium.Cartesian3(Math.max(extent.xmin, -that._tilingScheme.ellipsoid.maximumRadius * Math.PI), Math.max(extent.ymin, -that._tilingScheme.ellipsoid.maximumRadius * Math.PI), 0.0));
                            var ne = projection.unproject(new Cesium.Cartesian3(Math.min(extent.xmax, that._tilingScheme.ellipsoid.maximumRadius * Math.PI), Math.min(extent.ymax, that._tilingScheme.ellipsoid.maximumRadius * Math.PI), 0.0));
                            that._rectangle = new Cesium.Rectangle(sw.longitude, sw.latitude, ne.longitude, ne.latitude);
                        } else if (data.fullExtent.spatialReference.wkid === 4326) {
                            that._rectangle = Cesium.Rectangle.fromDegrees(data.fullExtent.xmin, data.fullExtent.ymin, data.fullExtent.xmax, data.fullExtent.ymax);
                            that._ellipsoid = new Cesium.Ellipsoid(6378137.0, 6378137.0, 6356752.31414035585);
                        } else if (data.fullExtent.spatialReference.wkid === 4490) {
                            that._rectangle = Cesium.Rectangle.fromDegrees(data.fullExtent.xmin, data.fullExtent.ymin, data.fullExtent.xmax, data.fullExtent.ymax);
                            that._ellipsoid = new Cesium.Ellipsoid(6378137.0, 6378137.0, 6356752.3142451793);
                        } else {
                            var extentMessage = 'fullExtent.spatialReference WKID ' + data.fullExtent.spatialReference.wkid + ' is not supported.';
                            metadataError = Cesium.TileProviderError.handleError(metadataError, that, that._errorEvent, extentMessage, undefined, undefined, undefined, requestMetadata);
                            return;
                        }
                    }
                } else {
                    that._rectangle = that._tilingScheme.rectangle;
                }
            }


            if (data.minScale !== 0) {
                that._minimumLevel = data.minScale;
            }

            that.addMoveEvent(that._viewer);
            // init image
            buildImageResource(that);
            that._ready = true;
            that._readyPromise.resolve(true);
            Cesium.TileProviderError.handleSuccess(metadataError);
        }

        function metadataFailure(e) {
            var message = 'An error occurred while accessing ' + that._resource.url + '.';
            metadataError = Cesium.TileProviderError.handleError(metadataError, that, that._errorEvent, message, undefined, undefined, undefined, requestMetadata);
            that._readyPromise.reject(new Cesium.RuntimeError(message));
        }

        function requestMetadata() {
            var resource = that._resource.getDerivedResource({
                queryParameters: {
                    f: 'json'
                }
            });
            var metadata = resource.fetchJsonp();
            Cesium.when(metadata, metadataSuccess, metadataFailure);
        }


        requestMetadata();

    }


    var metadataError;

    function metadataFailure(e) {
        var message = 'An error occurred while accessing ' + that._resource.url + '.';
        metadataError = Cesium.TileProviderError.handleError(metadataError, that, that._errorEvent, message, undefined, undefined, undefined, requestMetadata);
        that._readyPromise.reject(new Cesium.RuntimeError(message));
    }

    /**
     * The spatial reference of the exported image.
     The spatial reference can be specified as either a well-known ID or as a spatial reference json object.
     If the imageSR is not specified, the image will be exported in the spatial reference of the map.
     * @return {string}
     * @private
     */
    function _calculateBbox() {
        var rectangle = that._viewer.camera.computeViewRectangle(that._viewer.scene.globe.ellipsoid);
        var north = Cesium.Math.toDegrees(rectangle.north),
            south = Cesium.Math.toDegrees(rectangle.south),
            east = Cesium.Math.toDegrees(rectangle.east),
            west = Cesium.Math.toDegrees(rectangle.west);

        return [west, north, east, south].join(',');
    }

    function _calculateImageSize() {
        // ensure that we don't ask ArcGIS Server for a taller image than we have actual map displaying within the div

        var container = that._viewer.container;

        var clientWidth = container.clientWidth,
            clientHeigth = container.clientHeight;
        return clientWidth + ',' + clientHeigth;
    }


    function buildImageResource(imageryProvider) {

        var query = {
            bbox: _calculateBbox(),
            size: _calculateImageSize(),
            dpi: imageryProvider._dpi,
            format: imageryProvider._format,
            transparent: true,
            f: 'json'
        };

        // set sr === wkid
        if (imageryProvider.wkid) {
            query.imageSR = query.bboxSR = imageryProvider.wkid;
        }

        if (imageryProvider.layers) {
            query.layers = 'show:' + imageryProvider.layers;
        }

        // set layerDefs
        if (imageryProvider.layerDefs) {
            query.layerDefs = typeof imageryProvider.layerDefs === 'string' ? imageryProvider.layerDefs : JSON.stringify(imageryProvider.layerDefs);
        }

        // add time info
        if (imageryProvider.time) {
            query.time = imageryProvider.time;
        }

        var resource = imageryProvider._resource.getDerivedResource({
            url: 'export',
            queryParameters: query
        });

        var metadata = resource.fetchJsonp();
        Cesium.when(metadata, _metadataSuccess, metadataFailure);

        function _metadataSuccess(data) {
            var singleTile = new Cesium.SingleTileImageryProvider({
                url: data.href,
                rectangle: Cesium.Rectangle.fromDegrees(data.extent.xmin, data.extent.ymin, data.extent.xmax, data.extent.ymax),
                ellipsoid: imageryProvider._ellipsoid
            });

            imageryProvider._singleTile = new Cesium.ImageryLayer(singleTile);
            imageryProvider._viewer.scene.imageryLayers.add(imageryProvider._singleTile);
            imageryProvider._isExport = false;
        }
    }


    function onEventMoveEnd() {
        // 进行范围控制在小于最大范围的时候去请求
        // var rectangle = this._viewer.camera.computeViewRectangle(this._viewer.scene.globe.ellipsoid),
        //     intersectR = Cesium.Rectangle.simpleIntersection(this._rectangle, rectangle);
        // Cesium.Rectangle.equals(intersectR, this._rectangle) ? this._isExport = true : this._isExport = false;


        if (!this._isExport) {
            var layers = this._viewer.scene.imageryLayers;

            if (layers.contains(this._singleTile)) {
                layers.remove(this._singleTile);
            }

            that = this;
            buildImageResource(this);
        }

    }


    Cesium.defineProperties(ArcGisMapServerDynamicImageryProvider.prototype, {
        /**
         * Gets the URL of the ArcGIS MapServer.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         * @type {String}
         * @readonly
         */
        url: {
            get: function () {
                return this._resource._url;
            }
        },

        /**
         * Gets the ArcGIS token used to authenticate with the ArcGis MapServer service.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         * @type {String}
         * @readonly
         */
        token: {
            get: function () {
                return this._resource.queryParameters.token;
            }
        },

        /**
         * Gets the proxy used by this provider.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         * @type {Proxy}
         * @readonly
         */
        proxy: {
            get: function () {
                return this._resource.proxy;
            }
        },

        hasImagery: {
            get: function () {
                //>>includeStart('debug', pragmas.debug);
                if (!this._ready) {
                    throw new Cesium.DeveloperError('maximumLevel must not be called before the imagery provider is ready.');
                }
                //>>includeEnd('debug');

                return this._hasImagery;
            }
        },

        /**
         * Gets the maximum level-of-detail that can be requested.  This function should
         * not be called before {@link ArcGisMapServerDynamicImageryProvider#ready} returns true.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         * @type {Number}
         * @readonly
         */
        maximumLevel: {
            get: function () {
                //>>includeStart('debug', pragmas.debug);
                if (!this._ready) {
                    throw new Cesium.DeveloperError('maximumLevel must not be called before the imagery provider is ready.');
                }
                //>>includeEnd('debug');

                return this._maximumLevel;
            }
        },

        /**
         * Gets the minimum level-of-detail that can be requested.  This function should
         * not be called before {@link ArcGisMapServerDynamicImageryProvider#ready} returns true.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         * @type {Number}
         * @readonly
         */
        minimumLevel: {
            get: function () {
                //>>includeStart('debug', pragmas.debug);
                if (!this._ready) {
                    throw new Cesium.DeveloperError('minimumLevel must not be called before the imagery provider is ready.');
                }
                //>>includeEnd('debug');

                return 0;
            }
        },

        /**
         * Gets the tiling scheme used by this provider.  This function should
         * not be called before {@link ArcGisMapServerDynamicImageryProvider#ready} returns true.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         * @type {TilingScheme}
         * @readonly
         */
        tilingScheme: {
            get: function () {
                //>>includeStart('debug', pragmas.debug);
                if (!this._ready) {
                    throw new Cesium.DeveloperError('tilingScheme must not be called before the imagery provider is ready.');
                }
                //>>includeEnd('debug');

                return this._tilingScheme;
            }
        },

        /**
         * Gets the rectangle, in radians, of the imagery provided by this instance.  This function should
         * not be called before {@link ArcGisMapServerDynamicImageryProvider#ready} returns true.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         * @type {Rectangle}
         * @readonly
         */
        rectangle: {
            get: function () {
                //>>includeStart('debug', pragmas.debug);
                if (!this._ready) {
                    throw new Cesium.DeveloperError('rectangle must not be called before the imagery provider is ready.');
                }
                //>>includeEnd('debug');

                return this._rectangle;
            }
        },

        /**
         * Gets the tile discard policy.  If not undefined, the discard policy is responsible
         * for filtering out "missing" tiles via its shouldDiscardImage function.  If this function
         * returns undefined, no tiles are filtered.  This function should
         * not be called before {@link ArcGisMapServerDynamicImageryProvider#ready} returns true.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         * @type {TileDiscardPolicy}
         * @readonly
         */
        tileDiscardPolicy: {
            get: function () {
                //>>includeStart('debug', pragmas.debug);
                if (!this._ready) {
                    throw new Cesium.DeveloperError('tileDiscardPolicy must not be called before the imagery provider is ready.');
                }
                //>>includeEnd('debug');

                return this._tileDiscardPolicy;
            }
        },

        /**
         * Gets an event that is raised when the imagery provider encounters an asynchronous error.  By subscribing
         * to the event, you will be notified of the error and can potentially recover from it.  Event listeners
         * are passed an instance of {@link TileProviderError}.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         * @type {Event}
         * @readonly
         */
        errorEvent: {
            get: function () {
                return this._errorEvent;
            }
        },

        /**
         * Gets a value indicating whether or not the provider is ready for use.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         * @type {Boolean}
         * @readonly
         */
        ready: {
            get: function () {
                return this._ready;
            }
        },

        /**
         * Gets a promise that resolves to true when the provider is ready for use.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         * @type {Promise.<Boolean>}
         * @readonly
         */
        readyPromise: {
            get: function () {
                return this._readyPromise.promise;
            }
        },

        /**
         * Gets the credit to display when this imagery provider is active.  Typically this is used to credit
         * the source of the imagery.  This function should not be called before {@link ArcGisMapServerDynamicImageryProvider#ready} returns true.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         * @type {Credit}
         * @readonly
         */
        credit: {
            get: function () {
                return this._credit;
            }
        },

        /**
         * the container viewer
         * @memberof
         * @type {viewer}
         * @readonly
         */
        viewer: {
            get: function () {
                return this._viewer;
            }
        },

        /**
         * Gets a value indicating whether this imagery provider is using pre-cached tiles from the
         * ArcGIS MapServer.  If the imagery provider is not yet ready ({@link ArcGisMapServerDynamicImageryProvider#ready}), this function
         * will return the value of `options.usePreCachedTilesIfAvailable`, even if the MapServer does
         * not have pre-cached tiles.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         *
         * @type {Boolean}
         * @readonly
         * @default true
         */
        usingPrecachedTiles: {
            get: function () {
                return this._useTiles;
            }
        },

        /**
         * Gets a value indicating whether or not the images provided by this imagery provider
         * include an alpha channel.  If this property is false, an alpha channel, if present, will
         * be ignored.  If this property is true, any images without an alpha channel will be treated
         * as if their alpha is 1.0 everywhere.  When this property is false, memory usage
         * and texture upload time are reduced.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         *
         * @type {Boolean}
         * @readonly
         * @default true
         */
        hasAlphaChannel: {
            get: function () {
                return true;
            }
        },

        /**
         * Gets the comma-separated list of layer IDs to show.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         *
         * @type {String}
         */
        layers: {
            get: function () {
                return this._layers;
            }
        },

        /**
         * Gets the comma-separated list of layer IDs and Field to show.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         *
         * @type {Object}
         */
        layerDefs: {
            get: function () {
                return this._layerDefs;
            }
        },

        /**
         * Gets the comma-separated list of time and Field to show.
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         *
         * @type {Object}
         */

        time: {
            get: function () {
                return this._time;
            }
        },

        /**
         * to zoom map
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         *
         * @type {number}
         */
        zoomOffset: {
            get: function () {
                return this._zoomOffset;
            }
        },

        /**
         * The device resolution of the exported image (dots per inch). If the dpi is not specified, an image with a default DPI of 96 will be exported
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         *
         * @type {number}
         */
        dpi: {
            get: function () {
                return this._dpi;
            }
        },

        /**
         * Description: The format of the exported image. The default format is .png.
         * Values: png | png8 | png24 | jpg | pdf | bmp | gif | svg | svgz | emf | ps | png32
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         *
         * @type {number}
         */
        format: {
            get: function () {
                return this._format;
            }
        },
        /**
         * Description: is export map arcgis
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         *
         * @type {number}
         */
        isExport: {
            get: function () {
                return this._isExport;
            }
        },

        /**
         ArcGisMapServerDynamicImageryProvider
         * @memberof ArcGisMapServerDynamicImageryProvider.prototype
         *
         * @type {number}
         */
        singleTile: {
            get: function () {
                return this._singleTile;
            }
        }


    });


    /**
     * Gets the credits to be displayed when a given tile is displayed.
     *
     * @param {Number} x The tile X coordinate.
     * @param {Number} y The tile Y coordinate.
     * @param {Number} level The tile level;
     * @returns {Credit[]} The credits to be displayed when the tile is displayed.
     *
     * @exception {DeveloperError} <code>getTileCredits</code> must not be called before the imagery provider is ready.
     */
    ArcGisMapServerDynamicImageryProvider.prototype.getTileCredits = function (x, y, level) {
        return undefined;
    };
    /**
     * arcgis Dynamic layer remove  moveend event
     */
    ArcGisMapServerDynamicImageryProvider.prototype.removeMoveEvent = function () {
        this._viewer.camera.moveEnd.removeEventListener(onEventMoveEnd, this);
    };

    /**
     * arcgis Dynamic layer add moveend event
     */
    ArcGisMapServerDynamicImageryProvider.prototype.addMoveEvent = function () {
        this._viewer.camera.moveEnd.addEventListener(onEventMoveEnd, this);
    };


    /**
     * arcgis Dynamic layer add moveend event
     */
    ArcGisMapServerDynamicImageryProvider.prototype.removeLayer = function (event) {
        this.removeMoveEvent();
        var layers = this._viewer.scene.imageryLayers;
        if (this._singleTile && layers.contains(this._singleTile)) {
            layers.remove(this._singleTile);
        }
    };

    return ArcGisMapServerDynamicImageryProvider;
});
