package cn.gtmap.estateplat.log.resources;

import cn.gtmap.estateplat.core.ex.AppException;
import cn.gtmap.estateplat.model.resources.core.ZrzyXtRz;
import cn.gtmap.estateplat.utils.ClientInfoUtil;
import com.alibaba.fastjson.JSON;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.gtis.common.util.UUIDGenerator;
import com.gtis.generic.util.SessionUtils;
import com.gtis.plat.service.SysUserService;
import com.gtis.plat.vo.PfUserVo;
import com.gtis.web.SessionUtil;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.After;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.core.LocalVariableTableParameterNameDiscoverer;
import org.springframework.data.domain.Pageable;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.StandardEvaluationContext;
import org.springframework.ui.Model;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.servlet.mvc.support.RedirectAttributes;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.lang.reflect.Method;
import java.util.Calendar;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * 自然资源日志Aspect
 *
 * @author <a href="mailto:lijian@gtmap.cn">lijian</a>
 * @version 1.0, 2017/4/27
 */
@Aspect
public class ZrzyLogAspect {
    private ZrzyLogService zrzyLogService;
    private SysUserService sysUserService;

    private ExecutorService executor = Executors.newSingleThreadExecutor();
    private final static String ANONYMOUS_USER_ID = "-999";
    private final static String ANONYMOUS_USER_NAME = "匿名用户";
    //登录用户实际ip
    public static final String LOGIN_USER_IP = "login_user_ip";
    public static final String LOGIN_USER_MAC = "login_user_mac";
    public static final String LOGIN_USER_COMPUTERNAME = "login_user_computername";

    public void setZrzyLogService(ZrzyLogService zrzyLogService) {
        this.zrzyLogService = zrzyLogService;
    }

    public void setSysUserService(SysUserService sysUserService) {
        this.sysUserService = sysUserService;
    }

    @Pointcut("@annotation(cn.gtmap.estateplat.log.resources.ZrzyLog)")
    public void zrzyLogServiceAspect() {
    }

    @Before("zrzyLogServiceAspect()")
    public void doBefore(JoinPoint joinPoint) {
        Method method = getMethod(joinPoint);
        ZrzyLog auditMethodLog = method.getAnnotation(ZrzyLog.class);
        String userId = StringUtils.isNotBlank(auditMethodLog.userId()) ? parseUserId(auditMethodLog.userId(), method, joinPoint.getArgs()) : null;
        zrzyLog(auditMethodLog.name(), getAuditContent(joinPoint, auditMethodLog), userId);
    }

    @After("zrzyLogServiceAspect()")
    public void doAfter(JoinPoint joinPoint) {

    }

    private void zrzyLog(String name, String content, String userId) {
        final ZrzyXtRz zrzyXtRz = new ZrzyXtRz();
        zrzyXtRz.setRzid(UUIDGenerator.generate());
        zrzyXtRz.setCzrq(Calendar.getInstance().getTime());
        if (StringUtils.isNotBlank(SessionUtil.getCurrentUserId())) {
            zrzyXtRz.setYhid(SessionUtil.getCurrentUserId());
            zrzyXtRz.setYhmc(SessionUtil.getCurrentUser().getUsername());
        } else {
            if (StringUtils.isNotBlank(userId)) {
                zrzyXtRz.setYhid(userId);
                if (sysUserService != null) {
                    PfUserVo pfUserVo = sysUserService.getUserVo(userId);
                    if (pfUserVo != null)
                        zrzyXtRz.setYhmc(pfUserVo.getUserName());
                }
            } else {
                zrzyXtRz.setYhid(ANONYMOUS_USER_ID);
                zrzyXtRz.setYhmc(ANONYMOUS_USER_NAME);
            }
        }
        zrzyXtRz.setDyczff(name);
        zrzyXtRz.setFfcs(content);
        try {
            /**
             * 1.获取request信息
             * 2.根据request获取session
             * 3.从session中取出登录客户端信息
             */
            RequestAttributes ra = RequestContextHolder.getRequestAttributes();
            ServletRequestAttributes sra = (ServletRequestAttributes)ra;
            HttpServletRequest request = sra.getRequest();
            String ip = getRemoteAddress(request);
            if (StringUtils.isNotBlank(ip)) {
                zrzyXtRz.setIp(ip);
                zrzyXtRz.setJqm(ClientInfoUtil.getMACAddress(ip));
                zrzyXtRz.setJsjm(ClientInfoUtil.getComputerName(ip));
            }
        } catch (Exception ignored) {
        }
        executor.submit(new Runnable() {
            @Override
            public void run() {
                zrzyLogService.saveZrzyXtRz(zrzyXtRz);
            }
        });
    }

    private ZrzyLog getMethodLog(JoinPoint joinPoint) {
        try {
            String targetName = joinPoint.getTarget().getClass().getName();
            String methodName = joinPoint.getSignature().getName();
            Object[] arguments = joinPoint.getArgs();
            Class targetClass = Class.forName(targetName);
            Method[] methods = targetClass.getMethods();
            for (Method method : methods) {
                if (method.getName().equals(methodName)) {
                    Class[] clazzs = method.getParameterTypes();
                    if (clazzs.length == arguments.length) {
                        return method.getAnnotation(ZrzyLog.class);
                    }
                }
            }
        } catch (ClassNotFoundException e) {
            throw new AppException(e.getMessage());
        }

        return null;
    }

    private String getAuditContent(JoinPoint joinPoint, ZrzyLog zrzyLog) {
        Map contentMap = Maps.newHashMap();
        String description = null;
        if (StringUtils.isNotBlank(zrzyLog.description())) {
            description = zrzyLog.description();
        } else
            description = "操作内容";
        contentMap.put(description, filterArguments(joinPoint.getArgs()));
        return JSON.toJSONString(contentMap);
    }

    /**
     * 过滤掉一些无关的参数
     *
     * @param args
     * @return
     */
    private Object filterArguments(Object[] args) {
        List<Object> arguments = Lists.newArrayList();
        for (Object arg : args) {
            if (arg instanceof Model || arg instanceof Pageable || arg instanceof RedirectAttributes || arg instanceof HttpServletResponse
                    || arg instanceof HttpServletRequest)
                continue;
            arguments.add(arg);
        }
        return arguments;
    }

    /**
     * @param
     * @return
     * @author <a href="mailto:shenjian@gtmap.cn">shenjian</a>
     * @version 1.0, 2016/7/29
     * @description 根据SPEL表达式获取注解中的参数值
     */
    private String parseUserId(String userId, Method method, Object[] args) {
        LocalVariableTableParameterNameDiscoverer u =
                new LocalVariableTableParameterNameDiscoverer();
        String[] paraNameArr = u.getParameterNames(method);

        //使用SPEL进行key的解析
        ExpressionParser parser = new SpelExpressionParser();
        //SPEL上下文
        StandardEvaluationContext context = new StandardEvaluationContext();
        //把方法参数放入SPEL上下文中
        for (int i = 0; i < paraNameArr.length; i++) {
            context.setVariable(paraNameArr[i], args[i]);
        }
        return parser.parseExpression(userId).getValue(context, String.class);
    }

    /**
     * @param joinPoint 注解点
     * @return
     * @author <a href="mailto:shenjian@gtmap.cn">shenjian</a>
     * @version 1.0, 2016/7/29
     * @description 返回注解的方法对象
     */
    private Method getMethod(JoinPoint joinPoint) {
        Object[] args = joinPoint.getArgs();
        Class[] argTypes = new Class[joinPoint.getArgs().length];
        for (int i = 0; i < args.length; i++) {
            argTypes[i] = args[i].getClass();
        }
        Method method = null;
        try {
            method = joinPoint.getTarget().getClass().getMethod(joinPoint.getSignature().getName(), argTypes);
        } catch (NoSuchMethodException e) {
            e.printStackTrace();
        } catch (SecurityException e) {
            e.printStackTrace();
        }
        return method;
    }

    private String getRemoteAddress(HttpServletRequest request) {
        String ip = request.getHeader("x-forwarded-for");
        if (ip == null || ip.length() == 0 || ip.equalsIgnoreCase("unknown")) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || ip.equalsIgnoreCase("unknown")) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || ip.equalsIgnoreCase("unknown")) {
            ip = request.getRemoteAddr();
        }
        return ip;
    }
}
