package com.gtis.portal.web.interceptor;

import com.gtis.config.AppConfig;
import com.gtis.plat.service.SysWorkFlowInstanceService;
import com.gtis.plat.vo.PfWorkFlowInstanceVo;
import com.gtis.plat.wf.WorkFlowInfo;
import com.gtis.portal.model.WorkflowEvent;
import com.gtis.portal.service.LogService;
import com.gtis.portal.service.TaskGeneralService;
import com.gtis.portal.service.server.PfWorkFlowEventConfigurationService;
import org.apache.commons.lang.StringUtils;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;

/**
 * .
 * <p/>任务所有服务接口拦截器
 *
 * @author <a href="mailto:shenjian@gtmap.cn">shenjian</a>
 * @version 1.0, 2014/6/23
 */

@Aspect
@Component
public class TaskServiceInterceptor {
    private final Logger log = LoggerFactory.getLogger(TaskServiceInterceptor.class);
    @Autowired
    LogService logService;
    @Autowired
    TaskGeneralService taskGeneralService;
    @Autowired
    PfWorkFlowEventConfigurationService pfWorkFlowEventConfigurationService;
    @Autowired
    SysWorkFlowInstanceService workFlowIntanceService;

    /**
     * 删除任务执行前后执行
     *
     * @param wiid
     * @param taskid
     * @param reason
     */
    @Before("execution (* com.gtis.portal.service.TaskActionService.delTask(..)) && args(wiid,taskid,userid,reason,proid,pfWorkFlowInstanceVo,request)")
    public void beforeDelTask(String wiid, String taskid, String userid, String reason, String proid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request) throws Exception {
        logService.saveDelTaskLog(wiid, taskid, userid, reason, pfWorkFlowInstanceVo, request);
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_BeforeDel.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), proid, "");
    }

    /**
     * 删除任务执行成功后执行
     *
     * @param wiid
     * @param taskid
     * @param reason
     */
    @AfterReturning("execution (* com.gtis.portal.service.TaskActionService.delTask(..)) && args(wiid,taskid,userid,reason,proid,pfWorkFlowInstanceVo,request)")
    public void afterDelTask(String wiid, String taskid, String userid, String reason, String proid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request) throws Exception {
        //删除保存日志
//        logService.saveDelTaskLog(wiid, taskid, userid, reason, pfWorkFlowInstanceVo, request);
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_Del.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), proid, "");
    }

    /**
     * 已办任务取回任务执行前执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @Before("execution (* com.gtis.portal.service.TaskActionService.retrieveTask(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request)")
    public void beforeRetrieveTask(String wiid, String taskid, String userid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_BeforeRetrieve.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", "");
    }

    /**
     * 已办任务取回任务执行成功后执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @AfterReturning("execution (* com.gtis.portal.service.TaskActionService.retrieveTask(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request)")
    public void afterRetrieveTask(String wiid, String taskid, String userid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_Retrieve.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", "");
        //取回保存日志
        logService.saveRetrieveTaskLog(wiid, taskid, userid, pfWorkFlowInstanceVo, request);
    }

    /**
     * 任务转发前执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @Before("execution (* com.gtis.portal.service.TaskActionService.turnTaskByXml(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request,turnXml) ||" +
            "execution (* com.gtis.portal.service.TaskActionService.turnTaskByWorkFlowInfo(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request,turnXml) ||" +
            "execution (* com.gtis.portal.service.TaskActionService.autoTurnTask(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request,turnXml)")
    public void beforeTurnTask(String wiid, String taskid, String userid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request, String turnXml) throws Exception {

        //转发任务前执行返回异常
        taskGeneralService.turnBeforeTaskGeneralWork(wiid, taskid, userid);
        //获取工作流事件执行方法
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_BeforeTurn.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", turnXml);
    }

    /**
     * 任务转发后执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @AfterReturning("execution (* com.gtis.portal.service.TaskActionService.turnTaskByXml(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request,turnXml) ||" +
            "execution (* com.gtis.portal.service.TaskActionService.turnTaskByWorkFlowInfo(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request,turnXml) ||" +
            "execution (* com.gtis.portal.service.TaskActionService.autoTurnTask(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request,turnXml)")
    public void afterTurnTask(String wiid, String taskid, String userid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request, String turnXml) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_Turn.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", turnXml);
        //转发任务后执行通用方法
        taskGeneralService.turnAfterTaskGeneralWork(wiid, taskid, userid);
        //转发保存日志
        logService.saveTurnTaskLog(wiid, taskid, userid, pfWorkFlowInstanceVo, request);
    }


    /**
     * 任务退回前执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @Before("execution (* com.gtis.portal.service.TaskActionService.turnBackTask(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request,adids,remark)")
    public void beforeTurnBackTask(String wiid, String taskid, String userid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request, String adids, String remark) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_BeforeBack.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", "");
    }

    /**
     * 任务退回后执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @AfterReturning("execution (* com.gtis.portal.service.TaskActionService.turnBackTask(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request,adids,remark)")
    public void afterTurnBackTask(String wiid, String taskid, String userid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request, String adids, String remark) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_Back.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", "");
        //退回保存优先级为4,用于待办任务退回始终按照优先级排序
        String savePriorityEnable = AppConfig.getProperty("turnback.task.save.priority.enable");
        if (StringUtils.equals(savePriorityEnable, "true")) {
            workFlowIntanceService.updateWorkFlowIntancePriority(wiid, "4");
        }
        //退回保存日志
        logService.saveTurnBackTaskLog(wiid, taskid, userid, remark, pfWorkFlowInstanceVo, request);
    }

    /**
     * 任务办结前执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @Before("execution (* com.gtis.portal.service.TaskActionService.endTask(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request,turnXml,info)")
    public void beforeEndTask(String wiid, String taskid, String userid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request, String turnXml, WorkFlowInfo info) throws Exception {
        //转发任务前执行返回异常
        taskGeneralService.turnBeforeTaskGeneralWork(wiid, taskid, userid);
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_BeforeTurn.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", turnXml);
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_BeforeEnd.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", turnXml);
    }

    /**
     * 任务办结后执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @AfterReturning("execution (* com.gtis.portal.service.TaskActionService.endTask(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request,turnXml,info)")
    public void afterEndTask(String wiid, String taskid, String userid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request, String turnXml, WorkFlowInfo info) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_End.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", turnXml);
        //办结保存日志
        logService.saveEndTaskLog(wiid, taskid, userid, pfWorkFlowInstanceVo, request);
    }

    /**
     * 任务终止前执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @Before("execution (* com.gtis.portal.service.TaskActionService.stopTask(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request)")
    public void beforeStopTask(String wiid, String taskid, String userid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_BeforeStop.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", "");
    }

    /**
     * 任务终止后执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @AfterReturning("execution (* com.gtis.portal.service.TaskActionService.stopTask(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request)")
    public void afterStopTask(String wiid, String taskid, String userid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_Stop.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", "");
        //终止保存日志
        logService.saveStopTaskLog(wiid, taskid, userid, pfWorkFlowInstanceVo, request);
    }

    /**
     * 任务挂起前执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @Before("execution (* com.gtis.portal.service.TaskActionService.postTask(..)) && args(wiid,taskid,userid,remark,pfWorkFlowInstanceVo,request)")
    public void beforePostTask(String wiid, String taskid, String userid, String remark, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_BeforePost.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", "");
    }

    /**
     * 任务挂起后执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @AfterReturning("execution (* com.gtis.portal.service.TaskActionService.postTask(..)) && args(wiid,taskid,userid,remark,pfWorkFlowInstanceVo,request)")
    public void afterPostTask(String wiid, String taskid, String userid, String remark, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_Post.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", "");
        //挂起保存日志
        logService.savePostTaskLog(wiid, taskid, userid, remark, pfWorkFlowInstanceVo, request);
    }

    /**
     * 任务解挂前执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @Before("execution (* com.gtis.portal.service.TaskActionService.upPostTask(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request)")
    public void beforeUnPostTask(String wiid, String taskid, String userid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_BeforeUnPost.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", "");
    }

    /**
     * 任务解挂后执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @AfterReturning("execution (* com.gtis.portal.service.TaskActionService.upPostTask(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request)")
    public void afterUnPostTask(String wiid, String taskid, String userid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_UnPost.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", "");
        //办结保存日志
        logService.saveUpPostTaskLog(wiid, taskid, userid, pfWorkFlowInstanceVo, request);
    }

    /**
     * 任务创建前执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @Before("execution (* com.gtis.portal.service.TaskActionService.createTask(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request)")
    public void beforeCreateTask(String wiid, String taskid, String userid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_BeforeBegin.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", "");
    }

    /**
     * 任务创建后执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @AfterReturning("execution (* com.gtis.portal.service.TaskActionService.createTask(..)) && args(wiid,taskid,userid,pfWorkFlowInstanceVo,request)")
    public void afterCreateTask(String wiid, String taskid, String userid, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_Begin.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", "");
        //办结保存日志
        logService.saveCreateTaskLog(wiid, taskid, userid, pfWorkFlowInstanceVo, request);
    }

    /**
     * 任务调整优先前执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @Before("execution (* com.gtis.portal.service.TaskActionService.priorityTask(..)) && args(wiid,taskid,userid,priority,pfWorkFlowInstanceVo,request)")
    public void beforePriorityTask(String wiid, String taskid, String userid, String priority, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_BeforePriority.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", "");
    }

    /**
     * 任务调整优先后执行
     *
     * @param wiid
     * @param taskid
     * @param userid
     * @param priority
     * @param pfWorkFlowInstanceVo
     * @param request
     */
    @AfterReturning("execution (* com.gtis.portal.service.TaskActionService.priorityTask(..)) && args(wiid,taskid,userid,priority,pfWorkFlowInstanceVo,request)")
    public void afterPriorityTask(String wiid, String taskid, String userid, String priority, PfWorkFlowInstanceVo pfWorkFlowInstanceVo, HttpServletRequest request) throws Exception {
        pfWorkFlowEventConfigurationService.doWfEvent(wiid, taskid, userid, WorkflowEvent.WorkFlow_Priority.name(), pfWorkFlowInstanceVo.getWorkflowDefinitionId(), "", "");
        //办结保存日志
        logService.savePriorityTaskLog(wiid, taskid, userid, pfWorkFlowInstanceVo, request);
    }

}
