package com.gtis.plat.service.impl;

import com.google.common.collect.Sets;
import com.gtis.common.http.Utf8PostMethod;
import com.gtis.config.AppConfig;
import com.gtis.plat.service.*;
import com.gtis.plat.vo.PfOrganVo;
import com.gtis.plat.vo.PfRoleVo;
import com.gtis.plat.vo.PfUserVo;
import com.gtis.plat.wf.WorkFlowInfo;
import com.gtis.plat.wf.WorkFlowTransInfo;
import com.gtis.plat.wf.WorkFlowXml;
import com.gtis.plat.wf.WorkFlowXmlUtil;
import com.gtis.plat.wf.model.ActivityModel;
import com.gtis.plat.wf.model.PerformerTaskModel;
import com.gtis.web.SessionUtil;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.ServletActionContext;
import org.dom4j.*;
import org.dom4j.tree.DefaultText;

import java.io.IOException;
import java.util.*;

/**
 * 任务转发相关的服务
 * @author <a href="mailto:shenjian@gtmap.cn">shenjian</a>
 * @version 1.0, 2015/7/13
 */
public class SysTaskTurnServiceImpl implements SysTaskTurnService {

    private static final Log log = LogFactory.getLog(SysTaskTurnServiceImpl.class);
    /**子流程服务*/
    private SysSubProcessService sysSubProcessService;

    /** 工作流核心服务 */
    private WorkFlowCoreService workFlowService;

    /**活动服务*/
    private SysTaskService sysTaskService;

    private SysWorkFlowInstanceService workFlowIntanceService;

    private SysUserService sysUserService;


    /**
     * 获取用户待办任务的默认转发信息，XML格式
     * @param userId 用户Id
     * @param taskId 待办任务Id
     * @return
     */
    @Override
    public String getTaskDefaultTurnInfo(String userId, String taskId) throws Exception {
        WorkFlowInfo info = null;
        if(sysSubProcessService.isSubProcessFinishing(taskId)){
            info = workFlowService.getWorkFlowTurnInfo("-1", sysTaskService.getActivity(sysTaskService.getTask(taskId).getActivityId()).getWorkflowInstanceId());
        }else{
            info = workFlowService.getWorkFlowTurnInfo(userId,taskId);
        }

        return formatTurnInfo(info);
    }

    /**
     * 获取用户待办任务的默认转发活动
     * @param userId 用户Id
     * @param taskId 待办任务Id
     * @return
     * @throws Exception
     */
    @Override
    public Set<String> getTaskDefaultTurnActivity(String userId, String taskId) throws Exception{
        String turnInfo = getTaskDefaultTurnInfo(userId,taskId);
        Document fileDoc = DocumentHelper.parseText(turnInfo);
        Element rootElement = fileDoc.getRootElement();
        List<Element> activityElementList = rootElement.selectNodes("Activity");
        Set<String> activityIds = Sets.newHashSet();
        for(Element activityElement:activityElementList){
            activityIds.add(activityElement.attributeValue("Id"));
        }
        return activityIds;
    }

    /**
     * 获取用户待办任务的下一个活动默认参与人员Ids
     *
     * @param userId               用户Id
     * @param taskId               待办任务Id
     * @param activityDefinitionId 下一个活动定义Id
     * @return
     * @throws Exception
     */
    @Override
    public Set<String> getTaskDefaultTurnUser(String userId, String taskId, String activityDefinitionId) throws Exception {
        String turnInfo = getTaskDefaultTurnInfo(userId,taskId);
        Document fileDoc = DocumentHelper.parseText(turnInfo);
        Element rootElement = fileDoc.getRootElement();
        Element activityElement = (Element)rootElement.selectSingleNode("Activity[@Id='"+activityDefinitionId+"']");
        List<Element> userInfoElementList = activityElement.selectNodes("//User/UserInfo");
        Set<String> userIds = Sets.newHashSet();
        for(Element userInfoElement:userInfoElementList){
            userIds.add(userInfoElement.attributeValue("Id"));
        }
        return userIds;
    }

    /**
     * 获取用户待办任务的下一个活动默认参与部门或角色Ids
     *
     * @param userId               用户Id
     * @param taskId               待办任务Id
     * @param activityDefinitionId 下一个活动定义Id
     * @return
     * @throws Exception
     */
    @Override
    public Set<String> getTaskDefaultTurnOrganOrRole(String userId, String taskId, String activityDefinitionId) throws Exception {
        String turnInfo = getTaskDefaultTurnInfo(userId,taskId);
        Document fileDoc = DocumentHelper.parseText(turnInfo);
        Element rootElement = fileDoc.getRootElement();
        Element activityElement = (Element)rootElement.selectSingleNode("Activity[@Id='"+activityDefinitionId+"']");
        List<Element> userElementList = activityElement.selectNodes("User");
        Set<String> organOrRoleIds = Sets.newHashSet();
        for(Element userElement:userElementList){
            organOrRoleIds.add(userElement.attributeValue("Id"));
        }
        return organOrRoleIds;
    }

    /**
     * 向用户待办任务的下一个活动的参与者中，增加人员
     * @param userId 用户Id
     * @param taskId 待办任务Id
     * @param activityDefinitionId 下一个活动定义Id
     * @param users 新用户Ids
     */
    public void addUserToTurnInfo(String userId, String taskId, String activityDefinitionId, Collection<String> users) throws Exception {
        String wiid = sysTaskService.getActivity(sysTaskService.getTask(taskId).getActivityId()).getWorkflowInstanceId();
        String taskTurnInfo = getTurnInfoFromGlobalVal(taskId,wiid);
        if(StringUtils.isBlank(taskTurnInfo)){
            taskTurnInfo = getTaskDefaultTurnInfo(userId, taskId);
        }
        Document fileDoc = DocumentHelper.parseText(taskTurnInfo);
        Element rootElement = fileDoc.getRootElement();
        Element activityElement = (Element)rootElement.selectSingleNode("Activity[@Id='"+activityDefinitionId+"']");
        if(activityElement!=null){
            for(String addUserId:users){
                PfUserVo pfUserVo = sysUserService.getUserVo(addUserId);
                List<PfOrganVo> pfOrganVo = sysUserService.getOrganListByUser(addUserId);
                Element userElement = DocumentHelper.createElement("User");
                userElement.addAttribute("Name",pfOrganVo.get(0).getOrganName());
                userElement.addAttribute("Id","0");
                Element userInfoElement = userElement.addElement("UserInfo");
                userInfoElement.addAttribute("Name",pfUserVo.getUserName());
                userInfoElement.addAttribute("Id",addUserId);
                activityElement.add(userElement);
            }
        }
        taskTurnInfo = fileDoc.getRootElement().asXML();
        updateTurnInfoToGlobalVal(taskId,wiid,taskTurnInfo);
    }

    /**
     * 向用户待办任务的下一个活动的参与者中，增加角色
     * @param userId 用户Id
     * @param taskId 待办任务Id
     * @param activityDefinitionId 下一个活动定义Id
     * @param roles 新角色Ids
     */
    public void addRolesToTurnInfo(String userId, String taskId, String activityDefinitionId, Collection<String> roles) throws Exception {
        String wiid = sysTaskService.getActivity(sysTaskService.getTask(taskId).getActivityId()).getWorkflowInstanceId();
        String taskTurnInfo = getTurnInfoFromGlobalVal(taskId,wiid);
        if(StringUtils.isBlank(taskTurnInfo)){
            taskTurnInfo = getTaskDefaultTurnInfo(userId, taskId);
        }
        Document fileDoc = DocumentHelper.parseText(taskTurnInfo);
        Element rootElement = fileDoc.getRootElement();
        Element activityElement = (Element)rootElement.selectSingleNode("Activity[@Id='"+activityDefinitionId+"']");
        if(activityElement!=null){
            for(String roleId:roles){
                PfRoleVo pfRoleVo = sysUserService.getRoleVo(roleId);
                Element userElement = DocumentHelper.createElement("User");
                userElement.addAttribute("Name",pfRoleVo.getRoleName());
                userElement.addAttribute("Id",roleId);
                List<PfUserVo> userVoList=  sysUserService.getUserListByRole(roleId);
                for(PfUserVo pfUserVo:userVoList) {
                    Element userInfoElement = userElement.addElement("UserInfo");
                    userInfoElement.addAttribute("Name", pfUserVo.getUserName());
                    userInfoElement.addAttribute("Id", pfUserVo.getUserId());

                }
                activityElement.add(userElement);
            }
        }
        taskTurnInfo = fileDoc.getRootElement().asXML();
        updateTurnInfoToGlobalVal(taskId,wiid,taskTurnInfo);
    }

    /**
     * 向用户待办任务的下一个活动的参与者中，增加部门
     * @param userId 用户Id
     * @param taskId 待办任务Id
     * @param activityDefinitionId 下一个活动定义Id
     * @param organs 新部门Ids
     */
    public void addOrganToTurnInfo(String userId, String taskId, String activityDefinitionId, Collection<String> organs) throws Exception {
        String wiid = sysTaskService.getActivity(sysTaskService.getTask(taskId).getActivityId()).getWorkflowInstanceId();
        String taskTurnInfo = getTurnInfoFromGlobalVal(taskId,wiid);
        if(StringUtils.isBlank(taskTurnInfo)){
            taskTurnInfo = getTaskDefaultTurnInfo(userId, taskId);
        }
        Document fileDoc = DocumentHelper.parseText(taskTurnInfo);
        Element rootElement = fileDoc.getRootElement();
        Element activityElement = (Element)rootElement.selectSingleNode("Activity[@Id='"+activityDefinitionId+"']");
        if(activityElement!=null){
            for(String organId:organs){
                PfOrganVo pfOrganVo = sysUserService.getOrganVo(organId);
                Element userElement = DocumentHelper.createElement("User");
                userElement.addAttribute("Name",pfOrganVo.getOrganName());
                userElement.addAttribute("Id",organId);
                List<PfUserVo> userVoList=  sysUserService.getUserListByOragn(organId);
                for(PfUserVo pfUserVo:userVoList) {
                    Element userInfoElement = userElement.addElement("UserInfo");
                    userInfoElement.addAttribute("Name", pfUserVo.getUserName());
                    userInfoElement.addAttribute("Id", pfUserVo.getUserId());

                }
                activityElement.add(userElement);
            }
        }
        taskTurnInfo = fileDoc.getRootElement().asXML();
        updateTurnInfoToGlobalVal(taskId, wiid, taskTurnInfo);
    }

    /**
     * 从用户待办任务的下一个活动的参与者中，移除人员
     * @param userId 用户Id
     * @param taskId 待办任务Id
     * @param activityDefinitionId 下一个活动定义Id
     * @param users 已有用户Ids
     */
    @Override
    public void removeUserFromTurnInfo(String userId, String taskId, String activityDefinitionId, Collection<String> users) throws Exception {
        String wiid = sysTaskService.getActivity(sysTaskService.getTask(taskId).getActivityId()).getWorkflowInstanceId();
        String taskTurnInfo = getTurnInfoFromGlobalVal(taskId,wiid);
        if(StringUtils.isBlank(taskTurnInfo)){
            taskTurnInfo = getTaskDefaultTurnInfo(userId, taskId);
        }
        Document fileDoc = DocumentHelper.parseText(taskTurnInfo);
        Element rootElement = fileDoc.getRootElement();
        Element activityElement = (Element)rootElement.selectSingleNode("Activity[@Id='"+activityDefinitionId+"']");
        if(activityElement!=null){
            for(String removeUserId:users) {
                List<Element> userInfoElementList = activityElement.selectNodes("//User/UserInfo[@Id='" + removeUserId + "']");
                if(userInfoElementList!=null&&!userInfoElementList.isEmpty()) {
                    for(Element userInfoElement :userInfoElementList) {
                        Element userElement = userInfoElement.getParent();
                        userElement.remove(userInfoElement);
                        if (!userElement.hasContent())
                            userElement.getParent().remove(userElement);
                    }
                }
            }
        }
        taskTurnInfo = fileDoc.getRootElement().asXML();
        updateTurnInfoToGlobalVal(taskId,wiid,taskTurnInfo);
    }

    /**
     * 从用户待办任务的下一个活动的参与者中，移除角色
     * @param userId 用户Id
     * @param taskId 待办任务Id
     * @param activityDefinitionId 下一个活动定义Id
     * @param roles 已有角色Ids
     */
    @Override
    public void removeRoleFromTurnInfo(String userId, String taskId, String activityDefinitionId, Collection<String> roles) throws Exception {
        removeOrganFromTurnInfo(userId, taskId, activityDefinitionId, roles);
    }

    /**
     * 从用户待办任务的下一个活动的参与者中，移除部门
     * @param userId 用户Id
     * @param taskId 待办任务Id
     * @param activityDefinitionId 下一个活动定义Id
     * @param organs 已有部门Ids
     */
    @Override
    public void removeOrganFromTurnInfo(String userId, String taskId, String activityDefinitionId, Collection<String> organs) throws Exception {
        String wiid = sysTaskService.getActivity(sysTaskService.getTask(taskId).getActivityId()).getWorkflowInstanceId();
        String taskTurnInfo = getTurnInfoFromGlobalVal(taskId,wiid);
        if(StringUtils.isBlank(taskTurnInfo)){
            taskTurnInfo = getTaskDefaultTurnInfo(userId, taskId);
        }
        Document fileDoc = DocumentHelper.parseText(taskTurnInfo);
        Element rootElement = fileDoc.getRootElement();
        Element activityElement = (Element)rootElement.selectSingleNode("Activity[@Id='"+activityDefinitionId+"']");
        if(activityElement!=null){
            for(String organId:organs) {
                Element userElement = (Element) activityElement.selectSingleNode("User[@Id='" + organId + "']");
                if (userElement != null) {
                    activityElement.remove(userElement);
                }
            }
        }
        taskTurnInfo = fileDoc.getRootElement().asXML();
        updateTurnInfoToGlobalVal(taskId, wiid, taskTurnInfo);
    }

    /**
     * 重新设置用户待办任务的下一个活动的参与角色的人员
     * @param userId 用户Id
     * @param taskId 待办任务Id
     * @param activityDefinitionId 下一个活动定义Id
     * @param roleId 参与角色Id
     * @param users 新用户Ids
     */
    @Override
    public void resetRoleUser(String userId, String taskId, String activityDefinitionId, String roleId, Collection<String> users) throws Exception {
        String wiid = sysTaskService.getActivity(sysTaskService.getTask(taskId).getActivityId()).getWorkflowInstanceId();
        String taskTurnInfo = getTurnInfoFromGlobalVal(taskId,wiid);
        if(StringUtils.isBlank(taskTurnInfo)){
            taskTurnInfo = getTaskDefaultTurnInfo(userId, taskId);
        }
        Document fileDoc = DocumentHelper.parseText(taskTurnInfo);
        Element rootElement = fileDoc.getRootElement();
        Element activityElement = (Element)rootElement.selectSingleNode("Activity[@Id='"+activityDefinitionId+"']");
        if(activityElement!=null){
            Element roleElement = (Element)activityElement.selectSingleNode("User[@Id='"+roleId+"']");
            if(roleElement!=null){
                activityElement.remove(roleElement);
            }
            if(users!=null&&!users.isEmpty()) {
                PfRoleVo pfRoleVo = sysUserService.getRoleVo(roleId);
                Element userElement = DocumentHelper.createElement("User");
                userElement.addAttribute("Name", pfRoleVo.getRoleName());
                userElement.addAttribute("Id", roleId);
                for (String addUserId : users) {
                    PfUserVo pfUserVo = sysUserService.getUserVo(addUserId);
                    Element userInfoElement = userElement.addElement("UserInfo");
                    userInfoElement.addAttribute("Name", pfUserVo.getUserName());
                    userInfoElement.addAttribute("Id", addUserId);
                }
                activityElement.add(userElement);
            }
        }
        taskTurnInfo = fileDoc.getRootElement().asXML();
        updateTurnInfoToGlobalVal(taskId,wiid,taskTurnInfo);
    }

    /**
     * 重新设置用户待办任务的下一个活动的参与部门的人员
     * @param userId 用户Id
     * @param taskId 待办任务Id
     * @param activityDefinitionId 下一个活动定义Id
     * @param organId 参与部门Id
     * @param users 新用户Ids
     */
    @Override
    public void resetOrganUser(String userId, String taskId, String activityDefinitionId, String organId, Collection<String> users) throws Exception {
        String wiid = sysTaskService.getActivity(sysTaskService.getTask(taskId).getActivityId()).getWorkflowInstanceId();
        String taskTurnInfo = getTurnInfoFromGlobalVal(taskId,wiid);
        if(StringUtils.isBlank(taskTurnInfo)){
            taskTurnInfo = getTaskDefaultTurnInfo(userId, taskId);
        }
        Document fileDoc = DocumentHelper.parseText(taskTurnInfo);
        Element rootElement = fileDoc.getRootElement();
        Element activityElement = (Element)rootElement.selectSingleNode("Activity[@Id='"+activityDefinitionId+"']");
        if(activityElement!=null){
            Element organElement = (Element)activityElement.selectSingleNode("User[@Id='"+organId+"']");
            if(organElement!=null){
                activityElement.remove(organElement);
            }
            if(users!=null&&!users.isEmpty()) {
                PfOrganVo pfOrganVo = sysUserService.getOrganVo(organId);
                Element userElement = DocumentHelper.createElement("User");
                userElement.addAttribute("Name", pfOrganVo.getOrganName());
                userElement.addAttribute("Id", organId);
                for (String addUserId : users) {
                    PfUserVo pfUserVo = sysUserService.getUserVo(addUserId);
                    Element userInfoElement = userElement.addElement("UserInfo");
                    userInfoElement.addAttribute("Name", pfUserVo.getUserName());
                    userInfoElement.addAttribute("Id", addUserId);
                }
                activityElement.add(userElement);
            }
        }
        taskTurnInfo = fileDoc.getRootElement().asXML();
        updateTurnInfoToGlobalVal(taskId,wiid,taskTurnInfo);
    }

    /**
     * 清空用户待办任务的下一个活动的所有参与者信息
     * @param userId 用户Id
     * @param taskId 待办任务Id
     * @param activityDefinitionId 下一个活动定义Id
     * @param isActivityRemoved 参与者信息删除后，是否同时删除该活动信息。
     *                          如果删除了该活动信息，那么待办任务将无法转发给这个活动，如果不删除，则后续需往该活动中重新添加参与者信息。
     */
    @Override
    public void removeActivityUsers(String userId,String taskId,String activityDefinitionId,Boolean isActivityRemoved) throws Exception {
        String wiid = sysTaskService.getActivity(sysTaskService.getTask(taskId).getActivityId()).getWorkflowInstanceId();
        String taskTurnInfo = getTurnInfoFromGlobalVal(taskId,wiid);
        if(StringUtils.isBlank(taskTurnInfo)){
            taskTurnInfo = getTaskDefaultTurnInfo(userId, taskId);
        }
        Document fileDoc = DocumentHelper.parseText(taskTurnInfo);
        Element rootElement = fileDoc.getRootElement();
        Element activityElement = (Element)rootElement.selectSingleNode("Activity[@Id='"+activityDefinitionId+"']");
        if(activityElement!=null){
            List<Element> userElements = activityElement.selectNodes("User");
            if(userElements!=null&&!userElements.isEmpty()){
                for(Element useElement : userElements) {
                    activityElement.remove(useElement);
                }
            }
            if(isActivityRemoved!=null&&isActivityRemoved){
                activityElement.getParent().remove(activityElement);
            }
        }
        taskTurnInfo = fileDoc.getRootElement().asXML();
        updateTurnInfoToGlobalVal(taskId,wiid,taskTurnInfo);
    }


    private String formatTurnInfo(WorkFlowInfo info) throws Exception {

        // 获取工作流实例模型
        WorkFlowXml xmlDao = WorkFlowXmlUtil.getInstanceModel(info.getWorkFlowIntanceVo());
        // 获取当前活动定义
        ActivityModel activityModel = xmlDao.getActivity(info.getSourceActivity().getActivityDefinitionId());

        String filterType = null;
        List<String> filterStr = null;
        if(StringUtils.isNotBlank(activityModel.getFilterInfo())){
            //过滤信息对象
            Document fileDoc = DocumentHelper.parseText(activityModel.getFilterInfo());
            //过滤类型：User,Organ,Role
            String user = fileDoc.getRootElement().valueOf("@type");
            Node corNode = fileDoc.selectSingleNode("//Filter/" + user + "[@Id='" + SessionUtil.getUserId(ServletActionContext.getRequest()) + "']/Correspondence");
            if(corNode != null){
                filterType = corNode.valueOf("@type");
                List<DefaultText> organList = corNode.selectNodes(filterType + "/text()");
                if(organList != null && !organList.isEmpty()){
                    filterStr = new ArrayList<String>();
                    for(DefaultText text : organList){
                        filterStr.add(text.getText());
                    }
                }
            }
        }

        WorkFlowTransInfo transInfo=info.getTransInfo();
        Document doc = DocumentHelper.createDocument();
        Element root = doc.addElement("Activitys");
        if (transInfo.getTransType().equalsIgnoreCase("and"))
            root.addAttribute("RelType", "and");
        else
            root.addAttribute("RelType", "or");

        if(activityModel.getIsRequiredOpinion().equalsIgnoreCase("true")){
            root.addAttribute("ReqOpinion", "true");
        }else{
            root.addAttribute("ReqOpinion", "false");
        }
        if(activityModel.getIsSendSMS().equalsIgnoreCase("true")){
            root.addAttribute("SendSMS","true");
        }else{
            root.addAttribute("SendSMS","false");
        }
        if(StringUtils.isNotBlank(activityModel.getCooperRootId())){
            root.addAttribute("cooperRootId",activityModel.getCooperRootId());
        }

        List<ActivityModel> lstTrans = transInfo.getTranActivitys();
        if (lstTrans.size()<2)
            root.addAttribute("RelType", "or");
        for (ActivityModel aModel : lstTrans) {
            Element ActivityEle = root.addElement("Activity");
            ActivityEle.addAttribute("Name", aModel.getActivityDefineName());
            ActivityEle.addAttribute("Id", aModel.getDefineId());
            if (aModel.isMutiSelect())
                ActivityEle.addAttribute("MutiSelect", "true");

            if(!aModel.isSelectAll())
                ActivityEle.addAttribute("SelectAll","false");
            else
                ActivityEle.addAttribute("SelectAll","true");

            if(aModel.getActivityDefineName().equalsIgnoreCase(activityModel.getDefaultSelectName()))
                ActivityEle.addAttribute("DefaultSelected", "true");
            else
                ActivityEle.addAttribute("DefaultSelected", "false");
            String userName=null;
            if(StringUtils.isNotBlank(aModel.getExtendedAttribute("DefaultSelectedUser"))){
                userName=aModel.getExtendedAttribute("DefaultSelectedUser");
            }
            for (PerformerTaskModel userModel : aModel.getPerformerModelList()) {
                if(filterStr != null && userModel.getType().equals(filterType)){
                    if(!filterStr.contains(userModel.getId())){
                        continue;
                    }
                }
                Element User = ActivityEle.addElement("User");
                User.addAttribute("Name", userModel.getName());
                User.addAttribute("Id", userModel.getId());
                for (PfUserVo vo : userModel.getUserList()) {
                    Element UserInfo = User.addElement("UserInfo");
                    UserInfo.addAttribute("Name", vo.getUserName());
                    UserInfo.addAttribute("Id", vo.getUserId());
                    if (StringUtils.isNotBlank(userName) && vo.getUserName().equalsIgnoreCase(userName)){
                        User.addAttribute("DefaultSelected", "true");
                        UserInfo.addAttribute("DefaultSelected", "true");
                    }
                    if (vo.getUserId().equals(SessionUtil.getCurrentUserId())){
                        User.addAttribute("DefaultSelected", "true");
                        UserInfo.addAttribute("DefaultSelected", "true");
                    }
                }
            }
        }
        // 加入是否可以办结
        if (transInfo.isCanFinish()) {
            Element ActivityEle = root.addElement("Activity");
            ActivityEle.addAttribute("Name", "任务办结");
            ActivityEle.addAttribute("Id", "-1");
        }

        doc = httpFilterXml(doc,activityModel,info);

        return doc.getRootElement().asXML();
    }

    /**
     * 通过TurnInfo处理url
     * @param doc
     * @return
     */
    private Document httpFilterXml(Document doc,ActivityModel actModel,WorkFlowInfo info){
        if(StringUtils.isNotBlank(actModel.getTurnInfoUrl())){
            String url = actModel.getTurnInfoUrl();
            if(!url.startsWith("http")){
                url = AppConfig.getPlaceholderValue(url);
            }
            HttpClient httpClient = new org.apache.commons.httpclient.HttpClient();
            PostMethod postMethod = new Utf8PostMethod(url);
            NameValuePair[] data = {
                    new NameValuePair("turnInfo",doc.asXML()),
                    new NameValuePair("userId",info.getUserId()),
                    new NameValuePair("proId",(info.getWorkFlowIntanceVo() == null)? "":info.getWorkFlowIntanceVo().getProId())
            };

            postMethod.setRequestBody(data);
            int status = 0;
            try {
                status = httpClient.executeMethod(postMethod);
            } catch (IOException e) {
                log.error("TurnInfo filter url request failer{}",e);
            }
            if (status == HttpStatus.SC_OK){
                try {
                    String xml = postMethod.getResponseBodyAsString();
                    if(StringUtils.isNotBlank(xml)){
                        doc = DocumentHelper.parseText(xml);
                    }
                } catch (Exception e) {
                    log.error("TurnInfo filter url request success,prase failure{}",e);
                }
            }else{
                log.error("TurnInfo filter url request failer");
            }
        }
        return doc;
    }

    /**
     * 获取到全局变量中的待办任务的转发信息
     * @param taskId 待办任务Id
     * @param wiid 工作流Id
     * @return
     */
    public String getTurnInfoFromGlobalVal(String taskId,String wiid){
        Map<String,Object> globalVal = workFlowIntanceService.getGobalVal(wiid);
        if(globalVal==null)
            return null;
        String key = "taskTurnInfo_"+taskId;
        if(globalVal.containsKey(key)) {
            return String.valueOf(globalVal.get(key));
        }
        return null;
    }

    /**
     * 将待办任务的转发信息更新到全局变量中
     * @param taskId 待办任务Id
     * @param wiid 工作流实例Id
     * @param turnInfo xml格式的转发信息，请参考转发xml格式
     */
    public void updateTurnInfoToGlobalVal(String taskId,String wiid,String turnInfo){
        String key = "taskTurnInfo_"+taskId;
        workFlowIntanceService.updateGobalVal(wiid,key,turnInfo);
    }

    public void setSysSubProcessService(SysSubProcessService sysSubProcessService) {
        this.sysSubProcessService = sysSubProcessService;
    }

    public void setWorkFlowService(WorkFlowCoreService workFlowService) {
        this.workFlowService = workFlowService;
    }

    public void setSysTaskService(SysTaskService sysTaskService) {
        this.sysTaskService = sysTaskService;
    }

    public void setWorkFlowIntanceService(SysWorkFlowInstanceService workFlowIntanceService) {
        this.workFlowIntanceService = workFlowIntanceService;
    }

    public void setSysUserService(SysUserService sysUserService) {
        this.sysUserService = sysUserService;
    }
}
