/*
 * Author: xyang
 *
 * Project: egov
 *
 * File: PortalDaoImpl.java
 *
 * LastModified: 2009-11-30 08:40:51
 *
 * Copyright (c) 2009 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.plat.portal.impl;

import com.gtis.plat.portal.PortalDao;
import com.gtis.plat.portal.model.PortletEntity;
import com.gtis.plat.portal.model.Site;
import com.gtis.util.UUIDGenerator;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.jdbc.core.simple.SimpleJdbcTemplate;
import org.springframework.jdbc.core.support.AbstractLobCreatingPreparedStatementCallback;
import org.springframework.jdbc.support.lob.LobCreator;
import org.springframework.jdbc.support.lob.LobHandler;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import javax.sql.DataSource;
import java.io.*;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2009-11-30
 */
@Repository("portalDao")
public class PortalDaoImpl implements PortalDao {

    private static final Logger logger = LoggerFactory.getLogger(PortalDaoImpl.class);

    protected LobHandler lobHandler;
    private SimpleJdbcTemplate simpleJdbcTemplate;

    public void setLobHandler(LobHandler lobHandler) {
        this.lobHandler = lobHandler;
    }

    public void setDataSource(DataSource dataSource) {
        this.simpleJdbcTemplate = new SimpleJdbcTemplate(dataSource);
    }

    @Transactional(readOnly = true)
    public PortletEntity getPortletEntity(String id) {
        final String sql = "select entity from t_pf_portlet_entity where id=?";
        try {
            return simpleJdbcTemplate.queryForObject(sql, new PortletEntityRowMapper(), id);
        } catch (EmptyResultDataAccessException e) {
            return null;
        }
    }

    @Transactional(readOnly = true)
    public List<PortletEntity> getPortletEntities() {
        final String sql = "select entity from t_pf_portlet_entity";
        try {
            return simpleJdbcTemplate.query(sql, new PortletEntityRowMapper());
        } catch (EmptyResultDataAccessException e) {
            return new ArrayList<PortletEntity>();
        }
    }

    @Transactional
    public PortletEntity savePortletEntity(final PortletEntity portletEntity) {
        final String sql;
        if (StringUtils.isBlank(portletEntity.getId())) {
            portletEntity.setId(UUIDGenerator.generate());
            sql = "insert into t_pf_portlet_entity (entity,id) values (?,?)";
        } else
            sql = "update t_pf_portlet_entity set entity=? where id=?";
        simpleJdbcTemplate.getJdbcOperations().execute(sql, new AbstractLobCreatingPreparedStatementCallback(lobHandler) {
            protected void setValues(PreparedStatement ps, LobCreator lobCreator) throws SQLException, DataAccessException {
                try {
                    ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    ObjectOutputStream oos = new ObjectOutputStream(baos);
                    try {
                        oos.writeObject(portletEntity);
                        oos.flush();
                        lobCreator.setBlobAsBytes(ps, 1, baos.toByteArray());
                    } finally {
                        oos.close();
                    }
                } catch (IOException ex) {
                    throw new SQLException("I/O errors during LOB access: " + ex.getMessage());
                }
                ps.setString(2, portletEntity.getId());
            }
        });
        return portletEntity;
    }

    @Transactional
    public void removePortletEntity(String id) {
        final String sql = "delete from t_pf_portlet_entity where id=?";
        simpleJdbcTemplate.update(sql, id);
        logger.debug("remove portletEntity,id:({})", id);
    }

    public boolean portletEntityExist(String id) {
        final String sql = "select count(0) from t_pf_portlet_entity where id=?";
        return simpleJdbcTemplate.queryForInt(sql, id) > 0;
    }

    @Transactional(readOnly = true)
    public Site getSite(String owner) {
        final String sql = "select site from t_pf_portal_site where owner=?";
        try {
            return simpleJdbcTemplate.queryForObject(sql, new SiteRowMapper(), owner);
        } catch (EmptyResultDataAccessException e) {
            return null;
        }
    }

    @Transactional
    public Site saveSite(final Site site) {
        final String sql = siteExist(site.getOwner()) ?
                "update t_pf_portal_site set site=? where owner=?" :
                "insert into t_pf_portal_site (site,owner) values (?,?)";
        simpleJdbcTemplate.getJdbcOperations().execute(sql, new AbstractLobCreatingPreparedStatementCallback(lobHandler) {
            protected void setValues(PreparedStatement ps, LobCreator lobCreator) throws SQLException, DataAccessException {
                try {
                    ByteArrayOutputStream baos = new ByteArrayOutputStream();
                    ObjectOutputStream oos = new ObjectOutputStream(baos);
                    try {
                        oos.writeObject(site);
                        oos.flush();
                        lobCreator.setBlobAsBytes(ps, 1, baos.toByteArray());
                    } finally {
                        oos.close();
                    }
                } catch (IOException ex) {
                    throw new SQLException("I/O errors during LOB access: " + ex.getMessage());
                }
                ps.setString(2, site.getOwner());
            }
        });
        return site;
    }

    @Transactional
    public void removeSite(String owner) {
        final String sql = "delete from t_pf_portal_site where owner=?";
        simpleJdbcTemplate.update(sql, owner);
        logger.debug("remove site,owner:({})", owner);
    }

    public void resetUserSite() {
        final String sql = "delete from t_pf_portal_site where owner!='" + Site.GLOBAL_SITE + "' and owner not like '" + Site.ROLE_SITE_PREFIX + "%'";
        simpleJdbcTemplate.update(sql);
    }

    public Map<String, Site> getRoleSitesMap() {
        final String sql = "select site from t_pf_portal_site where owner like '" + Site.ROLE_SITE_PREFIX + "%'";
        Map<String, Site> map = new HashMap<String, Site>();
        for (Site site : simpleJdbcTemplate.query(sql, new SiteRowMapper())) {
            map.put(site.getOwner(), site);
        }
        return map;
    }

    public boolean siteExist(String owner) {
        final String sql = "select count(0) from t_pf_portal_site where owner=?";
        return simpleJdbcTemplate.queryForInt(sql, owner) > 0;
    }

    private class SiteRowMapper implements ParameterizedRowMapper<Site> {
        public Site mapRow(ResultSet rs, int rownum) throws SQLException {
            try {
                byte[] bytes = lobHandler.getBlobAsBytes(rs, 1);
                if (bytes != null && bytes.length > 0) {
                    ObjectInputStream ois = new ObjectInputStream(new ByteArrayInputStream(bytes));
                    try {
                        return (Site) ois.readObject();
                    } catch (ClassNotFoundException ex) {
                        throw new SQLException("Could not deserialize BLOB contents: " + ex.getMessage());
                    } finally {
                        ois.close();
                    }
                }
            } catch (IOException ignored) {
            }
            return null;
        }
    }

    private class PortletEntityRowMapper implements ParameterizedRowMapper<PortletEntity> {
        public PortletEntity mapRow(ResultSet rs, int rownum) throws SQLException {
            try {
                byte[] bytes = lobHandler.getBlobAsBytes(rs, 1);
                if (bytes != null && bytes.length > 0) {
                    ObjectInputStream ois = new ObjectInputStream(new ByteArrayInputStream(bytes));
                    try {
                        return (PortletEntity) ois.readObject();
                    } catch (ClassNotFoundException ex) {
                        throw new SQLException("Could not deserialize BLOB contents: " + ex.getMessage());
                    } finally {
                        ois.close();
                    }
                }
            } catch (IOException ignored) {
            }
            return null;
        }
    }
}
