package com.gtis.plat.service.impl;

import com.google.common.collect.Maps;
import com.gtis.plat.dao.SysTaskDao;
import com.gtis.plat.dao.SysTaskHistoryDao;
import com.gtis.plat.service.SysActivityService;
import com.gtis.plat.service.SysCalendarService;
import com.gtis.plat.service.SysTaskService;
import com.gtis.plat.vo.PfActivityVo;
import com.gtis.plat.vo.PfTaskVo;
import com.gtis.plat.wf.WorkFlowEnum;
import com.gtis.plat.wf.WorkFlowUtil;
import com.gtis.util.CalendarUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

public class SysTaskServiceImpl implements SysTaskService {
    private static final Log log = LogFactory.getLog(SysTaskServiceImpl.class);

    SysTaskDao taskDao;

    SysTaskHistoryDao taskHistoryDao;
    SysCalendarService sysCalendarService;
    SysActivityService activityService;


    public void setActivityService(SysActivityService activityService) {
        this.activityService = activityService;
    }

    public void setSysCalendarService(SysCalendarService sysCalendarService) {
        this.sysCalendarService = sysCalendarService;
    }

    public SysTaskHistoryDao getTaskHistoryDao() {
        return taskHistoryDao;
    }

    public void setTaskHistoryDao(SysTaskHistoryDao taskHistoryDao) {
        this.taskHistoryDao = taskHistoryDao;
    }

    public void createActivity(PfActivityVo activityVo) {
        if(log.isInfoEnabled())
            log.info("Creating activity id:"+activityVo.getActivityId()+",name:"+activityVo.getActivityName()+",workflow" +
                    " instance id:"+activityVo.getWorkflowInstanceId());
        // /避免创建相同的活动
        PfActivityVo aVo = taskDao.getActivityBywIdandadId(activityVo
                .getWorkflowInstanceId(), activityVo.getActivityDefinitionId());
        if (aVo == null) {
            activityVo.setBeginTime(Calendar.getInstance().getTime());
            taskDao.createActivity(activityVo);
        } else {
            this.updateActivityStadus(activityVo.getActivityId(),
                    activityVo.getActivityState());
        }
    }

    @Transactional
    public void createTask(PfTaskVo taskVo) {
        // 不要创建重复的任务
        List<PfTaskVo> lstTasks = taskDao.getTaskListByUserIdAndActivity(taskVo
                .getActivityId(), taskVo.getUserVo().getUserId());
        if (lstTasks == null || lstTasks.size() == 0) {
            Date beginDate = Calendar.getInstance().getTime();
            if (taskVo.getBeginTime()==null) {
                taskVo.setBeginTime(beginDate);
            }
            if (taskVo.getOverTime()==null) {
                //获得截至时间
                PfActivityVo activityVo = activityService.getActivityById(taskVo.getActivityId());
                if(activityVo!=null)
                    taskVo.setOverTime(sysCalendarService.getOverTime(beginDate, activityVo.getTimeLimit()));
            }
            taskDao.createTask(taskVo);
        } else {
            /////这里需要更新taskbefore
            if (!StringUtils.isBlank(taskVo.getTaskBefore())) {
                for (PfTaskVo aTask : lstTasks) {
                    taskDao.updateTasksByBefore(aTask.getTaskId(), taskVo.getTaskBefore());
                    //更新截至时间
                    PfActivityVo activityVo = activityService.getActivityById(aTask.getActivityId());
                    if(activityVo!=null&&aTask.getOverTime()==null)
                        taskDao.updateTaskOverTime(aTask.getTaskId(),sysCalendarService.getOverTime(aTask.getBeginTime(), activityVo.getTimeLimit()));
                }
            }
        }
    }

    /**
     * 批量创建任务
     *
     * @param tasklist
     */
    @Transactional
    public void createTasks(final List<PfTaskVo> tasklist) {
        for (PfTaskVo taskVo : tasklist) {
            this.createTask(taskVo);
        }
//		taskDao.createTasks(tasklist);
    }

    public List<PfActivityVo> getActivityByBefore(
            String activityId) {
        return this.taskDao.getActivityByBefore(activityId);
    }

    /**
     * 更新历史任务
     */
    public void updateHistoryTask(PfTaskVo taskVo) {
        this.taskHistoryDao.updateTaskHistory(taskVo);
    }

    public void updateTaskHistoryOverTime(String taskId,Date overTime) {
        this.taskHistoryDao.updateTaskHistoryOverTime(taskId,overTime);
    }



    public void deleteActivityTask(String activityId) {
        taskDao.deleteActivityTask(activityId);
    }

    public void deleteTask(String taskId) {
        taskDao.deleteTask(taskId);
    }

    public void updateTaskOverTime(String taskId,Date overTime){
        taskDao.updateTaskOverTime(taskId,overTime);
    }

    public PfActivityVo getActivity(String activityId) {
        return taskDao.getActivity(activityId);
    }

    public PfActivityVo getActivityBywIdandadId(String workflowInstanceId,
                                                String activityDefinitionId) {
        return taskDao.getActivityBywIdandadId(workflowInstanceId,
                activityDefinitionId);
    }

    public List<PfTaskVo> getTaskListByActivity(String activityId) {
        return taskDao.getTaskListByActivity(activityId);
    }

    public List<PfTaskVo> getHistoryTasks(String activityId) {
        return taskHistoryDao.getHistoryTasksListByActivity(activityId);
    }

    public int getAllTaskCountByActivityId(String activityId){
        return taskDao.getAllTaskCountByActivityId(activityId);
    }

    public PfTaskVo getTask(String taskId) {
        return taskDao.getTask(taskId);
    }

    public PfTaskVo getTaskAll(String taskId) {
        return taskDao.getTaskAll(taskId);
    }

    public List<PfActivityVo> getWorkFlowInstanceActivityList(
            String workflowInstanceId) {
        return taskDao.getWorkFlowInstanceActivityList(workflowInstanceId);
    }

    public void insertTaskHistory(String taskId) {
        PfTaskVo taskVo=taskDao.getTask(taskId);
        if (taskVo!=null) {
            taskVo.setFinishTime(Calendar.getInstance().getTime());
            taskHistoryDao.insertTaskHistory(taskVo);
        }
    }

    public void updateActivityBackStadus(String activityId, boolean isBack) {
        if (isBack)
            taskDao.updateActivityBack(activityId);
        else
            taskDao.updateActivityForword(activityId);
    }

    public void updateActivityRemark(String activityId, String remark) {
        taskDao.updateActivityRemark(activityId, remark);
    }

    /**
     * 更新活动状态
     *
     * @param activityId
     * @param stadus
     */
    public void updateActivityStadus(String activityId, int stadus) {
        updateActivityStatus(activityId,stadus);
    }

    public void updateActivityStatusNonTransactional(String activityId, int status) {
        updateActivityStatus(activityId,status);
    }

    private void updateActivityStatus(String activityId, int status) {
        PfActivityVo activityVo = new PfActivityVo();
        activityVo.setActivityId(activityId);
        switch (status) {
            case WorkFlowEnum.ACTIVITY_START:
                activityVo.setBeginTime(Calendar.getInstance().getTime());
                activityVo.setActivityState(WorkFlowEnum.ACTIVITY_START);
                break;
            case WorkFlowEnum.ACTIVITY_END:
                activityVo.setFinishTime(Calendar.getInstance().getTime());
                activityVo.setActivityState(WorkFlowEnum.ACTIVITY_END);
                break;
            case WorkFlowEnum.ACTIVITY_BACK:
                activityVo.setFinishTime(Calendar.getInstance().getTime());
                activityVo.setActivityState(WorkFlowEnum.ACTIVITY_BACK);
                break;
            default:
                activityVo.setActivityState(status);
                break;
        }
        taskDao.updateActivityStadus(activityVo);
    }

    public void updateActivityHandlesDay(String activityId, int handlesDay) {
        PfActivityVo activityVo = new PfActivityVo();
        activityVo.setActivityId(activityId);
        activityVo.setHandlesDay(handlesDay);
        taskDao.updateActivityHandlesDay(activityVo);
    }

    public void updateCooperateStateDown(String activityId) {
        taskDao.updateCooperateStateDown(activityId);
    }

    public void updateCooperateStateUp(String activityId) {
        taskDao.updateCooperateStateUp(activityId);
    }

    public SysTaskDao getTaskDao() {
        return taskDao;
    }

    public void setTaskDao(SysTaskDao taskDao) {
        this.taskDao = taskDao;
    }

    public List<PfTaskVo> getHistoryTaskListByInstance(String workFlowInstanceId) {
        List<PfTaskVo> result=new ArrayList<PfTaskVo>();
        List<PfActivityVo> activityList= taskDao.getWorkFlowInstanceAllActivityList(workFlowInstanceId);
        for(PfActivityVo activityVo:activityList){
            //List<PfTaskVo> tasks=taskDao.getTaskListByActivity(activityVo.getActivityId());
            List<PfTaskVo> tasksHistory=taskHistoryDao.getHistoryTasksListByActivity(activityVo.getActivityId());
            //result.addAll(tasks);
            result.addAll(tasksHistory);
        }
        return result;
    }

    public List<PfTaskVo> getHistoryTaskListByDefineId(String workFlowInstanceId, String activityDefineId) {
        List<PfTaskVo> result=new ArrayList<PfTaskVo>();
        PfActivityVo activityVo=taskDao.getActivityBywIdandadId(workFlowInstanceId,activityDefineId);
        if (result!=null && activityVo!=null){
            List<PfTaskVo> tasksHistory=taskHistoryDao.getHistoryTasksListByActivity(activityVo.getActivityId());
            if(CollectionUtils.isNotEmpty(tasksHistory))
                result.addAll(tasksHistory);
        }
        return result;
    }

    public List<PfTaskVo> getTaskListByInstance(String workFlowInstanceId) {
        return taskDao.getTaskListByInstance(workFlowInstanceId);
    }

    public List<PfActivityVo> getWorkFlowInstanceAllActivityList(
            String workFlowInstanceId) {
        return taskDao.getWorkFlowInstanceAllActivityList(workFlowInstanceId);
    }

    /**
     * 根据task去获取流程所经历的所有活动，因此这个返回结果中可能会出现重复的活动，因为存在退回和取回的情况
     *
     * @param workflowInstanceId
     * @return
     */
    @Override
    public List<PfActivityVo> getWorkflowInstanceActivityByTask(String workflowInstanceId) {
        return taskDao.getWorkflowInstanceActivityByTask(workflowInstanceId);
    }

    public PfTaskVo getHistoryTask(String taskId) {
        return taskHistoryDao.getHistoryTask(taskId);
    }

    /**
     * 前一个办理人
     * @param taskVo
     * @return
     */
    public List<PfTaskVo> getHistoryTaskByBefore(PfTaskVo taskVo){
        List<PfTaskVo> lstResult=new ArrayList<PfTaskVo>();
        if (!StringUtils.isBlank(taskVo.getTaskBefore())){
            String[] taskIds= StringUtils.split(taskVo.getTaskBefore(),",");
            for(String taskId:taskIds){
                if (!StringUtils.isBlank(taskId)){
                    PfTaskVo taskHisVo=this.getHistoryTask(taskId);
                    if (taskHisVo==null)
                        taskHisVo=this.getTask(taskId);
                    if (taskHisVo!=null)
                        lstResult.add(taskHisVo);
                }
            }
        }
        return lstResult;
    }

    public void deleteWorkFlowInstanceActivity(String workflowIntanceId) {
        taskDao.deleteWorkFlowInstanceActivity(workflowIntanceId);
    }

    @Transactional
    public void deleteHistoryTaskByActivityId(String activityId) {
        List<PfTaskVo> tasks=taskHistoryDao.getHistoryTasksListByActivity(activityId);
        if (tasks!=null){
            for(PfTaskVo task:tasks)
                taskHistoryDao.deleteHistoryTask(task);
        }
        //taskDao.deleteHistoryTaskByActivityId(activityId);
    }


    public void updateTask(PfTaskVo taskVo) {
        taskDao.updateTask(taskVo);
    }

    @Transactional
    public void updateTempTask(String activityId) {
        taskDao.updateTempTask(activityId);
        //如果是多个转给一个的时候，临时任务的理论办结时间应该按照最后一个转发过来的时间来处理，而不是最初临时任务创建的时间
        PfActivityVo activityVo = getActivity(activityId);
        List<PfTaskVo> taskList = getTaskListByActivity(WorkFlowUtil.buildTEMPActivityId(activityId));
        if (taskList == null || taskList.size() == 0){
            taskList = getTaskListByActivity(activityId);
        }
        if (taskList != null && !taskList.isEmpty()){
            for (int i = 0; i < taskList.size(); i++) {
                PfTaskVo taskVo = taskList.get(i);
                Date taskOverTime = sysCalendarService.getOverTime(CalendarUtil.getCurHMDate(), activityVo.getTimeLimit());
                taskDao.updateTaskOverTime(taskVo.getTaskId(),taskOverTime);
            }
        }
    }

    public List<PfTaskVo> getTasksByBefore(String taskId) {
        return taskDao.getTasksByBefore(taskId);
    }


    public void createTaskByHistroy(String taskId) {
        HashMap<String,Object> mapParam=Maps.newHashMap();
        mapParam.put("taskId",taskId);
        mapParam.put("beginTime",Calendar.getInstance().getTime());
        taskDao.createTaskByHistroy(mapParam);
    }


    public void deleteHistoryTask(String taskId) {
        PfTaskVo task=taskHistoryDao.getHistoryTask(taskId);
        if (task!=null)
            taskHistoryDao.deleteHistoryTask(task);
    }


    public void deleteActivity(String activityId) {
        taskDao.deleteActivity(activityId);
    }


    public void updateTaskStadus(String taskId) {
        //更新ISACCEPTED
        String acceptTaskId=taskDao.selectIsAcceptedTask(taskId);
        //判断是否已经接受过了
        if (StringUtils.isBlank(acceptTaskId))
            taskDao.updateTaskStadus(taskId);
    }

    public List<PfTaskVo> getHistoryTaskListByActivity(String activityId){
        return  taskHistoryDao.getHistoryTasksListByActivity(activityId);
    }


    public void deleteOtherTasks(String taskId){
        taskDao.deleteOtherTasks(taskId);
    }


    /**
     * 设置历史任务为重复历史任务
     */
    public void updateAssignmentHistoryRepeat(String assignmentId){
        PfTaskVo taskVo=taskHistoryDao.getHistoryTask(assignmentId);
        taskVo.setRepeat(true);
        taskHistoryDao.updateTaskHistory(taskVo);
    }

    /**
     * 根据用户id和活动id进行取件
     * @param activityId
     * @param userIds，英文逗号隔开,单个无需逗号
     */
    public String pickupWorkflowTask(String activityId,String userIds){
        /*取件逻辑：
        * 1、首先判断当前活动中是否存在该人员任务，不存在直接返回无权操作，存在进行下一步操作
        * 2、修改该活动下该人员的任务状态为已读
        * 3、删除该活动下其他任务（此处最好是建立一个过程表，用于存储这些删除的任务，便于退件）
        * */
        String msg = null;
        if (StringUtils.isNotBlank(activityId) && StringUtils.isNotBlank(userIds)){
            userIds = "," + userIds + ",";
            List<PfTaskVo> taskList = getTaskListByActivity(activityId);
            if (taskList != null && taskList.size() > 0){
                String taskId = null;
                List<String> taskIdList = new ArrayList<String>();
                //第一步判断
                for (int i = 0; i < taskList.size(); i++) {
                    PfTaskVo taskVo = taskList.get(i);
                    if (StringUtils.contains(userIds, "," + taskVo.getUserVo().getUserId() + ",")){
                        taskId = taskVo.getTaskId();
                        //从集合中去掉该任务（因其他任务在下一步需要被删掉）
                        taskIdList.add(taskId);
                        taskList.remove(i);
                        i--;
                    }
                }

                if (taskIdList != null && taskIdList.size() > 0){
                    //第二步修改状态
                    for (int i = 0; i < taskIdList.size(); i++) {
                        updateTaskStadus(taskIdList.get(i));
                    }
                    //第三步删除
                    for (int i = 0; i < taskList.size(); i++) {
                        deleteTask(taskList.get(i).getTaskId());
                    }
                }else {
                    msg = "当前活动无该人员待办任务记录，请检查参数！";
                }
            }else {
                msg = "当前活动无待办任务记录，请检查参数！";
            }
        }else {
            msg = "参数不完整，请检查参数！";
        }
        return msg;
     }
}