package com.gtis.plat.service.impl;

import com.gtis.plat.dao.SysCalendarDAO;
import com.gtis.plat.service.SysCalendarService;
import org.apache.commons.collections.CollectionUtils;

import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.joda.time.Period;
import org.joda.time.PeriodType;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Pattern;
/**
 * 根据整个工作日来进行超时计算，当天并不计算在内，而是从第二天（工作日）的零点开始计算
 * @author <a href="mailto:shenjian@gtmap.cn">shenjian</a>
 * @version 1.0, 2015/2/27
 */
public class SysCalendarServiceWholeDayImpl implements SysCalendarService {

    static Logger logger = LoggerFactory.getLogger(SysCalendarServiceWholeDayImpl.class);

    SysCalendarDAO calendarDAO;

    static List<String> specialHolidays=new ArrayList<String>();
    static List<String> specialWorkdays=new ArrayList<String>();

    /**
     * @author <a href="mailto:shenjian@gtmap.cn">shenjian</a>
     * @version 2.0, 2018/4/20
     * @description 使用Joda-Time包替换JDK的格式化，解决SimpleDateFormat线程不安全问题
     */
    static final DateTimeFormatter simpleDateFormat = DateTimeFormat.forPattern("yyyy-MM-dd");
    static final String SYS_MAX_DATE = "2999-12-31";


    /**
     * 初始化，只取去年、今年和明年的数据,3年的数据
     */
    public void init(){
        //从配置文件读取信息
        Calendar ca=Calendar.getInstance();
        int yearInt=Calendar.getInstance().get(Calendar.YEAR);
        ca.set(Calendar.YEAR,yearInt-1);
        ca.set(Calendar.DAY_OF_YEAR, 1);//本年第一天
        Date beginDate=ca.getTime();
        ca.set(Calendar.YEAR,yearInt+2);
        ca.set(Calendar.DAY_OF_YEAR, 1);//本年第一天
        ca.add(Calendar.DAY_OF_YEAR, -1);
        Date endDate=ca.getTime();
        List<Map> specialDates=calendarDAO.getSpecialDateList(beginDate,endDate);
        for(Map sDate:specialDates){
            if (sDate.get("CAL_TYPE").equals("工作日"))
                specialWorkdays.add(new DateTime(sDate.get("CAL_DATE")).toString(simpleDateFormat));
            else
                specialHolidays.add(new DateTime(sDate.get("CAL_DATE")).toString(simpleDateFormat));
        }
    }
    /**
     * 判断是否为double类型
     * @param str
     * @return
     */
    public static boolean isNumeric(String str){
        Pattern pattern = Pattern.compile("[-+]?[0-9]+(\\.[0-9]+)?");
        return pattern.matcher(str).matches();
    }

    @Override
    public Date getOverTime(Date beginDate, String timeLimit) {

        if (StringUtils.isNotBlank(timeLimit)){
            double days = 0;
            if(isNumeric(timeLimit)){
            	days = Double.parseDouble(timeLimit);
            }else{
            	days = Math.ceil(Double.parseDouble(timeLimit.substring(0,timeLimit.length()-1))/24);
            }
            if (days>0 && beginDate!=null){  //从第二天的0点开始计算工作日
                Date bDate=null;
                Calendar beginCalendar = Calendar.getInstance();
                beginCalendar.setTime(beginDate);
                beginCalendar.add(Calendar.DAY_OF_YEAR, 1);
                beginCalendar.set(Calendar.HOUR_OF_DAY,0);
                beginCalendar.set(Calendar.SECOND,0);
                beginCalendar.set(Calendar.MINUTE,0);
                beginCalendar.set(Calendar.MILLISECOND,0);
                if(IsHoliday(beginCalendar.getTime()))
                    bDate=getNextWorkDay(beginCalendar.getTime());
                else
                    bDate = beginCalendar.getTime();

                Calendar overCalendar = Calendar.getInstance();
                overCalendar.setTime(bDate);
                overCalendar.set(Calendar.HOUR_OF_DAY,23);
                overCalendar.set(Calendar.SECOND,59);
                overCalendar.set(Calendar.MINUTE,59);
                overCalendar.set(Calendar.MILLISECOND,999);

                Date overDate = overCalendar.getTime();
                for(int i=1;i<days;i++){
                    overDate=getNextWorkDay(overDate);
                }
                return overDate;
            }else if(days==0 && beginDate!=null){  //如果期限是0，那么到期时间为当天24点
                Calendar overTime = Calendar.getInstance();
                overTime.set(Calendar.HOUR_OF_DAY,23);
                overTime.set(Calendar.SECOND,59);
                overTime.set(Calendar.MINUTE,59);
                overTime.set(Calendar.MILLISECOND,999);
                return overTime.getTime();
            }
        }
        return getMaxDate();
    }

    @Override
    public Date getOverTime(Date beginDate, Integer timeLimit) {

        if (timeLimit != null){
            double days = timeLimit;
            if (days>0 && beginDate!=null){  //从第二天的0点开始计算工作日
                Date bDate=null;
                Calendar beginCalendar = Calendar.getInstance();
                beginCalendar.setTime(beginDate);
                beginCalendar.add(Calendar.DAY_OF_YEAR, 1);
                beginCalendar.set(Calendar.HOUR_OF_DAY,0);
                beginCalendar.set(Calendar.SECOND,0);
                beginCalendar.set(Calendar.MINUTE,0);
                beginCalendar.set(Calendar.MILLISECOND,0);
                if(IsHoliday(beginCalendar.getTime()))
                    bDate=getNextWorkDay(beginCalendar.getTime());
                else
                    bDate = beginCalendar.getTime();

                Calendar overCalendar = Calendar.getInstance();
                overCalendar.setTime(bDate);
                overCalendar.set(Calendar.HOUR_OF_DAY,23);
                overCalendar.set(Calendar.SECOND,59);
                overCalendar.set(Calendar.MINUTE,59);
                overCalendar.set(Calendar.MILLISECOND,999);

                Date overDate = overCalendar.getTime();
                for(int i=1;i<days;i++){
                    overDate=getNextWorkDay(overDate);
                }
                return overDate;
            }else if(days==0 && beginDate!=null){  //如果期限是0，那么到期时间为当天24点
                Calendar overTime = Calendar.getInstance();
                overTime.set(Calendar.HOUR_OF_DAY,23);
                overTime.set(Calendar.SECOND,59);
                overTime.set(Calendar.MINUTE,59);
                overTime.set(Calendar.MILLISECOND,999);
                return overTime.getTime();
            }
        }
        return getMaxDate();
    }

    @Override
    public double getWorkDayHours(Date beginDate, Date endDate) {
        Calendar beginCalendar = Calendar.getInstance();
        beginCalendar.setTime(beginDate);
        beginCalendar.set(Calendar.HOUR_OF_DAY,0);
        beginCalendar.set(Calendar.SECOND,0);
        beginCalendar.set(Calendar.MINUTE,0);
        beginCalendar.set(Calendar.MILLISECOND,0);

        Calendar endCalendar = Calendar.getInstance();
        endCalendar.setTime(endDate);
        endCalendar.set(Calendar.HOUR_OF_DAY,23);
        endCalendar.set(Calendar.SECOND,59);
        endCalendar.set(Calendar.MINUTE,59);
        endCalendar.set(Calendar.MILLISECOND,999);

        double result = 0;
        while(beginCalendar.before(endCalendar)){
            if(!IsHoliday(beginCalendar.getTime())) {
                result += 24;
            }
            beginCalendar.add(Calendar.DAY_OF_YEAR, 1);
        }
        return result;
    }

    @Override
    public Date getMaxDate() {
        try {
            return simpleDateFormat.parseDateTime(SYS_MAX_DATE).toDate();
        }catch (Exception ex){
            logger.error(ex.getMessage());
            return null;
        }
    }

    @Override
    public boolean IsHoliday(Date bDate) {
        if (bDate != null) {
            Calendar ca=Calendar.getInstance();
            ca.setTime(bDate);
            String dateStr= new DateTime(bDate).toString(simpleDateFormat);
            if (ca.get(Calendar.DAY_OF_WEEK)==1 || ca.get(Calendar.DAY_OF_WEEK)==7){
                return !specialWorkdays.contains(dateStr);
            }else{
                return specialHolidays.contains(dateStr);
            }
        }else {
            return false;
        }
    }

    @Override
    public Date getNextWorkDay(Date bDate) {
        Calendar ca=Calendar.getInstance();
        ca.setTime(bDate);
        ca.add(Calendar.DAY_OF_YEAR,1);
        while(IsHoliday(ca.getTime())){
            ca.add(Calendar.DAY_OF_YEAR,1);
        }
        return ca.getTime();
    }

    @Override
    public void reloadWorkdaysAndHolidays() {
        init();
    }

    @Override
    public int getTimeLeft(Date overDate) {
        Calendar overCalendar=Calendar.getInstance();
        overCalendar.setTime(overDate);
        overCalendar.set(Calendar.HOUR_OF_DAY,23);
        overCalendar.set(Calendar.SECOND,59);
        overCalendar.set(Calendar.MINUTE,59);
        overCalendar.set(Calendar.MILLISECOND, 999);
        DateTime overDateTime = new DateTime(overCalendar.getTime());
        Calendar sysCalendar = Calendar.getInstance();
        sysCalendar.set(Calendar.HOUR_OF_DAY,0);
        sysCalendar.set(Calendar.SECOND,0);
        sysCalendar.set(Calendar.MINUTE,0);
        sysCalendar.set(Calendar.MILLISECOND, 0);
        DateTime sysDateTime = new DateTime(sysCalendar.getTime());

        Period hourPeriod = new Period(sysDateTime,overDateTime, PeriodType.hours());
        Period dayPeriod = new Period(sysDateTime,overDateTime, PeriodType.days());
        int hours = hourPeriod.getHours();
        if(sysDateTime.isAfter(overDateTime)){
            int leftDays = -1;
            sysCalendar.set(Calendar.DAY_OF_YEAR,-1);
            sysCalendar.set(Calendar.HOUR_OF_DAY,23);
            sysCalendar.set(Calendar.SECOND,59);
            sysCalendar.set(Calendar.MINUTE,59);
            sysCalendar.set(Calendar.MILLISECOND, 999);
            sysDateTime = new DateTime(sysCalendar.getTime());
            for(int i=dayPeriod.getDays();i<0;i++){
                if(!IsHoliday(sysDateTime.plusDays(i).toDate()))
                    leftDays--;
            }
            return leftDays;
        }else {
            if (hours >= 0 && hours <= 24)
                return 0;
            else {
                int leftDays = 0;
                for (int i = 1; i <= dayPeriod.getDays(); i++) {
                    if (!IsHoliday(sysDateTime.plusDays(i).toDate()))
                        leftDays++;
                }
                return leftDays;
            }
        }
    }

    public void setCalendarDAO(SysCalendarDAO calendarDAO) {
        this.calendarDAO = calendarDAO;
    }

    @Override
    public List<Date> getWorkDateList(Date beginDate, Date endDate) {
        return calendarDAO.getDateListByCalType("工作日",beginDate,endDate);
    }

    @Override
    public List<Date> getHolidayList(Date beginDate, Date endDate) {
        return calendarDAO.getDateListByCalType("节假日",beginDate,endDate);
    }
}
