package com.gtis.plat.service.impl;

import com.gtis.config.AppConfig;
import com.gtis.plat.dao.SysCalendarDAO;
import com.gtis.plat.service.SysCalendarService;
import com.gtis.util.CalendarUtil;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.joda.time.DateTime;
import org.joda.time.Period;
import org.joda.time.PeriodType;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.regex.Pattern;


public class SysCalendarServiceDefaultImpl implements SysCalendarService {
    static Logger logger = LoggerFactory.getLogger(SysCalendarServiceDefaultImpl.class);

    SysCalendarDAO calendarDAO;
    static List<String> specialHolidays = new ArrayList<String>();
    static List<String> specialWorkdays = new ArrayList<String>();
    /**
     * @author <a href="mailto:shenjian@gtmap.cn">shenjian</a>
     * @version 2.0, 2018/4/20
     * @description 使用Joda-Time包替换JDK的格式化，解决SimpleDateFormat线程不安全问题
     */
    static final DateTimeFormatter simpleDateFormat = DateTimeFormat.forPattern("yyyy-MM-dd");
    static final DateTimeFormatter simpleDateFullFormat = DateTimeFormat.forPattern("yyyy-MM-dd HH:mm");
    static final DateTimeFormatter simpleDateTimeFormat = DateTimeFormat.forPattern("HH:mm");
    static final String SYS_MAX_DATE = "2999-12-31";
    String workDayAMStartTime = "08:30";
    String workDayAMEndTime = "12:00";
    String workDayPMStartTime = "13:00";
    String workDayPMEndTime = "17:30";
    double workDayHours = 8.0;   //计算每天的上班小时数

    public void setCalendarDAO(SysCalendarDAO calendarDAO) {
        this.calendarDAO = calendarDAO;
    }

    /**
     * 初始化，只取去年、今年和明年的数据,3年的数据
     */
    public void init() {

        //从配置文件读取信息
        if (StringUtils.isNotBlank(AppConfig.getProperty("WorkDayAMStartTime")))
            workDayAMStartTime = AppConfig.getProperty("WorkDayAMStartTime");
        if (StringUtils.isNotBlank(AppConfig.getProperty("WorkDayAMEndTime")))
            workDayAMEndTime = AppConfig.getProperty("WorkDayAMEndTime");
        if (StringUtils.isNotBlank(AppConfig.getProperty("WorkDayPMStartTime")))
            workDayPMStartTime = AppConfig.getProperty("WorkDayPMStartTime");
        if (StringUtils.isNotBlank(AppConfig.getProperty("WorkDayPMEndTime")))
            workDayPMEndTime = AppConfig.getProperty("WorkDayPMEndTime");

        Calendar ca = Calendar.getInstance();
        int yearInt = Calendar.getInstance().get(Calendar.YEAR);
        //Date beginDate=ca.set(Calendar.ye);
        ca.set(Calendar.YEAR, yearInt - 1);
        ca.set(Calendar.DAY_OF_YEAR, 1);//本年第一天
        Date beginDate = ca.getTime();
        ca.set(Calendar.YEAR, yearInt + 2);
        ca.set(Calendar.DAY_OF_YEAR, 1);//本年第一天
        ca.add(Calendar.DAY_OF_YEAR, -1);
        Date endDate = ca.getTime();
        List<Map> specialDates = calendarDAO.getSpecialDateList(beginDate, endDate);
        for (Map sDate : specialDates) {
            if (sDate.get("CAL_TYPE").equals("工作日"))
                specialWorkdays.add(new DateTime(sDate.get("CAL_DATE")).toString(simpleDateFormat));
            else
                specialHolidays.add(new DateTime(sDate.get("CAL_DATE")).toString(simpleDateFormat));
        }
        ///计算一天的工作小时
        try {
            workDayHours =
                    getDateDiffHour(simpleDateFullFormat.parseDateTime("2014-01-01 " + workDayAMStartTime).toDate(), simpleDateFullFormat.parseDateTime("2014-01-01 " + workDayAMEndTime).toDate())
                            + getDateDiffHour(simpleDateFullFormat.parseDateTime("2014-01-01 " + workDayPMStartTime).toDate(), simpleDateFullFormat.parseDateTime("2014-01-01 " + workDayPMEndTime).toDate());
        } catch (Exception ex) {
            logger.error(ex.getMessage());
        }

    }

    /**
     * 得到两个时间的差多少小时
     *
     * @param beginDate
     * @param endDate
     * @return
     */
    private static double getDateDiffHour(Date beginDate, Date endDate) {
        long time = endDate.getTime() - beginDate.getTime();
        return time * 1.0 / (1000 * 60 * 60);
    }

    /**
     * 判断是否非工作日(节假日)
     *
     * @param bDate
     * @return
     */
    public boolean IsHoliday(Date bDate) {
        Calendar ca = Calendar.getInstance();
        ca.setTime(bDate);
        String dateStr = new DateTime(bDate).toString(simpleDateFormat);
        if (ca.get(Calendar.DAY_OF_WEEK) == 1 || ca.get(Calendar.DAY_OF_WEEK) == 7) {
            return !specialWorkdays.contains(dateStr);
        } else {
            return specialHolidays.contains(dateStr);
        }
    }

    /**
     * 得到下一个工作日
     *
     * @param bDate
     * @return
     */
    public Date getNextWorkDay(Date bDate) {
        Calendar ca = Calendar.getInstance();
        ca.setTime(bDate);
        ca.add(Calendar.DAY_OF_YEAR, 1);
        while (IsHoliday(ca.getTime())) {
            ca.add(Calendar.DAY_OF_YEAR, 1);
        }
        try {
            ca.setTime(simpleDateFullFormat.parseDateTime(new DateTime(ca.getTime()).toString(simpleDateFormat) + " " + workDayAMStartTime).toDate());
        } catch (Exception ex) {
            logger.error(ex.getMessage());
        }
        return ca.getTime();
    }

    @Override
    public void reloadWorkdaysAndHolidays() {
        init();
    }

    /**
     * 计算超期和剩余时间（小时）：为了提高效率，按照配置每天工作时间，不查找数据库中每天工作时间，后续完善
     *
     * @param overDate
     * @return
     */
    @Override
    public int getTimeLeft(Date overDate) {
        Calendar overCalendar=Calendar.getInstance();
        overCalendar.setTime(overDate);
        overCalendar.set(Calendar.HOUR_OF_DAY,23);
        overCalendar.set(Calendar.SECOND,59);
        overCalendar.set(Calendar.MINUTE,59);
        overCalendar.set(Calendar.MILLISECOND, 999);
        DateTime overDateTime = new DateTime(overCalendar.getTime());
        Calendar sysCalendar = Calendar.getInstance();
        sysCalendar.set(Calendar.HOUR_OF_DAY,0);
        sysCalendar.set(Calendar.SECOND,0);
        sysCalendar.set(Calendar.MINUTE,0);
        sysCalendar.set(Calendar.MILLISECOND, 0);
        DateTime sysDateTime = new DateTime(sysCalendar.getTime());
        Date sysDate=new Date();
        Period hourPeriod = new Period(sysDateTime,overDateTime, PeriodType.hours());
        Period dayPeriod = new Period(sysDateTime,overDateTime, PeriodType.days());
        int hours = 0;
        if (sysDate.after(overDate)) {
            int leftDays = 0;
            sysCalendar.set(Calendar.DAY_OF_YEAR,-1);
            sysCalendar.set(Calendar.HOUR_OF_DAY,23);
            sysCalendar.set(Calendar.SECOND,59);
            sysCalendar.set(Calendar.MINUTE,59);
            sysCalendar.set(Calendar.MILLISECOND, 999);
            sysDateTime = new DateTime(sysCalendar.getTime());
            for(int i=dayPeriod.getDays();i<0;i++){
                if(!IsHoliday(sysDateTime.plusDays(i).toDate()))
                    leftDays--;
            }
            if(Math.abs(leftDays)>0){
                hours = (int) ((Math.abs(leftDays)-1) * workDayHours);
                //计算当天的已经工作时间（小时）
                if (!IsHoliday(sysDate))
                    hours = hours + ((int) (workDayHours - getWorkHours(sysDate)));
                //计算最后一天剩余工作时间（小时）
                hours = hours + getWorkHours(overDate);
            }else {
                hours = hours + getWorkHoursBetweenDate(overDate,sysDate);
            }
            return -hours;
        }else {
            int leftDays = 0;
            for (int i = 1; i <= dayPeriod.getDays(); i++) {
                if (!IsHoliday(sysDateTime.plusDays(i).toDate()))
                    leftDays++;
            }
            if(leftDays>0){
                hours = (int) ((leftDays-1) * workDayHours);
                //计算当天的剩余工作时间（小时）
                if (!IsHoliday(sysDate))
                    hours = hours + getWorkHours(sysDate);
                //计算最后一天已经工作时间（小时）
                hours = hours + ((int) (workDayHours - getWorkHours(overDate)));
            }else {
                hours = hours + getWorkHoursBetweenDate(sysDate,overDate);
            }
            return hours;
        }
    }

    @Override
    public Date getMaxDate() {
        try {
            return simpleDateFormat.parseDateTime(SYS_MAX_DATE).toDate();
        } catch (Exception ex) {
            logger.error(ex.getMessage());
            return null;
        }
    }

    /**
     * 判断是否为double类型
     *
     * @param str
     * @return
     */
    public static boolean isNumeric(String str) {
        Pattern pattern = Pattern.compile("[-+]?[0-9]+(\\.[0-9]+)?");
        return pattern.matcher(str).matches();
    }

    @Override
    public Date getOverTime(Date beginDate, String timeLimit) {
        if (StringUtils.isNotBlank(timeLimit)) {
            if (isNumeric(timeLimit)) {
                double days = Double.parseDouble(timeLimit);
                //按工作日计算
                return getOverTime(beginDate, days * workDayHours);

            } else {
                double hours = Double.parseDouble(timeLimit.substring(0, timeLimit.length() - 1));
                //按工作小时计算
                return getOverTime(beginDate, hours);
            }
        }
        return getMaxDate();
    }
    @Override
    public Date getOverTime(Date beginDate, Integer timeLimit) {
        if (timeLimit != null) {
            double days = timeLimit;
            //按工作日计算
            return getOverTime(beginDate, days * workDayHours);
        }
        return getMaxDate();
    }

    public Date getOverTime(Date beginDate, double hours) {
        Date bDate = beginDate;
        if (hours > 0 && beginDate != null) {
            if (IsHoliday(bDate))
                //判断开始时间是否不是工作日
                bDate = getNextWorkDay(beginDate);
            else {
                //判断开始时间是否工作日的非工作时间
                if (bDate.before(buildDate(bDate, workDayAMStartTime))) {
                    bDate = buildDate(bDate, workDayAMStartTime);
                } else if (bDate.after(buildDate(bDate, workDayAMEndTime)) && bDate.before(buildDate(bDate, workDayPMStartTime))) {
                    bDate = buildDate(bDate, workDayPMStartTime);
                } else if (bDate.after(buildDate(bDate, workDayPMEndTime))) {
                    bDate = getNextWorkDay(bDate);
                }
            }
            //开始计算时间
            Date overDate = bDate;
            int days = (int) Math.floor(hours / workDayHours);
            for (int i = 0; i < days; i++) {
                overDate = getNextWorkDay(overDate);
            }
            overDate = buildDate(overDate, new DateTime(bDate).toString(simpleDateTimeFormat));
            //开始计算小时
            double otherHours = hours % workDayHours;
            overDate = addWorkHoursToDate(overDate, otherHours);
            return overDate;
        }
        return getMaxDate();
    }

    @Override
    public double getWorkDayHours(Date beginDate, Date endDate) {
        double result = 0;
        Date bDate = beginDate;
        if (IsHoliday(bDate))
            //判断开始时间是否不是工作日
            bDate = getNextWorkDay(beginDate);
        else {
            //判断开始时间是否工作日的非工作时间
            if (bDate.before(buildDate(bDate, workDayAMStartTime))) {
                bDate = buildDate(bDate, workDayAMStartTime);
            } else if (bDate.after(buildDate(bDate, workDayAMEndTime)) && bDate.before(buildDate(bDate, workDayPMStartTime))) {
                bDate = buildDate(bDate, workDayPMStartTime);
            } else if (bDate.after(buildDate(bDate, workDayPMEndTime))) {
                bDate = getNextWorkDay(bDate);
            }
        }

        //计算天
        Calendar ca = Calendar.getInstance();
        int days = 0;
        while (bDate.before(endDate)) {
            ca.setTime(bDate);
            ca.add(Calendar.DAY_OF_YEAR, 1);
            if (ca.getTime().before(endDate) && !IsHoliday(ca.getTime())) {
                days++;
                bDate = ca.getTime();
            } else {
                break;
            }
        }
        return days * workDayHours + calWorkDayHoursDiff(bDate, endDate);
    }

    private double calWorkDayHoursDiff(Date beginDate, Date endDate) {
        Date bDate = beginDate;
        double result = 0;

        Date amDate = buildDate(bDate, workDayAMEndTime);
        if (bDate.before(amDate) && endDate.after(amDate)) {
            if (endDate.after(amDate)) {
                result = result + getDateDiffHour(bDate, amDate);
                return result + calWorkDayHoursDiff(buildDate(bDate, workDayPMStartTime), endDate);
            } else {
                return getDateDiffHour(bDate, endDate);
            }
        }
        Date pmDate = buildDate(bDate, workDayPMEndTime);
        if (bDate.before(pmDate)) {
            if (endDate.after(pmDate)) {
                result = result + getDateDiffHour(bDate, pmDate);
                return result + calWorkDayHoursDiff(getNextWorkDay(bDate), endDate);
            } else {
                return getDateDiffHour(bDate, endDate);
            }
        }
        return 0;
    }


    private Date addWorkHoursToDate(Date endDate, double hours) {
        Date amEndTime = buildDate(endDate, workDayAMEndTime);
        if (endDate.before(amEndTime)) {
            Date oDate = addHoursToDate(endDate, hours);
            if (oDate.before(amEndTime))
                return oDate;
            else {
                double diffHour = getDateDiffHour(endDate, amEndTime);
                return addWorkHoursToDate(buildDate(endDate, workDayPMStartTime), hours - diffHour);
            }
        }
        Date pmEndTime = buildDate(endDate, workDayPMEndTime);
        if (endDate.before(pmEndTime)) {
            Date oDate = addHoursToDate(endDate, hours);
            if (oDate.before(pmEndTime))
                return oDate;
            else {
                double diffHour = getDateDiffHour(endDate, pmEndTime);
                return addWorkHoursToDate(getNextWorkDay(endDate), hours - diffHour);
            }
        }
        return endDate;
    }

    private Date buildDate(Date cDate, String timeStr) {
        try {
            return simpleDateFullFormat.parseDateTime(new DateTime(cDate).toString(simpleDateFormat) + " " + timeStr).toDate();
        } catch (Exception ex) {
            logger.error(ex.getMessage());
            return null;
        }
    }

    private Date addHoursToDate(Date cDate, double hours) {
        int hour = (int) Math.floor(hours);
        int min = (int) Math.round((hours - hour) * 60);
        Calendar ca = Calendar.getInstance();
        ca.setTime(cDate);
        ca.add(Calendar.HOUR_OF_DAY, hour);
        ca.add(Calendar.MINUTE, min);
        return ca.getTime();
    }

    public static void main(String[] arg) throws Exception {
//        String workDayAMStartTime="08:30";
//        String workDayAMEndTime="12:00";
//        String workDayPMStartTime="13:00";
//        String workDayPMEndTime="17:30";
//        SimpleDateFormat simpleDateFullFormat=new SimpleDateFormat("yyyy-MM-dd HH:mm");
//        double dd=getDateDiffHour(simpleDateFullFormat.parse("2014-01-01 " + workDayAMEndTime),simpleDateFullFormat.parse("2014-01-01 " + workDayAMStartTime))
//                +getDateDiffHour(simpleDateFullFormat.parse("2014-01-01 " + workDayPMEndTime),simpleDateFullFormat.parse("2014-01-01 " + workDayPMStartTime));
//        System.out.println(dd);

        SysCalendarServiceDefaultImpl calendarService = new SysCalendarServiceDefaultImpl();
//        System.out.println(simpleDateFullFormat.format(calendarService.getOverTime(simpleDateFullFormat.parse("2014-10-09 04:30 "),"2")));
//        System.out.println(simpleDateFullFormat.format(calendarService.getOverTime(simpleDateFullFormat.parse("2014-10-09 08:30 "),"3")));
//        System.out.println(simpleDateFullFormat.format(calendarService.getOverTime(simpleDateFullFormat.parse("2014-10-09 09:30 "),"1")));
//        System.out.println(simpleDateFullFormat.format(calendarService.getOverTime(simpleDateFullFormat.parse("2014-10-09 04:30 "),"1h")));
//        System.out.println(simpleDateFullFormat.format(calendarService.getOverTime(simpleDateFullFormat.parse("2014-10-09 15:33 "),"6h")));
//        System.out.println(simpleDateFullFormat.format(calendarService.getOverTime(simpleDateFullFormat.parse("2014-10-09 15:33 "),"17h")));

//        System.out.println(calendarService.getWorkDateList(simpleDateFullFormat.parse("2014-10-09 04:30 "),simpleDateFullFormat.parse("2014-10-10 09:30 ")));
        System.out.println(calendarService.getTimeLeft(simpleDateFullFormat.parseDateTime("2017-10-20 10:50 ".trim()).toDate()));
    }

    @Override
    public List<Date> getWorkDateList(Date beginDate, Date endDate) {
        return calendarDAO.getDateListByCalType("工作日", beginDate, endDate);
    }

    @Override
    public List<Date> getHolidayList(Date beginDate, Date endDate) {
        return calendarDAO.getDateListByCalType("节假日", beginDate, endDate);
    }

    /**
     * 计算一天剩余工作时间
     *
     * @param date
     * @return
     */
    public int getWorkHours(Date date) {
        int hours = 0;
        try {
            Date amStartTime = simpleDateFullFormat.parseDateTime(CalendarUtil.formateDatetoStr(date) + " " + workDayAMStartTime).toDate();
            Date amEndTime = simpleDateFullFormat.parseDateTime(CalendarUtil.formateDatetoStr(date) + " " + workDayAMEndTime).toDate();
            Date pmStartTime = simpleDateFullFormat.parseDateTime(CalendarUtil.formateDatetoStr(date) + " " + workDayPMStartTime).toDate();
            Date pmEndTime = simpleDateFullFormat.parseDateTime(CalendarUtil.formateDatetoStr(date) + " " + workDayPMEndTime).toDate();
            if (date.before(amStartTime))
                hours += workDayHours;
            else if (date.after(amStartTime) && date.before(amEndTime)) {
                //上午
                Period amPeriod = new Period(date.getTime(), amEndTime.getTime(), PeriodType.hours());
                Period pmPeriod = new Period(pmStartTime.getTime(), pmEndTime.getTime(), PeriodType.hours());
                hours = hours + amPeriod.getHours() + pmPeriod.getHours();
            } else if (date.after(amEndTime) && date.before(pmStartTime)) {
                Period pmPeriod = new Period(pmStartTime.getTime(), pmEndTime.getTime(), PeriodType.hours());
                hours = hours + pmPeriod.getHours();
            } else if (date.after(pmStartTime) && date.before(pmEndTime)) {
                Period pmPeriod = new Period(date.getTime(), pmEndTime.getTime(), PeriodType.hours());
                hours = hours + pmPeriod.getHours();
            }
        } catch (Exception e) {
        }

        return hours;
    }

    /**
     * 计算一天l两个时间之间工作时间
     *
     * @param beginDate
     * @param endDate
     * @return
     */
    public int getWorkHoursBetweenDate(Date beginDate, Date endDate) {
        int hours = 0;
        try {
            Date amStartTime = simpleDateFullFormat.parseDateTime(CalendarUtil.formateDatetoStr(beginDate) + " " + workDayAMStartTime).toDate();
            Date amEndTime = simpleDateFullFormat.parseDateTime(CalendarUtil.formateDatetoStr(beginDate) + " " + workDayAMEndTime).toDate();
            Date pmStartTime = simpleDateFullFormat.parseDateTime(CalendarUtil.formateDatetoStr(beginDate) + " " + workDayPMStartTime).toDate();
            Date pmEndTime = simpleDateFullFormat.parseDateTime(CalendarUtil.formateDatetoStr(beginDate) + " " + workDayPMEndTime).toDate();
            if (beginDate.before(amStartTime) && endDate.after(amStartTime) && endDate.before(amEndTime)) {
                //开始时间<上午上班开始时间<结束时间<上午上班结束时间
                Period amPeriod = new Period(amStartTime.getTime(), amEndTime.getTime(), PeriodType.hours());
                hours = hours + amPeriod.getHours();
            } else if (beginDate.after(amStartTime) && beginDate.before(amEndTime) && endDate.after(amStartTime) && endDate.before(amEndTime)) {
                //上午上班开始时间<开始时间<结束时间<上午上班结束时间
                Period amPeriod = new Period(amStartTime.getTime(), endDate.getTime(), PeriodType.hours());
                hours = hours + amPeriod.getHours();
            } else if (beginDate.after(amStartTime) && beginDate.before(amEndTime) && endDate.after(amEndTime) && endDate.before(pmStartTime)) {
                //上午上班开始时间<开始时间<上午上班结束时间<结束时间<下午上班开始时间
                Period amPeriod = new Period(beginDate.getTime(), amEndTime.getTime(), PeriodType.hours());
                hours = hours + amPeriod.getHours();
            } else if (beginDate.after(amStartTime) && beginDate.before(amEndTime) && endDate.after(pmStartTime) && endDate.before(pmEndTime)) {
                //上午上班开始时间<开始时间<上午上班结束时间<下午上班开始时间<结束时间<下午上班结束时间
                Period amPeriod = new Period(beginDate.getTime(), amEndTime.getTime(), PeriodType.hours());
                Period pmPeriod = new Period(pmStartTime.getTime(), endDate.getTime(), PeriodType.hours());
                hours = hours + amPeriod.getHours() + pmPeriod.getHours();
            } else if (beginDate.after(amStartTime) && beginDate.before(amEndTime) && endDate.after(pmEndTime)) {
                //上午上班开始时间<开始时间<上午上班结束时间<下午上班结束时间<结束时间
                Period amPeriod = new Period(beginDate.getTime(), amEndTime.getTime(), PeriodType.hours());
                Period pmPeriod = new Period(pmStartTime.getTime(), pmEndTime.getTime(), PeriodType.hours());
                hours = hours + amPeriod.getHours() + pmPeriod.getHours();
            } else if (beginDate.after(amEndTime) && beginDate.before(pmStartTime) && endDate.after(pmStartTime) && endDate.before(pmEndTime)) {
                //上午上班结束时间<开始时间<下午上班开始时间<结束时间<下午上班结束时间
                Period pmPeriod = new Period(pmStartTime.getTime(), endDate.getTime(), PeriodType.hours());
                hours = hours + pmPeriod.getHours();
            } else if (beginDate.after(amEndTime) && beginDate.before(pmStartTime) && endDate.after(pmEndTime)) {
                //上午上班结束时间<开始时间<下午上班开始时间<下午上班结束时间<结束时间
                Period pmPeriod = new Period(pmStartTime.getTime(), pmEndTime.getTime(), PeriodType.hours());
                hours = hours + pmPeriod.getHours();
            } else if (beginDate.after(pmStartTime) && beginDate.before(pmEndTime) && endDate.after(pmStartTime) && endDate.before(pmEndTime)) {
                //开始时间<下午上班开始时间<开始时间<结束时间<下午上班结束时间
                Period pmPeriod = new Period(beginDate.getTime(), endDate.getTime(), PeriodType.hours());
                hours = hours + pmPeriod.getHours();
            } else if (beginDate.after(pmStartTime) && beginDate.before(pmEndTime) && endDate.after(pmEndTime)) {
                //<下午上班开始时间<开始时间<下午上班结束时间<结束时间
                Period pmPeriod = new Period(beginDate.getTime(), pmEndTime.getTime(), PeriodType.hours());
                hours = hours + pmPeriod.getHours();
            }
        } catch (Exception e) {
        }

        return hours;
    }
}
