package com.gtis.support.hibernate;

import com.gtis.common.Page;
import org.hibernate.Criteria;
import org.hibernate.SessionFactory;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.hibernate.metadata.ClassMetadata;
import org.springframework.util.Assert;

import java.io.Serializable;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Arrays;
import java.util.List;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-8-13
 */
@SuppressWarnings("unchecked")
public class HibernateTemplate<T, PK extends Serializable> extends HibernateDao {
// ------------------------------ FIELDS ------------------------------

    protected Class<T> entityClass;

// --------------------------- CONSTRUCTORS ---------------------------

    /**
     * 用于Dao层子类使用的构造函数.
     * 通过子类的泛型定义取得对象类型Class.
     * eg.
     * public class UserDao extends SimpleHibernateDao<User, Long>
     */
    public HibernateTemplate() {
        Class clazz;
        Type genType = getClass().getGenericSuperclass();

        if (!(genType instanceof ParameterizedType)) {
            clazz = Object.class;
        } else {
            Type[] params = ((ParameterizedType) genType).getActualTypeArguments();
            if (!(params[0] instanceof Class)) {
                clazz = Object.class;
            } else {
                clazz = (Class) params[0];
            }
        }
        this.entityClass = clazz;
    }

    public HibernateTemplate(SessionFactory sessionFactory) {
        this();
        this.sessionFactory = sessionFactory;
    }

    public HibernateTemplate(HibernateDao hibernateDao, Class entityClass) {
        this.sessionFactory = hibernateDao.getSessionFactory();
        this.entityClass = entityClass;
    }

// -------------------------- OTHER METHODS --------------------------

    /**
     * 删除对象.
     *
     * @param entity 对象必须是session中的对象或含id属性的transient对象.
     */
    public void delete(final T entity) {
        Assert.notNull(entity, "entity不能为空");
        getSession().delete(entity);
        logger.debug("delete entity: {}", entity);
    }

    /**
     * 按id删除对象.
     */
    public void delete(final PK id) {
        Assert.notNull(id, "id不能为空");
        delete(get(id));
        logger.debug("delete entity {},id is {}", entityClass.getSimpleName(), id);
    }

    /**
     * 按id load对象,支持延时加载，但是对象不存在会报错
     */
    public T load(final PK id) {
        Assert.notNull(id, "id不能为空");
        return (T) getSession().load(entityClass, id);
    }

    /**
     * 按id get对象，不支持延时加载，不存在返回null
     */
    public T get(final PK id) {
        Assert.notNull(id, "id不能为空");
        return (T) getSession().get(entityClass, id);
    }

    /**
     * 按属性查找对象列表,匹配方式为相等.
     */
    public List<T> findBy(final String propertyName, final Object value) {
        Assert.hasText(propertyName, "propertyName不能为空");
        Criterion criterion = Restrictions.eq(propertyName, value);
        return find(criterion);
    }

    /**
     * 按Criteria查询对象列表.
     *
     * @param criterions 数量可变的Criterion.
     */
    public List<T> find(final Criterion... criterions) {
        return createCriteria(criterions).list();
    }

    /**
     * 按id列表获取对象.
     */
    public List<T> findByIds(List<PK> ids) {
        return find(Restrictions.in(getIdName(), ids));
    }

    /**
     * 取得对象的主键名.
     */
    public String getIdName() {
        ClassMetadata meta = getSessionFactory().getClassMetadata(entityClass);
        return meta.getIdentifierPropertyName();
    }

    /**
     * 按Criteria查询唯一对象.
     *
     * @param criterions 数量可变的Criterion.
     */
    public T findUnique(final Criterion... criterions) {
        return (T) createCriteria(criterions).uniqueResult();
    }

    /**
     * 按属性查找唯一对象,匹配方式为相等.
     */
    public T findUniqueBy(final String propertyName, final Object value) {
        Assert.hasText(propertyName, "propertyName不能为空");
        Criterion criterion = Restrictions.eq(propertyName, value);
        return (T) createCriteria(criterion).uniqueResult();
    }

    /**
     * 获取全部对象.
     */
    public List<T> getAll() {
        return find();
    }

    /**
     * 获取全部对象,支持排序.
     */
    public List<T> getAll(String orderBy, boolean isAsc) {
        Criteria c = createCriteria();
        if (isAsc) {
            c.addOrder(Order.asc(orderBy));
        } else {
            c.addOrder(Order.desc(orderBy));
        }
        return c.list();
    }

    /**
     * 获取分页的对象,支持排序.
     */
    public Page<T> search(int start, int limit, Order[] orders, Criterion... criterions) {
        return search(Arrays.asList(criterions), Arrays.asList(orders), start, limit);
    }

    /**
     * 获取分页的对象,支持排序.
     */
    public Page<T> search(int start, int limit, Order order, Criterion... criterions) {
        return search(Arrays.asList(criterions), Arrays.asList(order), start, limit);
    }

    /**
     * 获取分页的对象,支持排序.
     */
    public Page<T> search(List<? extends Criterion> criterions, List<Order> orders, int start, int limit) {
        return search(entityClass, criterions, orders, start, limit);
    }

    /**
     * 根据Criterion条件创建Criteria.
     * <p/>
     * 本类封装的find()函数全部默认返回对象类型为T,当不为T时使用本函数.
     *
     * @param criterions 数量可变的Criterion.
     */
    public Criteria createCriteria(final Criterion... criterions) {
        Criteria criteria = getSession().createCriteria(entityClass);
        for (Criterion c : criterions) {
            criteria.add(c);
        }
        return criteria;
    }

    /**
     * 保存新增或修改的对象.
     */
    public T save(final T entity) {
        Assert.notNull(entity, "entity不能为空");
        getSession().saveOrUpdate(entity);
        logger.debug("save entity: {}", entity);
        return entity;
    }

    /**
     * 保存新增或修改的对象.
     */
    public T merge(final T entity) {
        Assert.notNull(entity, "entity不能为空");
        getSession().merge(entity);
        logger.debug("save entity: {}", entity);
        return entity;
    }
}
