package com.gtis.fileCenter.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.gtis.fileCenter.Constants;
import com.gtis.fileCenter.CustomFilePart;
import com.gtis.fileCenter.ex.NodeExistsException;
import com.gtis.fileCenter.ex.NodeNotFoundException;
import com.gtis.fileCenter.model.Downloader;
import com.gtis.fileCenter.model.Node;
import com.gtis.fileCenter.service.FileService;
import com.gtis.fileCenter.service.NodeService;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpConnectionManager;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.httpclient.methods.multipart.MultipartRequestEntity;
import org.apache.commons.httpclient.methods.multipart.Part;
import org.apache.commons.httpclient.methods.multipart.StringPart;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;
import org.springframework.util.FileCopyUtils;

import java.io.*;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 11-10-13
 */
public class FileServiceImpl implements FileService {

    private static final Logger logger = LoggerFactory.getLogger(FileServiceImpl.class);

    private NodeService nodeService;
    private String fileCenterUrl;
    private HttpClient httpClient;

    public FileServiceImpl() {
        HttpConnectionManager connectionManager = new MultiThreadedHttpConnectionManager();
        connectionManager.getParams().setDefaultMaxConnectionsPerHost(10);
        connectionManager.getParams().setConnectionTimeout(10000);
        connectionManager.getParams().setSoTimeout(10000);
        httpClient = new HttpClient(connectionManager);
    }

    public void setNodeService(NodeService nodeService) {
        this.nodeService = nodeService;
    }

    public String getFileCenterUrl() {
        return fileCenterUrl;
    }

    public void setFileCenterUrl(String fileCenterUrl) {
        this.fileCenterUrl = fileCenterUrl;
    }

    public Node uploadFile(File file, Integer parentId) throws NodeNotFoundException, NodeExistsException, IOException {
        return uploadFile(file, parentId, null);
    }

    public Node uploadFile(File file, Integer parentId, String name) throws NodeNotFoundException, NodeExistsException, IOException {
        return uploadFile(file, parentId, null, null, false, true);
    }

    public Node uploadFile(InputStream is, Integer parentId, String name) throws NodeNotFoundException, NodeExistsException, IOException {
        return uploadFile(is, parentId, name, null, false, true);
    }

    public Node uploadFile(InputStream is, Integer parentId, String name, String path, boolean cover, boolean rename) throws NodeNotFoundException, NodeExistsException, IOException {
        Assert.notNull(name);
        File tempFile = null;
        try {
            tempFile = File.createTempFile("fileCenter", null);
            FileCopyUtils.copy(is, new FileOutputStream(tempFile));
            return uploadFile(tempFile, parentId, name, null, false, true);
        } catch (IOException e) {
            logger.error("error to upload:" + name, e);
        } finally {
            if (tempFile != null) {
                tempFile.delete();
            }
        }
        return null;
    }

    public Node uploadFile(File file, Integer parentId, String name, String path, boolean cover, boolean rename) throws NodeNotFoundException, NodeExistsException, IOException {
        Node node = nodeService.getNode(parentId, path);
        PostMethod filePost = new PostMethod(fileCenterUrl + "/file/upload.do");
        Part[] parts = {
                new StringPart(Constants.NODE_ID, parentId.toString()),
                new StringPart(Constants.TOKEN, nodeService.getToken(node)),
                new StringPart("cover", Boolean.toString(cover)),
                new StringPart("rename", Boolean.toString(rename)),
                new CustomFilePart(name == null ? file.getName() : name, file)
        };
        filePost.setRequestEntity(
                new MultipartRequestEntity(parts, filePost.getParams())
        );
        HttpClient client = new HttpClient();
        client.executeMethod(filePost);
        StringWriter sw = new StringWriter();
        IOUtils.copy(filePost.getResponseBodyAsStream(), sw, "UTF-8");
        JSONObject json = JSON.parseObject(sw.toString());
        int code = json.getInteger("type");
        switch (code) {
            case 0:
                return nodeService.getNode(json.getJSONObject("file").getInteger("id"));
            case 1:
                throw new NodeNotFoundException(parentId, name);
            case 2:
                throw new NodeExistsException(parentId, name);
        }
        throw new IOException("upload file " + file.getAbsolutePath() + " to nodeId " + parentId + " error,msg:[" + json.getString("msg") + "]");
    }

    public void download(Integer nodeId, Downloader downloader) throws NodeNotFoundException, IOException {
        download(nodeId, null, downloader);
    }

    public void download(Integer nodeId, String path, Downloader downloader) throws NodeNotFoundException, IOException {
        Node node = nodeService.getNode(nodeId, path);
        StringBuilder sb = new StringBuilder(fileCenterUrl);
        sb.append("/file/get.do?fid=");
        sb.append(node.getId());
        if (node.getScope() != Node.SCOPE_DEFAULT) {
            sb.append("&token=").append(nodeService.getToken(node));
        }
        GetMethod get = new GetMethod(sb.toString());
        InputStream is = null;
        try {
            if (httpClient.executeMethod(get) != HttpStatus.SC_OK) {
                throw new NodeNotFoundException(nodeId, "run http get error");
            }
            is = get.getResponseBodyAsStream();
            downloader.download(is, node);
        } catch (IOException e) {
            throw new NodeNotFoundException(nodeId, "run http get error");
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException ignored) {
                }
            }
            get.releaseConnection();
        }
    }

    public void downloadToFile(Integer nodeId, String savePath) throws NodeNotFoundException, IOException {
        downloadToFile(nodeId, null, savePath);
    }

    public void downloadToFile(final Integer nodeId, final String path, final String savePath) throws NodeNotFoundException, IOException {
        download(nodeId, path, new Downloader() {
            public void download(InputStream is, Node node) {
                File file = new File(savePath);
                File pFile = file.getParentFile();
                if (!pFile.exists()) {
                    pFile.mkdirs();
                }
                try {
                    FileCopyUtils.copy(is, new BufferedOutputStream(new FileOutputStream(file)));
                } catch (IOException e) {
                    throw new NodeNotFoundException(nodeId, "save to path:" + savePath + " error");
                }
            }
        });
    }
}
