/*
 * Author: xyang
 *
 * Project: fileCenter
 *
 * File: AwtImageTransform.java
 *
 * LastModified: 2009-09-24 10:41:27
 *
 * Copyright (c) 2009 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.generic.image.impl;

import com.gtis.generic.image.ImageTransform;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.awt.image.ImagingOpException;
import java.io.File;
import java.io.IOException;

/**
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 */
public class AwtImageTransform implements ImageTransform {

    private static final Logger logger = LoggerFactory.getLogger(AwtImageTransform.class);
    private BufferedImage srcBImage;
    private BufferedImage destBImage;
    private int imgWidth;
    private int imgHeight;

    public boolean load(String srcFilePath) {
        try {
            srcBImage = ImageIO.read(new java.io.File(srcFilePath));
            imgWidth = srcBImage.getWidth();
            imgHeight = srcBImage.getHeight();
        } catch (IOException e) {
            logger.info("Cannot open image [{}]", srcFilePath);
            return false;
        }
        return true;
    }

    public boolean save(String destFilePath) {
        if (destBImage == null)
            return false;
        try {
            String postfix = destFilePath.substring(destFilePath.lastIndexOf(".") + 1);
            ImageIO.write(destBImage, postfix, new File(destFilePath));
        } catch (IOException e) {
            logger.info("Cannot open image [{}]", destFilePath);
            return false;
        }
        return true;
    }

    public void resize(int width, int height) {
        double xRatio = (double) height / imgHeight;
        double yRatio = (double) width / imgWidth;
        double ratio = (xRatio < yRatio) ? xRatio : yRatio;
        zoomImage(imgWidth, imgHeight, ratio);
    }

    public void rotate(double rotate) {
        rotateImage(imgWidth, imgHeight, rotate, 0);
    }

    public void resizeWithMaxWidth(int maxWidth) {
        if (imgWidth > maxWidth) {
            double ratio = (double) maxWidth / imgWidth;
            zoomImage(imgWidth, imgHeight, ratio);
        } else
            destBImage = srcBImage;
    }

    public void rotateWithMaxWidth(double rotate, int maxWidth) {
        rotateImage(imgWidth, imgHeight, rotate, maxWidth);
    }

    private void zoomImage(int width, int height, double ratio) {
        try {
            int w = (int) (width * ratio);
            int h = (int) (height * ratio);
            destBImage = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
            Graphics2D g = destBImage.createGraphics();
            g.setColor(Color.white);
            g.fillRect(0, 0, w, h);
            destBImage.getGraphics().drawImage(srcBImage.getScaledInstance(w, h, Image.SCALE_SMOOTH), 0, 0, null);

        } catch (Exception e) {
            throw new ImagingOpException("Unable to transform src image");
        }
    }

    private void rotateImage(int width, int height, double rotate, int maxWidth) {
        double radian = rotate * Math.PI / 180;  //get radian 
        double h = height * Math.cos(radian) + width * Math.sin(radian);  //get rotated width
        double w = height * Math.sin(radian) + width * Math.cos(radian);  //get rotated height
        double ratio = maxWidth != 0 && w > maxWidth ? maxWidth / w : 1;  //if maxWidth=0 do not zoom
        AffineTransform transform = new AffineTransform();
        transform.setToScale(ratio, ratio);
        transform.rotate(radian, w / 2, h / 2);
        transform.translate(w / 2 - width / 2, h / 2 - height / 2);
        AffineTransformOp ato = new AffineTransformOp(transform, null);
        w *= ratio;
        h *= ratio;
        try {
            destBImage = new BufferedImage((int) w, (int) h, srcBImage.getType());
            Graphics gs = destBImage.getGraphics();
            gs.setColor(Color.white); //set canvas background to white
            gs.fillRect(0, 0, (int) w, (int) h);
            ato.filter(srcBImage, destBImage);
        } catch (Exception e) {
            throw new ImagingOpException("Unable to transform src image");
        }
    }
}
