package com.gtis.search;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.*;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 12-1-11
 */
public abstract class EntityIndexProvider<T> implements IndexProvider {
    protected final Logger logger = LoggerFactory.getLogger(getClass());
    protected Map<String, BusinessFactory> businessFactories = new HashMap<String, BusinessFactory>();
    protected Business defaultBusiness;

    public void setBusinessFactories(List<BusinessFactory> businessFactories) {
        for (BusinessFactory bizf : businessFactories) {
            this.businessFactories.put(bizf.getId(), bizf);
            if (defaultBusiness == null) {
                defaultBusiness = bizf.getBusiness();
            }
        }
    }

    public boolean ping() {
        return true;
    }

    public List<Business> getBusinesses() {
        List<Business> list = new ArrayList<Business>(businessFactories.size());
        for (BusinessFactory biz : businessFactories.values()) {
            list.add(biz.getBusiness());
        }
        return list;
    }

    public List<Index> getIndexes(String categoryId, int start, int size) {
        List<Index> list = new ArrayList<Index>();
        for (T entity : getEntities(categoryId, start, size)) {
            list.add(toIndex(entity));
        }
        return list;
    }

    public Index getIndex(String categoryId, String id) {
        T entity = getEntity(categoryId, id);
        if (entity == null)
            return null;
        return toIndex(entity);
    }

    public Index toIndex(T entity) {
        Index index = createIndex(entity);
        if (index != null) {
            extractEntity(entity, index);
            prepareIndex(index);
            if (index.getDate() == null) {
                index.setDate(new Date());
            }
        }
        return index;
    }

    protected Index prepareIndex(Index index) {
        BusinessFactory bf = businessFactories.get(index.getBusinessId());
        if (bf != null) {
            CategoryFactory cf = bf.getCategoryFactory(index.getCategoryId());
            if (cf != null) {
                List<DataFetcher> dataFetchers = cf.getDataFetchers();
                if (dataFetchers != null) {
                    for (DataFetcher dataFetcher : dataFetchers) {
                        try {
                            for (Map.Entry<String, Serializable> entry : dataFetcher.fetchData(index.getId()).entrySet()) {
                                String key = entry.getKey();
                                if (cf.getUnSearchableFields().contains(key)) {
                                    index.addField(key, entry.getValue());
                                } else {
                                    index.addSearchableField(key, entry.getValue());
                                }
                            }
                        } catch (RuntimeException e) {
                            logger.error("fetch data from form " + dataFetcher.getDescription() + " error", e);
                        }
                    }
                }
            }
        }
        return index;
    }

    public Category getDefaultCategory() {
        return defaultBusiness.getCategories().get(0);
    }

    public Business getDefaultBusiness() {
        return defaultBusiness;
    }

    protected Index createIndex(T entity) {
        return new Index();
    }

    protected abstract T getEntity(String categoryId, String id);

    protected abstract void extractEntity(T entity, Index index);

    protected abstract List<T> getEntities(String categoryId, int start, int size);
}
