/*
 * Author: xyang
 *
 * Project: trunk
 *
 * File: SystemConfig.java
 *
 * LastModified: 2010-01-26 12:44:25
 *
 * Copyright (c) 2010 gtis. All Rights Reserved.
 *
 * Copying of this document or code and giving it to others and the
 * use or communication of the contents thereof, are forbidden without
 * expressed authority. Offenders are liable to the payment of damages.
 * All rights reserved in the event of the grant of a invention patent or the
 * registration of a utility model, design or code.
 *
 * Issued by gtis Ltd.
 */

package com.gtis.egov.generic;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;

import java.util.Map;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-1-26
 */
public final class SystemConfig implements InitializingBean {
    private static Logger log = LoggerFactory.getLogger(SystemConfig.class);
    private String[] necessaryConfigs;

    private static Map systemProperties;

    public void setConfiguration(final Map properties) {
        systemProperties = properties;
    }

    public static String getProperty(final String key) {
        return (String) systemProperties.get(key);
    }

    /**
     * 获取配置文件中的参数值.
     *
     * @param key 参数名称.
     * @return 参数值.
     */
    public static int getIntProperty(final String key) {
        return getIntProperty(key, 0);
    }

    /**
     * 获取配置文件中的参数值.
     *
     * @param key          参数名称.
     * @param defaultValue 默认值.
     * @return 参数值.
     */
    public static int getIntProperty(final String key, int defaultValue) {
        String property = getProperty(key);
        int value = defaultValue;

        try {
            value = Integer.parseInt(property);
        } catch (NumberFormatException e) {
            log.warn(e.toString());
        }

        return value;
    }

    /**
     * 获取配置文件中的整型参数值.
     *
     * @param key 参数名称.
     * @return 整型参数值.
     */
    public static Integer getIntegerProperty(final String key) {
        return getIntegerProperty(key, null);
    }

    /**
     * 获取配置文件中的整型参数值.
     *
     * @param key          参数名称.
     * @param defaultValue 缺省值.
     * @return 整型参数值.
     */
    public static Integer getIntegerProperty(final String key, final Integer defaultValue) {
        String stringValue = getProperty(key);
        Integer value = defaultValue;

        try {
            value = Integer.valueOf(stringValue);
        } catch (NumberFormatException e) {
            log.warn(e.toString());
        }

        return value;
    }

    /**
     * 获取配置文件中的布尔参数值.
     *
     * @param key 参数名称.
     * @return 布尔参数值.
     */
    public static boolean getBooleanProperty(final String key) {
        return getBooleanProperty(key, false);
    }

    /**
     * 获取配置文件中的布尔参数.如果文件中没有该参数就返回defaultValue.
     *
     * @param key          参数名称.
     * @param defaultValue 参数默认值.
     * @return 布尔参数值.
     */
    public static boolean getBooleanProperty(final String key, final boolean defaultValue) {
        String value = getProperty(key);

        if (null != value) {
            return "true".equalsIgnoreCase(value) || "on".equalsIgnoreCase(value)
                    || "yes".equalsIgnoreCase(value)
                    || "1".equalsIgnoreCase(value);
        } else {
            return defaultValue;
        }
    }

    public static Map getProperties() {
        return systemProperties;
    }

    /**
     * 检查系统必须的配置参数是否设置
     *
     * @throws Exception
     */
    public void afterPropertiesSet() throws Exception {
        // 检查某些关键的配置顶是否存在，不存在就报初始化错误
        final String[] keys = this.necessaryConfigs;
        if (keys == null)
            return;
        for (int i = 0, n = keys.length; i < n; i++) {
            String key = StringUtils.trimToEmpty(keys[i]);
            if (!systemProperties.containsKey(key)) {
                throw new IllegalStateException("Can not find property \"" + key + "\" in configuration file.");
            }
        }
    }

    public void setNecessaryConfigs(String[] necessaryConfigs) {
        this.necessaryConfigs = necessaryConfigs;
    }
}
