package com.gtis.egov.calendar.web;


import com.gtis.egov.calendar.model.Calendar;
import com.gtis.egov.calendar.model.Event;
import com.gtis.egov.calendar.model.PersonalCalendar;
import com.gtis.egov.calendar.model.Reminder;
import com.gtis.egov.calendar.service.CalendarExchangeManager;
import com.gtis.egov.calendar.service.CalendarManager;
import com.gtis.egov.calendar.service.EventManager;
import com.gtis.egov.calendar.service.ReminderManager;
import com.gtis.egov.calendar.utils.DateUtils;
import com.gtis.egov.generic.SystemConfig;
import com.gtis.egov.generic.User;
import com.gtis.plat.service.SysUserService;
import com.gtis.plat.vo.PfOrganVo;
import com.sun.syndication.feed.synd.*;
import com.sun.syndication.io.SyndFeedOutput;
import com.wiscom.generic.base.util.FileWebInput;
import com.wiscom.generic.base.util.JsonUtils;
import com.wiscom.generic.base.util.WebInput;
import com.wiscom.generic.base.util.WebOutput;
import net.fortuna.ical4j.model.Recur;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.io.InputStream;
import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 * @version V1.0, 2009-5-18 13:38:09
 */
public class CalendarController {

    /**
     * Module name.
     */
    public static final String moduleName = "calendar";

    /**
     * Logger.
     */
    public static final String DATE_PATTERN = "yyyy-MM-dd HH:mm";
    private static final SimpleDateFormat tsdf = new SimpleDateFormat(DATE_PATTERN);
    public static final String DAY_PATTERN = "yyyy-MM-dd";

    public static final String MODE_DAY = "day";
    public static final String MODE_MONTH = "month";
    public static final String MODE_LIST = "list";
    private static final String IS_MANAGER = "is_calendar_manager";
    private static final String USER_ID = "___userId___";
    private static final String GROUP_ID = "___groupId___";

    private static final Logger logger = LoggerFactory.getLogger(CalendarController.class);

    public boolean isInit = false;
    public boolean isSingleMethod = false;
    private CalendarManager calendarManager;
    private EventManager eventManager;
    private ReminderManager reminderManager;
    private SysUserService sysUserService;

    /**
     * 初始化Service,由系统自动调用.
     *
     * @param calendarManager calendarManager service
     * @param eventManager    eventManager service
     * @param reminderManager reminderManager
     * @param sysUserService  sysUserService
     */
    public void initObject(CalendarManager calendarManager, EventManager eventManager, ReminderManager reminderManager, SysUserService sysUserService) {
        this.calendarManager = calendarManager;
        this.eventManager = eventManager;
        this.reminderManager = reminderManager;
        this.sysUserService = sysUserService;
    }

    public ModelAndView standardAdmin(HttpSession session) {
        if (session.getAttribute(IS_MANAGER) == null)
            session.setAttribute(IS_MANAGER, Boolean.TRUE);
        return new ModelAndView("redirect:admin.do");
    }

    public ModelAndView groupAdmin(User loginUser, WebInput in, HttpSession session) {
        if (session.getAttribute(IS_MANAGER) == null)
            session.setAttribute(IS_MANAGER, Boolean.TRUE);
        String groupId = in.getString("groupId");
        if (StringUtils.isNotBlank(groupId)) {
            initGroupCalendar(sysUserService.getOrganVo(groupId), loginUser.getId());
            return new ModelAndView("redirect:group.do?" + GROUP_ID + "=" + groupId);
        }
        Map params = new HashMap();
        params.put("groups", sysUserService.getAllOrganList());
        return new ModelAndView("groupAdmin", params);
    }

    public ModelAndView initGroupCalendars(User loginUser) {
        int count = 0;
        List groups = sysUserService.getAllOrganList();
        for (int i = 0; i < groups.size(); i++) {
            PfOrganVo group = (PfOrganVo) groups.get(i);
            if (initGroupCalendar(group, loginUser.getId()))
                count++;
        }
        return new ModelAndView("redirect:groupAdmin.do" + (count == 0 ? "" : "?count=" + count));
    }

    private boolean initGroupCalendar(PfOrganVo group, String userId) {
        List calendars = calendarManager.getGroupCalendars(group.getOrganId());
        if (calendars.isEmpty()) {
            Calendar calendar = new Calendar();
            calendar.setCreator(userId);
            calendar.setCreateTime(new Date());
            calendar.setName(group.getOrganName());
            calendar.setType(Calendar.GROUP);
            calendar.setOwner(group.getOrganId());
            calendarManager.saveCalendar(calendar);
            return true;
        }
        return false;
    }

    public ModelAndView personalView(WebInput in, HttpSession session) {
        if (session.getAttribute(IS_MANAGER) == null)
            session.setAttribute(IS_MANAGER, Boolean.TRUE);
        String groupId = in.getString("groupId");
        String userId = in.getString("userId");
        if (StringUtils.isNotBlank(userId)) {
            return new ModelAndView("redirect:index.do?" + USER_ID + "=" + userId);
        }
        Map params = new HashMap();
        params.put("groups", sysUserService.getAllOrganList());
        if (StringUtils.isNotBlank(groupId)) {
            params.put("users", sysUserService.getUserListByOragn(groupId));
        }
        return new ModelAndView("personalView", params);

    }

    /**
     * 显示首页
     *
     * @param loginUser user
     * @param in        web in
     * @param webout    web out
     * @return 首页
     */
    public ModelAndView index(User loginUser, WebInput in, WebOutput webout) {
        Map params = new HashMap();
        String userId = in.getString(USER_ID);
        if (StringUtils.isBlank(userId))
            userId = loginUser.getId();
        List personalCalendars = calendarManager.getPersonalCalendars(userId, Boolean.TRUE);

        boolean jsonMethod = in.getBoolean("__json", false);
        if (jsonMethod) {
            outputEvents(userId, personalCalendars, in, webout, true);
            return null;
        }

        boolean isGetCalMethod = in.getBoolean("__cal", false);
        if (isGetCalMethod) {
            try {
                webout.toJson(personalCalendars);
            } catch (IOException e) {
                // ignore.
            }
            return null;
        }

        params.put("calendars", JsonUtils.toString(personalCalendars));
        params.put("personalCalendars", personalCalendars);
        params.put("isManager", Boolean.valueOf(isManager(in)));
        params.put("viewOnly", Boolean.valueOf(StringUtils.isBlank(userId)));
        return new ModelAndView("index", params);
    }

    /**
     * 返回rss消息
     *
     * @param loginUser user
     * @param in        web in
     * @param webout    web out
     * @return rss
     */
    public ModelAndView rss(User loginUser, WebInput in, WebOutput webout) {
        List personalCalendars = calendarManager.getPersonalCalendars(loginUser.getId(), Boolean.TRUE);
        Date start = new Date();
        Date end = DateUtils.addDays(start, in.getInt("day", 7));
        Long[] ids = new Long[personalCalendars.size()];
        for (int i = 0; i < personalCalendars.size(); i++) {
            PersonalCalendar personalCalendar = (PersonalCalendar) personalCalendars.get(i);
            if (personalCalendar.isSelected() && personalCalendar.isVisible())
                ids[i] = personalCalendar.getCalendarId();
        }
        List events = eventManager.getEvents(loginUser.getId(), ids, start, end);
        Collections.sort(events);
        SyndFeed feed = buildFeed(events, loginUser.getId());
        webout.setContentType("text/xml", "utf-8");
        SyndFeedOutput output = new SyndFeedOutput();
        try {
            output.output(feed, webout.getResponse().getWriter());
        } catch (Exception e) {
            logger.error("Could not write the rss", e);
        }
        return null;
    }

    private SyndFeed buildFeed(List events, String userId) {
        SyndFeed feed = new SyndFeedImpl();
        feed.setFeedType("rss_2.0");
        feed.setAuthor(userId);
        feed.setTitle(SystemConfig.getProperty("calendar.feed.name"));
        feed.setDescription("");
        feed.setLink(SystemConfig.getProperty("calendar.url"));
        List entries = new ArrayList();
        for (int i = 0; i < events.size(); i++) {
            Event event = (Event) events.get(i);
            SyndEntry entry = new SyndEntryImpl();
            entry.setLink(SystemConfig.getProperty("calendar.url"));
            entry.setTitle(event.getTitle());
            SyndContent description = new SyndContentImpl();
            description.setType("text/html");
            description.setValue(getContent(event));
            entry.setDescription(description);
            entry.setPublishedDate(event.getStartTime());
            entries.add(entry);
        }
        feed.setEntries(entries);
        return feed;
    }

    private String getContent(Event event) {
        return MessageFormat.format(SystemConfig.getProperty("calendar.remind.content.rss"),
                new Object[]{
                        event.isWholeDay() ? tsdf.format(event.getStartTime()) : tsdf.format(event.getStartTime()) + " to " + tsdf.format(event.getEndTime()),
                        StringUtils.trimToEmpty(event.getDescription()),
                        StringUtils.trimToEmpty(event.getLocation()),
                        StringUtils.trimToEmpty(event.getUrl()),
                        event.getColor()
                });
    }

    /**
     * 显示群组日历
     *
     * @param loginUser user
     * @param in        web in
     * @param webout    web out
     * @return 群组日历页面
     */
    public ModelAndView group(User loginUser, WebInput in, WebOutput webout) {
        if (!isManager(in))
            return new ModelAndView("redirect:index.do");
        Map params = new HashMap();
        String groupId = in.getString(GROUP_ID);

        List calendars = calendarManager.getGroupCalendars(groupId);
        if (calendars == null || calendars.size() == 0)
            throw new RuntimeException("group not found");
        List groupCalendars = calendarManager.getGroupPersonalCalendars(loginUser.getId(), groupId);

        boolean changed = false;
        for (int i = 0; i < calendars.size(); i++) {
            Calendar calendar = (Calendar) calendars.get(i);
            boolean found = false;
            for (int j = 0; j < groupCalendars.size(); j++) {
                PersonalCalendar pCal = (PersonalCalendar) groupCalendars.get(j);
                if (pCal.getCalendarId().equals(calendar.getId())) {
                    found = true;
                    break;
                }
            }
            if (!found) {
                PersonalCalendar personalCalendar = new PersonalCalendar();
                personalCalendar.setCalendarId(calendar.getId());
                personalCalendar.setName(calendar.getName());
                personalCalendar.setCreator(loginUser.getId());
                personalCalendar.setAddTime(new Date());
                personalCalendar.setColor(calendar.getColor());
                calendarManager.savePersonalCalendar(personalCalendar);
                changed = true;
            }
        }
        if (changed)
            groupCalendars = calendarManager.getGroupPersonalCalendars(loginUser.getId(), groupId);

        boolean jsonMethod = in.getBoolean("__json", false);
        if (jsonMethod) {
            outputEvents(loginUser.getId(), groupCalendars, in, webout, true);
            return null;
        }

        boolean isGetCalMethod = in.getBoolean("__cal", false);
        if (isGetCalMethod) {
            try {
                webout.toJson(groupCalendars);
            } catch (IOException e) {
                // ignore.
            }
            return null;
        }

        params.put("calendars", JsonUtils.toString(groupCalendars));
        params.put("groupCalendars", groupCalendars);
        params.put("isManager", Boolean.valueOf(isManager(in)));
        return new ModelAndView("group", params);
    }

    /**
     * 管理标准日历
     *
     * @param loginUser user
     * @param in        web in
     * @param webout    web out
     * @return 首页
     */
    public ModelAndView admin(User loginUser, WebInput in, WebOutput webout) {
        if (!isManager(in))
            return new ModelAndView("redirect:index.do");
        Map params = new HashMap();

        List standardCalendars = calendarManager.getStandardPersonalCalendars(loginUser.getId());

        boolean jsonMethod = in.getBoolean("__json", false);
        if (jsonMethod) {
            outputEvents(loginUser.getId(), standardCalendars, in, webout, true);
            return null;
        }


        boolean isGetCalMethod = in.getBoolean("__cal", false);
        if (isGetCalMethod) {
            try {
                webout.toJson(standardCalendars);
            } catch (IOException e) {
                // ignore.
            }
            return null;
        }

        params.put("calendars", JsonUtils.toString(standardCalendars));
        params.put("standardCalendars", standardCalendars);
        params.put("isManager", Boolean.valueOf(isManager(in)));
        return new ModelAndView("admin", params);
    }

    /**
     * 管理日历列表
     *
     * @param loginUser 登陆用户
     * @param in        webin
     * @param webout    webout
     * @return 管理页面
     */
    public ModelAndView manage(User loginUser, WebInput in, WebOutput webout) {
        Map params = new HashMap();
        List personalCalendars = calendarManager.getPersonalCalendars(loginUser.getId(), null);
        boolean isGetCalMethod = in.getBoolean("__cal", false);
        if (isGetCalMethod) {
            try {
                webout.toJson(personalCalendars);
            } catch (IOException e) {
                // ignore.
            }
            return null;
        }
        params.put("calendars", JsonUtils.toString(personalCalendars));
        params.put("personalCalendars", personalCalendars);
        params.put("isManager", Boolean.valueOf(isManager(in)));
        return new ModelAndView("manage", params);
    }

    /**
     * 添加一个公用日历
     *
     * @param loginUser 登陆用户
     * @param in        webin
     * @return 跳转到列表页
     */
    public ModelAndView add(User loginUser, WebInput in) {
        Long calendarId = in.getLong("id");
        String backUrl = getBackUrl(in).replaceAll("&msg=\\w+", "");
        String msg;
        backUrl = backUrl + (backUrl.indexOf("?") > -1 ? "&" : "?");
        PersonalCalendar personalCalendar = calendarManager.getPersonalCalendar(calendarId, loginUser.getId());
        if (personalCalendar == null) {
            Calendar calendar = calendarManager.getCalendar(calendarId);
            personalCalendar = new PersonalCalendar();
            personalCalendar.setCalendarId(calendar.getId());
            personalCalendar.setName(calendar.getName());
            personalCalendar.setCreator(loginUser.getId());
            personalCalendar.setAddTime(new Date());
            personalCalendar.setColor(calendar.getColor());
            calendarManager.savePersonalCalendar(personalCalendar);
            msg = "msg=success";
        } else
            msg = "msg=exist";
        return new ModelAndView(backUrl + msg);
    }

    /**
     * 日历创建或编辑
     *
     * @param loginUser user
     * @param in        web input
     * @param webout    web output
     * @return 跳转到前一页
     */
    public ModelAndView editCalendar(User loginUser, WebInput in, WebOutput webout) {
        Long calendarId = in.getLong("calendarId");
        String owner = in.getString("owner");
        String name = in.getString("name");
        String description = in.getString("description");
        String color = in.getString("color");
        short type = (short) in.getInt("type", Calendar.PRIVATE);
        Date now = new Date();
        String userId = loginUser.getId();

        if (StringUtils.isBlank(name) || StringUtils.isBlank(color))
            return null;


        PersonalCalendar personalCalendar;
        Calendar calendar;
        if (calendarId == null) {
            calendar = new Calendar();
            calendar.setCreator(userId);
            calendar.setCreateTime(now);
            calendar.setName(name);
            calendar.setDescription(description);
            calendar.setColor(color);
            calendar.setType(type);
            if (owner != null)
                calendar.setOwner(owner);
            calendarManager.saveCalendar(calendar);
            personalCalendar = new PersonalCalendar();
            personalCalendar.setCreator(userId);
            personalCalendar.setAddTime(now);
            personalCalendar.setCalendarId(calendar.getId());
        } else {
            personalCalendar = calendarManager.getPersonalCalendar(calendarId);
            calendar = calendarManager.getCalendar(personalCalendar.getCalendarId());
            if (personalCalendar.getCalendarCreator().equals(loginUser.getId()) || isManager(in) || hasManagePermission(calendar, loginUser.getId())) {
                calendar.setName(name);
                calendar.setDescription(description);
                calendar.setColor(color);
                calendar.setType(type);
                if (owner != null)
                    calendar.setOwner(owner);
                calendarManager.saveCalendar(calendar);
            }
        }

        if (personalCalendar.getCreator().equals(loginUser.getId())) {
            personalCalendar.setName(name);
            personalCalendar.setColor(color);
            calendarManager.savePersonalCalendar(personalCalendar);
        }

        Map params = new HashMap();
        params.put("status", "1");
        try {
            webout.toJson(params);
        } catch (IOException e) {
            // ignore.
        }
        return null;
    }

    /**
     * 编辑事件
     *
     * @param loginUser user
     * @param in        web input
     * @param webout    web output
     * @return 跳转到前一页
     */
    public ModelAndView editEvent(User loginUser, WebInput in, WebOutput webout) {
        Long calendarId = in.getLong("calendarId");
        Date startTime = in.getDate("startTime", DATE_PATTERN, null);
        Date endTime = in.getDate("endTime", DATE_PATTERN, null);
        Date repeatEndTime = in.getDate("repeatEndTime", DATE_PATTERN, null);
        Long eventId = in.getLong("eventId");
        String recurrenceRule = getRecurrenceRule(in);
        String title = in.getString("title");

        if (calendarId == null || StringUtils.isBlank(title) || (startTime == null && StringUtils.isBlank(recurrenceRule)))
            return null;

        Event event = eventId == null ? new Event() : eventManager.getEvent(eventId);
        Calendar calendar = calendarManager.getCalendar(calendarId);
        if (eventId == null || event.getCreator().equals(loginUser.getId()) || isManager(in) || hasManagePermission(calendar, loginUser.getId())) {
            event.setCalendarId(calendarId);
            event.setTitle(in.getString("title"));
            event.setUrl(in.getString("url"));
            event.setLocation(in.getString("location"));
            event.setDescription(in.getString("description"));
            event.setStartTime(startTime);
            event.setEndTime(endTime);
            event.setUpdateTime(new Date());
            event.setWholeDay(in.getBoolean("wholeDay"));
            event.setCreator(loginUser.getId());
            if (recurrenceRule != null) {
                event.setRecurrenceRule(recurrenceRule);
                event.setRepeatEndTime(repeatEndTime);
            }
            eventManager.saveEvent(event);
            saveReminders(loginUser, event.getId(), in);
            Map params = new HashMap();
            params.put("status", "1");
            try {
                webout.toJson(params);
            } catch (IOException e) {
                // ignore.
            }
        }
        return null;
    }

    /**
     * 保存提醒
     *
     * @param loginUser 登陆用户
     * @param eventId   eventId
     * @param in        webin
     */
    public void saveReminders(User loginUser, Long eventId, WebInput in) {
        String[] ids = in.getStrings("id");
        List reminders = reminderManager.getReminders(eventId);
        for (int i = 0; i < reminders.size(); i++) {
            Reminder reminder = (Reminder) reminders.get(i);
            Long id = reminder.getId();
            if (isIdExist(id, ids)) {
                short method = (short) in.getInt("method" + id, 0);
                int unit = in.getInt("unit" + id, 0);
                int number = in.getInt("number" + id, 0);
                reminder.setMethod(method);
                reminder.setSeconds(unit * number);
                reminderManager.saveReminder(reminder);
            } else {
                reminderManager.removeReminder(id);
            }
        }
        if (ids != null)
            for (int i = 0; i < ids.length; i++) {
                String id = ids[i];
                try {
                    Long.valueOf(id);
                } catch (NumberFormatException e) {
                    short method = (short) in.getInt("method" + id, 0);
                    int unit = in.getInt("unit" + id, 0);
                    int number = in.getInt("number" + id, 0);
                    Reminder reminder = new Reminder();
                    reminder.setMethod(method);
                    reminder.setCreator(loginUser.getId());
                    reminder.setEventId(eventId);
                    reminder.setSeconds(unit * number);
                    if (reminders.indexOf(reminder) == -1) {//如果存在相同remind则不保存
                        reminders.add(reminder);
                        reminderManager.saveReminder(reminder);
                    }
                }
            }
    }

    /**
     * 判断id是否存在
     *
     * @param id  id
     * @param ids id字符串列表
     * @return 是否存在
     */
    private static boolean isIdExist(Long id, String[] ids) {
        if (ids != null)
            for (int i = 0; i < ids.length; i++) {
                if (ids[i].equals(String.valueOf(id)))
                    return true;
            }
        return false;
    }

    /**
     * 删除事件
     *
     * @param loginUser 登陆用户
     * @param in        webin
     * @param webout    webout
     * @return 状态
     */
    public ModelAndView delEvent(User loginUser, WebInput in, WebOutput webout) {
        Long eventId = in.getLong("id");
        Event event = eventManager.getEvent(eventId);
        if (event.getCreator().equals(loginUser.getId())) {
            eventManager.removeEvent(eventId);
        }
        Map params = new HashMap();
        params.put("status", "1");
        try {
            webout.toJson(params);
        } catch (IOException e) {
            // ignore.
        }
        return null;
    }

    /**
     * @param loginUser 登陆用户
     * @param in        webin
     * @param webout    webout
     * @return 状态
     */
    public ModelAndView delCal(User loginUser, WebInput in, WebOutput webout) {
        Map params = new HashMap();
        Long calendarId = in.getLong("id");

        PersonalCalendar personalCalendar = calendarManager.getPersonalCalendar(calendarId);
        Calendar calendar = calendarManager.getCalendar(personalCalendar.getCalendarId());

        if (personalCalendar.getCalendarCreator().equals(loginUser.getId()) || isManager(in) || hasManagePermission(calendar, loginUser.getId())) {
            if (logger.isDebugEnabled())
                logger.debug("remove calendar success,personalCalendar is ({}) ", personalCalendar);
            calendarManager.removeCalendar(personalCalendar.getCalendarId());
        }
        if (personalCalendar.getCreator().equals(loginUser.getId()))
            calendarManager.removePersonalCalendar(calendarId);
        params.put("status", "1");
        try {
            webout.toJson(params);
        } catch (IOException e) {
            // ignore.
        }
        return null;
    }

    /**
     * 切换个人日历状态
     *
     * @param loginUser 登陆用户
     * @param in        登陆用户
     * @param webout    webout
     * @return 状态
     */
    public ModelAndView toggleCalendar(User loginUser, WebInput in, WebOutput webout) {
        Map params = new HashMap();
        Long id = in.getLong("id");
        String type = in.getString("type");
        boolean checked = in.getBoolean("checked");
        if ("selected".equals(type))
            calendarManager.changePersonalCalendarSelectedStatus(loginUser.getId(), new Long[]{id}, checked);
        else if ("visible".equals(type))
            calendarManager.changePersonalCalendarVisibleStatus(loginUser.getId(), new Long[]{id}, checked);
        params.put("status", "1");
        try {
            webout.toJson(params);
        } catch (IOException e) {
            // ignore.
        }
        return null;
    }

    /**
     * 选中某个日历，去掉其他日历的选择
     *
     * @param loginUser 登陆用户
     * @param in        webin
     * @param webout    webout
     * @return 状态
     */
    public ModelAndView selectCalendar(User loginUser, WebInput in, WebOutput webout) {
        Map params = new HashMap();
        Long calendarId = in.getLong("id");
        Long[] oIds = in.getLongObjects("oId");
        calendarManager.changePersonalCalendarSelectedStatus(loginUser.getId(), new Long[]{calendarId}, true);
        calendarManager.changePersonalCalendarSelectedStatus(loginUser.getId(), oIds, false);
        params.put("status", "1");
        try {
            webout.toJson(params);
        } catch (IOException e) {
            // ignore.
        }
        return null;
    }

    /**
     * 导入日历
     *
     * @param loginUser               登陆用户
     * @param in                      webin
     * @param fin                     文件输入
     * @param calendarExchangeManager 日历导入导出服务
     * @return 显示导入结果
     */
    public ModelAndView importCalendar(User loginUser, WebInput in, FileWebInput fin, CalendarExchangeManager calendarExchangeManager) {
        String backUrl = getBackUrl(in).replaceAll("[?&]count=\\w+", "");
        String msg = "";
        MultipartFile mfile = fin.getMultipartFile("file");
        Long calendarId = in.getLong("calendarId");

        Calendar calendar = calendarManager.getCalendar(calendarId);

        if ((calendar.getCreator().equals(loginUser.getId()) || isManager(in) || hasManagePermission(calendar, loginUser.getId())) && mfile.getSize() > 0) {
            InputStream is = null;
            try {
                is = mfile.getInputStream();
                int count = calendarExchangeManager.importIcsCalendarEvents(is, calendarId);
                msg = (backUrl.indexOf("?") > -1 ? "&" : "?") + "count=" + count;
            } catch (IOException e) {
                logger.error("import calendar {} failed!", mfile.getOriginalFilename());
            } finally {
                IOUtils.closeQuietly(is);
            }
        }
        return new ModelAndView(backUrl + msg);
    }

    /**
     * 导出日历
     *
     * @param in                      webin
     * @param webout                  webout
     * @param calendarExchangeManager 日历导入导出服务
     * @return 下载导出文件
     * @throws Exception exception
     */
    public ModelAndView exportCalendar(WebInput in, WebOutput webout, CalendarExchangeManager calendarExchangeManager) throws Exception {
        Long id = in.getLong("id");
        PersonalCalendar personalCalendar = calendarManager.getPersonalCalendar(id);
        String fname = personalCalendar.getName() + ".ics";
        HttpServletResponse res = webout.getResponse();
        webout.setContentType("text/calendar", "UTF-8");
        res.setHeader("Content-Disposition", "attachment; filename=\"" + new String(fname.getBytes("gb2312"), "iso8859-1") + "\"");
        try {
            calendarExchangeManager.exportIcsCalendarEvents(res.getOutputStream(), personalCalendar.getCalendarId());
        } catch (IOException e) {
            logger.error("export calendar {} failed!", fname);
        }
        return null;
    }

    /**
     * 获取某个事件的提醒
     *
     * @param in     webin
     * @param webout webout
     * @return 提醒列表json数据
     */
    public ModelAndView reminders(WebInput in, WebOutput webout) {
        Long eventId = in.getLong("eventId");
        try {
            webout.toJson(reminderManager.getReminders(eventId));
        } catch (IOException e) {
            // ignore.
        }
        return null;
    }

    /**
     * 输入事件json数据
     *
     * @param userId            用户id
     * @param personalCalendars 用户个人日历列表
     * @param in                webin
     * @param webout            webout
     * @param selectedOnly      是否只显示已选择
     */
    private void outputEvents(String userId, List personalCalendars, WebInput in, WebOutput webout, boolean selectedOnly) {
        Date start = in.getDate("__s", DAY_PATTERN, null);
        Date end = in.getDate("__e", DAY_PATTERN, null);
        if (start == null || end == null || personalCalendars == null || personalCalendars.size() == 0)
            return;
        Long[] ids = new Long[personalCalendars.size()];
        for (int i = 0; i < personalCalendars.size(); i++) {
            PersonalCalendar personalCalendar = (PersonalCalendar) personalCalendars.get(i);
            if (!selectedOnly || personalCalendar.isSelected() && personalCalendar.isVisible())
                ids[i] = personalCalendar.getCalendarId();
        }
        try {
            webout.toJson(eventManager.getEvents(userId, ids, start, end));
        } catch (IOException e) {
            // ignore.
        }
    }

    /**
     * 检查是否拥有对日历的管理权限
     *
     * @param calendar 基本日历
     * @param userId   用户Id
     * @return 是否拥有权限
     */
    private boolean hasManagePermission(Calendar calendar, String userId) {
        /*if (calendar.getType() == Calendar.STANDARD || calendar.getType() == Calendar.PROTECTED) {
            return true;
        } else if (calendar.getType() == Calendar.GROUP) {
            return true;
        }*/
        return false;
    }

    private boolean isManager(WebInput in) {
        Boolean isManager = (Boolean) in.getRequest().getSession().getAttribute(IS_MANAGER);
        return isManager != null && isManager.booleanValue();
    }

    /**
     * 获取日历重复规则
     *
     * @param in webin
     * @return 日历重复规则字符串
     */
    private static String getRecurrenceRule(WebInput in) {
        String interval = in.getString("interval");
        String recurType = in.getString("recurType");
        if (StringUtils.isNotBlank(recurType)) {
            StringBuffer sb = new StringBuffer();
            if (!"NONE".equals(recurType)) {
                sb.append("FREQ=").append(recurType).append(";WKST=SU;INTERVAL=").append(interval);
                if (Recur.MONTHLY.equals(recurType)) {
                    Date startTime = in.getDate("startTime", DATE_PATTERN, null);
                    java.util.Calendar cl = java.util.Calendar.getInstance();
                    cl.setTime(startTime);
                    sb.append(";BYDAY=SU,MO,TU,WE,TH,FR,SA;BYSETPOS=").append(cl.get(java.util.Calendar.DAY_OF_MONTH));
                } else if (Recur.WEEKLY.equals(recurType)) {
                    String[] byday = in.getStrings("byday");
                    if (byday == null)
                        return null;
                    sb.append(";BYDAY=").append(StringUtils.join(byday, ","));
                }
            }
            return sb.toString();
        }
        return null;
    }

    /**
     * 获取返回地址
     *
     * @param in webin
     * @return 返回地址
     */
    private static String getBackUrl(WebInput in) {
        return "redirect:" + in.getString("__bUrl");
    }
}
