package com.gtis.egov.calendar.utils;

import org.apache.commons.lang.StringUtils;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

/**
 * 日期类型转换工具类.
 */
public final class DateUtils extends org.apache.commons.lang.time.DateUtils {

    /**
     * Default date formate pattern.
     */
    public static final String defaultPattern = "yyyy-MM-dd HH:mm:ss";

    public static final String defaultPatternWithZone = "yyyy-MM-dd HH:mm:ss z";

    /**
     * Short pattern of date.
     */
    public static final String shortPattern = "yyyy-MM-dd";

    public static final String TIME_PATTERN = "yyyy-MM-dd HH:mm";


    /**
     * 将长时间格式字符串转换为时间 yyyy-MM-dd HH:mm:ss
     *
     * @param strDate Date in string.
     * @return Date.
     */
    public static Date strToDateLong(String strDate) {
        return parse(strDate, defaultPattern);
    }

    /**
     * 将短时间格式时间转换为字符串, yyyy-MM-dd
     *
     * @param date Date.
     * @return Date in string.
     */
    public static String dateToStr(java.util.Date date) {
        return formatDate(date, shortPattern);
    }

    /**
     * 将短时间格式字符串转换为时间, yyyy-MM-dd
     *
     * @param strDate Date in string.
     * @return Date.
     */
    public static Date parseShortDate(String strDate) {
        return parse(strDate, shortPattern);
    }

    /**
     * Parse date from string.
     *
     * @param dateString Date string.
     * @param pattern    Pattern.
     * @return Date.
     */
    public static Date parse(final String dateString, final String pattern) {
        if (StringUtils.isBlank(dateString)) {
            return null;
        }
        String pat = pattern;
        if (StringUtils.isBlank(pat)) {
            pat = defaultPattern;
        }
        try {
            return new SimpleDateFormat(pat).parse(dateString);
        } catch (ParseException e) {
            // ignore.
            return null;
        }
    }

    /**
     * Format date with given pattern.
     *
     * @param date    Date to be formated.
     * @param pattern Pattern.
     * @return Date in proper format.
     */
    public static String formatDate(Date date, String pattern) {
        if (StringUtils.isBlank(pattern)) {
            pattern = defaultPattern;
        }

        return new SimpleDateFormat(pattern).format(date);
    }

    public static String formatDate(Date date, String pattern, String timeZone ) {
        if (StringUtils.isBlank(pattern)) {
            pattern = defaultPatternWithZone;
        }

        if(StringUtils.isBlank(timeZone))
        {
            timeZone = "GMT";
        }

        SimpleDateFormat sdf = new SimpleDateFormat(pattern);
        sdf.setTimeZone(TimeZone.getTimeZone(timeZone));

        return sdf.format(date);
    }

    /**
     * Format date with given pattern.
     *
     * @param date Date to be formated.
     * @return Date in proper format.
     */
    public static String formatDate(Date date) {
        return formatDate(date, defaultPattern);
    }

    /**
     * Format date.
     * <p/>
     * 如果在一天之内发生，显示xx小时xx分钟xx秒前
     * 其它显示长日期，yyyy-MM-dd hh:mm
     *
     * @param date Date to be format.
     * @return Display string.
     */
    public static String formatResentDate(final Date date) {
        final long dif = Math.abs(date.getTime() - System.currentTimeMillis());
        String ret;

        // 24小时之外
        if (dif > DateUtils.MILLIS_PER_DAY) {
            ret = formatDate(date);
        }

        // 24小时之内
        else if (dif > 3600000) {
            ret = String.valueOf(dif / (60 * 60 * 1000)) + "小时前";
        }
        // 1小时之内
        else if (dif > 60000) {
            ret = String.valueOf(dif / (60 * 1000)) + "分钟前";
        }
        // 1分钟内
        else /*if (dif > 1000)*/ {
            ret = String.valueOf((int) (dif / 1000)) + "秒前";
        }

        return ret;
    }

    /**
     * 得到两个日期的间隔
     *
     * @param now     现在时间
     * @param endDate 结束时间
     * @return String
     */
    public static String getDifferenceDate(Date now, Date endDate) {
        final long dif = (endDate.getTime() - now.getTime());
        long times = 24 * 60 * 60 * 1000;
        String ret;
        if (dif <= 0) {
            return "-1";
        }

        // 24小时之外
        if (dif > times) {
            long days = dif / times;
            if (dif % times != 0L) {
                days += 1;
                ret = "不到" + days + "天";
            } else {
                ret = days + "天";
            }

        }

        // 24小时之内
        else if (dif > 3600000) {
            ret = String.valueOf(dif / (60 * 60 * 1000)) + "小时";
        }
        // 1小时之内
        else if (dif > 60000) {
            ret = String.valueOf(dif / (60 * 1000)) + "分钟";
        }
        // 1分钟内
        else /*if (dif > 1000)*/ {
            ret = String.valueOf((int) (dif / 1000)) + "秒";
        }

        return ret;
    }

    /**
     *  两个日期之间相隔天数的共通.
     *
     * @param nowDate
     * @param endDate
     * @return
     */
    public static String getDaysBetweenTwoDates(Date nowDate, Date endDate) {
        return getDaysBetweenTwoDates(formatDate(nowDate),formatDate(endDate));
    }

    /**
     * 两个日期之间相隔天数的共通.
     *
     * @param nowDate 開始時間
     * @param endDate   　終了時間
     * @return　天数
     */
    public static String getDaysBetweenTwoDates(String nowDate, String endDate) {
        Date dtFrom;
        Date dtEnd;
        dtFrom = parse(nowDate, shortPattern);
        dtEnd = parse(endDate, shortPattern);
        long begin = dtFrom.getTime();
        long end = dtEnd.getTime();
        long inter = end - begin;
        if (inter < 0) {
            inter = inter * (-1);
        }
        long dateMillSec = 24 * 60 * 60 * 1000;

        long dateCnt = inter / dateMillSec;

        long remainder = inter % dateMillSec;

        if (remainder != 0) {
            dateCnt++;
        }
        return String.valueOf(dateCnt);
    }

    public static Date getEndOfDay(Date day) {
        return getEndOfDay(day, Calendar.getInstance());
    }

    public static Date getEndOfDay(Date day, Calendar cal) {
        if (day == null) day = new Date();
        cal.setTime(day);
        cal.set(Calendar.HOUR_OF_DAY, cal.getMaximum(Calendar.HOUR_OF_DAY));
        cal.set(Calendar.MINUTE, cal.getMaximum(Calendar.MINUTE));
        cal.set(Calendar.SECOND, cal.getMaximum(Calendar.SECOND));
        cal.set(Calendar.MILLISECOND, cal.getMaximum(Calendar.MILLISECOND));
        return cal.getTime();
    }

    public static Date getEndOfMonth(Date day) {
        return getEndOfMonth(day, Calendar.getInstance());
    }

    public static Date getStartOfMonth(Date day) {
        return getStartOfMonth(day, Calendar.getInstance());
    }

    private static Date getStartOfMonth(Date day, Calendar cal) {
        if (day == null) day = new Date();
        cal.setTime(day);

        // set time to start of day
        cal.set(Calendar.HOUR_OF_DAY, cal.getMinimum(Calendar.HOUR_OF_DAY));
        cal.set(Calendar.MINUTE, cal.getMinimum(Calendar.MINUTE));
        cal.set(Calendar.SECOND, cal.getMinimum(Calendar.SECOND));
        cal.set(Calendar.MILLISECOND, cal.getMinimum(Calendar.MILLISECOND));

        // set time to first day of month
        cal.set(Calendar.DAY_OF_MONTH, 1);

        return cal.getTime();
    }

    public static Date getEndOfMonth(Date day, Calendar cal) {
        if (day == null) day = new Date();
        cal.setTime(day);

        // set time to end of day
        cal.set(Calendar.HOUR_OF_DAY, cal.getMaximum(Calendar.HOUR_OF_DAY));
        cal.set(Calendar.MINUTE, cal.getMaximum(Calendar.MINUTE));
        cal.set(Calendar.SECOND, cal.getMaximum(Calendar.SECOND));
        cal.set(Calendar.MILLISECOND, cal.getMaximum(Calendar.MILLISECOND));

        // set time to first day of month
        cal.set(Calendar.DAY_OF_MONTH, 1);

        // add one month
        cal.add(Calendar.MONTH, 1);

        // back up one day
        cal.add(Calendar.DAY_OF_MONTH, -1);

        return cal.getTime();
    }

    public static Date getStartOfDay(Date day) {
        return getStartOfDay(day, Calendar.getInstance());
    }

    public static Date getStartOfDay(Date day, Calendar cal) {
        if (day == null) day = new Date();
        cal.setTime(day);
        cal.set(Calendar.HOUR_OF_DAY, cal.getMinimum(Calendar.HOUR_OF_DAY));
        cal.set(Calendar.MINUTE, cal.getMinimum(Calendar.MINUTE));
        cal.set(Calendar.SECOND, cal.getMinimum(Calendar.SECOND));
        cal.set(Calendar.MILLISECOND, cal.getMinimum(Calendar.MILLISECOND));
        return cal.getTime();
    }

    public static Date getNoonOfDay(Date day, Calendar cal) {
        if (day == null) day = new Date();
        cal.setTime(day);
        cal.set(Calendar.HOUR_OF_DAY, 12);
        cal.set(Calendar.MINUTE, cal.getMinimum(Calendar.MINUTE));
        cal.set(Calendar.SECOND, cal.getMinimum(Calendar.SECOND));
        cal.set(Calendar.MILLISECOND, cal.getMinimum(Calendar.MILLISECOND));
        return cal.getTime();
    }
}