package com.gtis.egov.calendar.service.impl;

import com.gtis.egov.calendar.Helper;
import com.gtis.egov.calendar.dao.ReminderDAO;
import com.gtis.egov.calendar.model.Event;
import com.gtis.egov.calendar.model.PersonalCalendar;
import com.gtis.egov.calendar.model.Reminder;
import com.gtis.egov.calendar.service.CalendarManager;
import com.gtis.egov.calendar.service.EventManager;
import com.gtis.egov.calendar.service.ReminderManager;
import com.gtis.egov.calendar.utils.DateUtils;
import com.gtis.egov.calendar.utils.StringUtils;
import com.gtis.egov.generic.Message;
import com.gtis.egov.generic.MessageService;
import com.gtis.egov.generic.SystemConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.MessageFormat;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 * @version V1.0, 2009-5-14 20:21:29
 */
public class ReminderManagerImpl implements ReminderManager {

    private static final Logger logger = LoggerFactory.getLogger(ReminderManagerImpl.class);

    private CalendarManager calendarManager;
    private EventManager eventManager;
    private ReminderDAO reminderDAO;
    private MessageService messageService;
    private Set remindersQueue = new TreeSet();

    private static final SimpleDateFormat tsdf = new SimpleDateFormat("yyyy-MM-dd HH:mm");
    private int remindInterval;


    public void setCalendarManager(CalendarManager calendarManager) {
        this.calendarManager = calendarManager;
    }

    public void setEventManager(EventManager eventManager) {
        this.eventManager = eventManager;
    }

    public void setReminderDAO(ReminderDAO reminderDAO) {
        this.reminderDAO = reminderDAO;
    }

    public void setMessageService(MessageService messageService) {
        this.messageService = messageService;
    }

    public void setRemindInterval(int remindInterval) {
        this.remindInterval = remindInterval;
    }

    /**
     * @param reminder 提醒
     * @return 提醒
     */
    public Reminder saveReminder(Reminder reminder) {
        reminderDAO.saveReminder(reminder);
        Event event = reminder.getEvent();
        if (event == null) {
            event = eventManager.getEvent(reminder.getEventId());
            reminder.setEvent(event);
        }
        if (event.getColor() == null) {
            event.setColor(calendarManager.getCalendar(event.getCalendarId()).getColor());
        }
        Date now = new Date();
        Date end = DateUtils.addSeconds(now, remindInterval);
        reminder = parseReminder(reminder, now, end);
        if (reminder != null && reminder.getRemindDate().after(now) && reminder.getRemindDate().before(end))
            remindersQueue.add(reminder);
        return reminder;
    }

    /**
     * 获取提醒
     *
     * @param reminderId 提醒Id
     * @return 提醒
     */
    public Reminder getReminder(Long reminderId) {
        return reminderDAO.getReminder(reminderId);
    }

    /**
     * 获取事件的所有提醒
     *
     * @param eventId 事件Id
     * @return 提醒列表
     */
    public List getReminders(Long eventId) {
        return reminderDAO.getReminders(eventId);
    }

    /**
     * 获取时间段内需要处理的提醒
     *
     * @param startDate startDate
     * @param endDate   endDate
     * @return 提醒列表
     */
    public List getReminders(Date startDate, Date endDate) {
        List baseReminders = reminderDAO.getBaseReminders(startDate, endDate);
        List reminders = new ArrayList();
        for (int i = 0; i < baseReminders.size(); i++) {
            Reminder reminder = parseReminder((Reminder) baseReminders.get(i), startDate, endDate);
            if (reminder != null)
                reminders.add(reminder);
        }
        return reminders;
    }

    private Reminder parseReminder(Reminder reminder, Date startDate, Date endDate) {
        if (StringUtils.isNotBlank(reminder.getEvent().getRecurrenceRule()))
            return Helper.getNearRecurrenceEventReminder(reminder, startDate, endDate);
        return reminder;
    }

    /**
     * 获取时间段内到指定时间点需要处理的提醒列表
     *
     * @param date date
     * @return 提醒列表
     */
    public List getNeedProcessRemindersFromQueue(Date date) {
        List list = new ArrayList();
        Iterator it = remindersQueue.iterator();
        while (it.hasNext()) {
            Reminder reminder = (Reminder) it.next();
            if (reminder.getRemindDate().before(date)) {
                Reminder r = getReminder(reminder.getId());
                r.setEvent(eventManager.getEvent(reminder.getEventId()));
                if (reminder.equals(r)) { //判断reminder有没有发生改变，如果改变了，则删除提醒
                    list.add(reminder);
                } else {
                    it.remove();
                }
            } else
                break;
        }
        return list;
    }

    /**
     * 更新时间段内需要处理的提醒队列
     *
     * @param startDate startDate
     * @param endDate   endDate
     */
    public void updateRemindersQueue(Date startDate, Date endDate) {
        remindersQueue.addAll(getReminders(startDate, endDate));
    }

    /**
     * 从队列中删除指定reminder
     *
     * @param reminder reminder
     */
    public void removeReminderFromQueue(Reminder reminder) {
        remindersQueue.remove(reminder);
    }

    /**
     * @param reminderId 提醒Id
     */
    public void removeReminder(Long reminderId) {
        reminderDAO.removeReminder(reminderId);
    }

    /**
     * 发送消息提醒
     *
     * @param reminder 提醒
     * @return 是否成功
     */
    public boolean sendReminder(Reminder reminder) {
        try {
            List personalCalendars = calendarManager.getPersonalCalendars(reminder.getEvent().getCalendarId());
            String[] receivers = new String[personalCalendars.size()];
            for (int i = 0; i < personalCalendars.size(); i++) {
                receivers[i] = ((PersonalCalendar) personalCalendars.get(i)).getCreator();
            }
            Message message = new Message();
            message.setReceivers(receivers);
            message.setSender(SystemConfig.getProperty("calendar.remind.sender"));
            message.setCreateTime(new Date());
            message.setSubject(getSubject(reminder));
            message.setContent(getContent(reminder, reminder.getMethod() == MessageService.SMS, calendarManager.getCalendar(reminder.getEvent().getCalendarId()).getName()));
            logger.debug("send reminder:[{}]", reminder);
            switch (reminder.getMethod()) {
                case MessageService.MESSAGE:
                    return messageService.sendMessage(message);
                case MessageService.EMAIL:
                    return messageService.sendMail(message);
                case MessageService.SMS:
                    return messageService.sendSms(message);
            }
        } catch (Exception e) {
            logger.debug("send reminder error:[{}]", e);
        }
        return true;
    }

    private String getSubject(Reminder reminder) {
        Event event = reminder.getEvent();
        return MessageFormat.format(SystemConfig.getProperty("calendar.remind.subject"), new Object[]{
                event.getTitle(),
                event.isWholeDay() ? tsdf.format(event.getStartTime()) : tsdf.format(event.getStartTime()) + " to " + tsdf.format(event.getEndTime()),
                StringUtils.trimToEmpty(event.getLocation())
        });
    }

    private String getContent(Reminder reminder, boolean isSms, String calendarName) {
        Event event = reminder.getEvent();
        return MessageFormat.format(SystemConfig.getProperty(isSms ? "calendar.remind.content.sms" : "calendar.remind.content"),
                new Object[]{
                        SystemConfig.getProperty("calendar.url"),
                        event.getTitle(),
                        event.isWholeDay() ? tsdf.format(event.getStartTime()) : tsdf.format(event.getStartTime()) + " to " + tsdf.format(event.getEndTime()),
                        StringUtils.trimToEmpty(event.getDescription()),
                        StringUtils.trimToEmpty(event.getLocation()),
                        StringUtils.trimToEmpty(event.getUrl()),
                        calendarName,
                        event.getColor()
                });
    }
}
