package com.gtis.egov.calendar.service.impl;

import com.gtis.egov.calendar.Helper;
import com.gtis.egov.calendar.dao.EventDAO;
import com.gtis.egov.calendar.model.Event;
import com.gtis.egov.calendar.model.Reminder;
import com.gtis.egov.calendar.service.EventManager;
import com.gtis.egov.calendar.service.ReminderManager;
import com.gtis.egov.calendar.utils.StringUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * 事件管理接口实现
 * <p/>
 *
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 * @version V1.0, 2009-5-14 18:36:36
 */
public class EventManagerImpl implements EventManager {

    private EventDAO eventDAO;
    private ReminderManager reminderManager;

    public void setEventDAO(EventDAO eventDAO) {
        this.eventDAO = eventDAO;
    }

    public void setReminderManager(ReminderManager reminderManager) {
        this.reminderManager = reminderManager;
    }

    /**
     * 保存事件
     *
     * @param event 事件
     * @return 事件
     */
    public Event saveEvent(Event event) {
        eventDAO.saveEvent(event);
        List reminders = reminderManager.getReminders(event.getId());
        for (int i = 0; i < reminders.size(); i++) {
            Reminder reminder = (Reminder) reminders.get(i);
            reminder.setEvent(event);
            reminderManager.saveReminder(reminder);
        }
        return event;
    }

    /**
     * 获取事件
     *
     * @param eventId 事件Id
     * @return 事件
     */
    public Event getEvent(Long eventId) {
        return eventDAO.getEvent(eventId);
    }

    /**
     * 获取时间段内的事件
     *
     * @param userId      用户Id
     * @param calendarIds 日历Ids
     * @param startDate   开始时间
     * @param endDate     结束时间    @return 事件列表
     */
    public List getEvents(String userId, Long[] calendarIds, Date startDate, Date endDate) {
        return convertEvents(eventDAO.getBaseEvents(userId, calendarIds, startDate, endDate), startDate, endDate);
    }

    /**
     * 获取用户时间段内所有事件
     *
     * @param userId    用户Id
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @return 事件列表
     */
    public List getAllEvents(String userId, Date startDate, Date endDate) {
        return convertEvents(eventDAO.getUserBaseEvents(userId, startDate, endDate), startDate, endDate);
    }

    /**
     * 删除事件
     *
     * @param eventId 事件Id
     */
    public void removeEvent(Long eventId) {
        eventDAO.removeEvent(eventId);
        reminderManager.removeReminder(eventId);
    }

    /**
     * 处理基本事件，将循环事件展开
     *
     * @param baseEvents 基本事件列表
     * @param startDate  开始时间
     * @param endDate    结束时间
     * @return 事件列表
     */
    private static List convertEvents(List baseEvents, Date startDate, Date endDate) {
        List events = new ArrayList();
        for (int i = 0; i < baseEvents.size(); i++) {
            Event baseEvent = (Event) baseEvents.get(i);
            if (StringUtils.isBlank(baseEvent.getRecurrenceRule()))//如果是重复事件，需要先计算重复生成的事件
                events.add(baseEvent);
            else
                events.addAll(Helper.getRecurrenceEvents(baseEvent, startDate, endDate));
        }
        return events;
    }
}
