package com.gtis.egov.calendar.service.impl;

import com.gtis.egov.calendar.Color;
import com.gtis.egov.calendar.dao.CalendarDAO;
import com.gtis.egov.calendar.dao.EventDAO;
import com.gtis.egov.calendar.dao.ReminderDAO;
import com.gtis.egov.calendar.model.Calendar;
import com.gtis.egov.calendar.model.PersonalCalendar;
import com.gtis.egov.calendar.service.CalendarManager;
import com.gtis.egov.generic.SystemConfig;
import com.gtis.egov.generic.User;
import com.gtis.egov.generic.UserService;
import com.wiscom.generic.base.util.PaginationSupport;
import net.sf.ehcache.Ehcache;

import java.util.Date;
import java.util.List;

/**
 * 日历管理实现
 * <p/>
 *
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 * @version V1.0, 2009-5-14 13:52:53
 */
public class CalendarManagerImpl implements CalendarManager {

    private CalendarDAO calendarDAO;
    private EventDAO eventDAO;
    private ReminderDAO reminderDAO;
    private UserService userService;
    private Ehcache userNameCache;

    public void setCalendarDAO(CalendarDAO calendarDAO) {
        this.calendarDAO = calendarDAO;
    }


    public void setEventDAO(EventDAO eventDAO) {
        this.eventDAO = eventDAO;
    }

    public void setReminderDAO(ReminderDAO reminderDAO) {
        this.reminderDAO = reminderDAO;
    }

    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    /**
     * 保存基本日历
     *
     * @param calendar 日历
     * @return 日历
     */
    public Calendar saveCalendar(Calendar calendar) {
        calendarDAO.saveCalendar(calendar);
        return calendar;
    }

    /**
     * 获取基本日历对象
     *
     * @param calendarId 日历Id
     * @return 日历
     */
    public Calendar getCalendar(Long calendarId) {
        return calendarDAO.getCalendar(calendarId);
    }

    /**
     * 检索基本日历
     *
     * @param term  检索关键字 可为空
     * @param types 日历类型 -1为所有
     * @param start 起始序号
     * @param num   每页记录数
     * @return 分页对象
     */
    public PaginationSupport searchCalendars(String term, Short[] types, int start, int num) {
        return calendarDAO.searchCalendars(term, types, start, num);
    }

    /**
     * 检索好友可见基本日历
     *
     * @param term   检索关键字 可为空
     * @param userId 用户Id
     * @param start  起始序号
     * @param num    每页记录数
     * @return 分页对象
     */
    public PaginationSupport searchFriendCalendars(String term, String userId, int start, int num) {
        return calendarDAO.searchFriendCalendars(term, userId, start, num);
    }

    /**
     * 删除基本日历
     *
     * @param calendarId 日历Id
     */
    public void removeCalendar(Long calendarId) {
        calendarDAO.removeCalendar(calendarId);
        calendarDAO.removePersonalCalendars(calendarId);
        eventDAO.removeEvents(calendarId);
        reminderDAO.removeRemindersByCalendarId(calendarId);
    }

    /**
     * 保存个人日历
     *
     * @param personalCalendar 个人日历
     * @return 个人日历
     */
    public PersonalCalendar savePersonalCalendar(PersonalCalendar personalCalendar) {
        return calendarDAO.savePersonalCalendar(personalCalendar);
    }

    /**
     * @param personalCalendarId 个人日历Id
     * @return 个人日历
     */
    public PersonalCalendar getPersonalCalendar(Long personalCalendarId) {
        return calendarDAO.getPersonalCalendar(personalCalendarId);
    }

    /**
     * 根据基本日历id和用户名获取日历
     *
     * @param calendarId 日历Id
     * @param userId     用户Id
     * @return 个人日历
     */
    public PersonalCalendar getPersonalCalendar(Long calendarId, String userId) {
        return calendarDAO.getPersonalCalendar(calendarId, userId);
    }

    /**
     * 获取日历相关的个人日历列表
     *
     * @param calendarId 日历id
     * @return 个人日历列表
     */
    public List getPersonalCalendars(Long calendarId) {
        return calendarDAO.getPersonalCalendars(calendarId);
    }

    /**
     * 同步群组基本日历到个人日历
     *
     * @param userId 用户Id
     */
    public void syncGroupCalendarToPersonalCalendar(String userId) {
        List groupCalendars = calendarDAO.getAsynchronousGroupCalendars(userId);
        for (int i = 0; i < groupCalendars.size(); i++) {
            Calendar calendar = (Calendar) groupCalendars.get(i);
            PersonalCalendar personalCalendar = new PersonalCalendar();
            personalCalendar.setCalendarId(calendar.getId());
            personalCalendar.setName(calendar.getName());
            personalCalendar.setCreator(userId);
            personalCalendar.setAddTime(new Date());
            personalCalendar.setColor(calendar.getColor());
            calendarDAO.savePersonalCalendar(personalCalendar);
        }
    }

    /**
     * 同步标准基本日历到个人日历
     *
     * @param userId 用户Id
     */
    public void syncStandardCalendarToPersonalCalendar(String userId) {
        List standardCalendars = calendarDAO.getAsynchronousStandardCalendars(userId);
        for (int i = 0; i < standardCalendars.size(); i++) {
            Calendar calendar = (Calendar) standardCalendars.get(i);
            PersonalCalendar personalCalendar = new PersonalCalendar();
            personalCalendar.setCalendarId(calendar.getId());
            personalCalendar.setName(calendar.getName());
            personalCalendar.setCreator(userId);
            personalCalendar.setAddTime(new Date());
            personalCalendar.setColor(calendar.getColor());
            calendarDAO.savePersonalCalendar(personalCalendar);
        }
    }

    /**
     * @param userId    用户Id
     * @param isVisible 是否可见 null为全部
     * @return 个人日历列表
     */
    public List getPersonalCalendars(String userId, Boolean isVisible) {
        syncGroupCalendarToPersonalCalendar(userId);
        syncStandardCalendarToPersonalCalendar(userId);
        List personalCalendars = calendarDAO.getPersonalCalendars(userId, Calendar.PRIMARY);
        if (personalCalendars.isEmpty()) {
            Calendar calendar = new Calendar();
            Date now = new Date();
            String name = SystemConfig.getProperty("calendar.defaultPersonalCalendarName");
            calendar.setCreator(userId);
            calendar.setCreateTime(now);
            calendar.setName(name);
            calendar.setColor(Color.RGB668CB3);
            calendar.setType(Calendar.PRIMARY);
            saveCalendar(calendar);
            PersonalCalendar personalCalendar = new PersonalCalendar();

            personalCalendar.setCreator(userId);
            personalCalendar.setAddTime(now);
            personalCalendar.setCalendarId(calendar.getId());
            personalCalendar.setName(name);
            personalCalendar.setColor(Color.RGB668CB3);
            savePersonalCalendar(personalCalendar);
        }
        personalCalendars = calendarDAO.getPersonalCalendars(userId, isVisible);
        upadteCreatorName(personalCalendars);
        return personalCalendars;
    }

    /**
     * 获取标准日历
     *
     * @param userId 用户Id
     * @return 标准日历列表
     */
    public List getStandardPersonalCalendars(String userId) {
        syncStandardCalendarToPersonalCalendar(userId);
        List personalCalendars = calendarDAO.getStandardPersonalCalendars(userId);
        upadteCreatorName(personalCalendars);
        return personalCalendars;
    }

    /**
     * 获取群组基本日历
     *
     * @param groupId 群组Id
     * @return 群组日历
     */
    public List getGroupCalendars(String groupId) {
        return calendarDAO.getGroupCalendars(groupId);
    }

    /**
     * 获取群组日历
     *
     * @param userId  用户Id
     * @param groupId 群组Id
     * @return 群组日历列表
     */
    public List getGroupPersonalCalendars(String userId, String groupId) {
        syncGroupCalendarToPersonalCalendar(userId);
        List personalCalendars = calendarDAO.getGroupPersonalCalendars(userId, groupId);
        upadteCreatorName(personalCalendars);
        return personalCalendars;
    }

    /**
     * 删除个人日历
     *
     * @param personalCalendarId 个人日历Id
     */
    public void removePersonalCalendar(Long personalCalendarId) {
        calendarDAO.removePersonalCalendar(personalCalendarId);
    }

    /**
     * @param userId              用户Id
     * @param personalCalendarIds 个人日历Id数组
     * @param isVisible           是否可见
     */
    public void changePersonalCalendarVisibleStatus(String userId, Long[] personalCalendarIds, boolean isVisible) {
        for (int i = 0; i < personalCalendarIds.length; i++) {
            Long id = personalCalendarIds[i];
            PersonalCalendar personalCalendar = calendarDAO.getPersonalCalendar(id);
            if (!personalCalendar.getCreator().equals(userId) || isVisible == personalCalendar.isVisible())
                continue;
            personalCalendar.setVisible(isVisible);
            calendarDAO.savePersonalCalendar(personalCalendar);
        }
    }

    /**
     * @param userId              用户Id
     * @param personalCalendarIds 个人日历Id数组
     * @param isSelected          是否选中
     */
    public void changePersonalCalendarSelectedStatus(String userId, Long[] personalCalendarIds, boolean isSelected) {
        for (int i = 0; i < personalCalendarIds.length; i++) {
            Long id = personalCalendarIds[i];
            PersonalCalendar personalCalendar = calendarDAO.getPersonalCalendar(id);
            if (!personalCalendar.getCreator().equals(userId) || isSelected == personalCalendar.isSelected())
                continue;
            personalCalendar.setSelected(isSelected);
            calendarDAO.savePersonalCalendar(personalCalendar);
        }
    }

    /**
     * @param userId             用户Id
     * @param personalCalendarId 个人日历Id
     * @param color              日历颜色
     */
    public void changePersonalCalendarColor(String userId, Long personalCalendarId, String color) {
        PersonalCalendar personalCalendar = calendarDAO.getPersonalCalendar(personalCalendarId);
        if (color.equals(personalCalendar.getColor()))
            return;
        personalCalendar.setColor(color);
        calendarDAO.savePersonalCalendar(personalCalendar);
    }

    /**
     * 更新个人日历中文名
     *
     * @param personalCalendars 个人日历列表
     */
    private void upadteCreatorName(List personalCalendars) {
        for (int i = 0; i < personalCalendars.size(); i++) {
            PersonalCalendar personalCalendar = (PersonalCalendar) personalCalendars.get(i);
            User user = userService.loadUser(personalCalendar.getCalendarCreator());
            personalCalendar.setCreatorName(user.getName());
        }
    }
}
