package com.gtis.egov.calendar.service.impl;

import com.gtis.egov.calendar.model.Calendar;
import com.gtis.egov.calendar.model.Event;
import com.gtis.egov.calendar.dao.CalendarDAO;
import com.gtis.egov.calendar.dao.EventDAO;
import com.gtis.egov.calendar.service.CalendarExchangeManager;
import com.gtis.egov.calendar.utils.DateUtils;
import com.gtis.egov.calendar.utils.StringUtils;
import net.fortuna.ical4j.data.CalendarBuilder;
import net.fortuna.ical4j.data.CalendarOutputter;
import net.fortuna.ical4j.data.FoldingWriter;
import net.fortuna.ical4j.data.ParserException;
import net.fortuna.ical4j.model.*;
import net.fortuna.ical4j.model.component.VEvent;
import net.fortuna.ical4j.model.component.VTimeZone;
import net.fortuna.ical4j.model.parameter.Value;
import net.fortuna.ical4j.model.property.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

/**
 * 日历导入导出管理实现
 * <p/>
 *
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 * @version V1.0, 2009-5-14 10:37:32
 */
public class CalendarExchangeManagerImpl implements CalendarExchangeManager {

    private static final Logger logger = LoggerFactory.getLogger(CalendarExchangeManagerImpl.class);

    private CalendarDAO calendarDAO;
    private EventDAO eventDAO;

    public void setCalendarDAO(CalendarDAO calendarDAO) {
        this.calendarDAO = calendarDAO;
    }

    public void setEventDAO(EventDAO eventDAO) {
        this.eventDAO = eventDAO;
    }

    /**
     * 导入iCalendar格式日历文件
     *
     * @param inputStream 输入流
     * @param calendarId  日历Id
     * @return 成功导入事件数量
     */
    public int importIcsCalendarEvents(InputStream inputStream, Long calendarId) throws IOException {
        CalendarBuilder builder = new CalendarBuilder();
        net.fortuna.ical4j.model.Calendar calendar;
        int count = 0;
        try {
            calendar = builder.build(inputStream);
        } catch (ParserException e) {
            logger.error("Parser input file error");
            return count;
        }
        Calendar cal = calendarDAO.getCalendar(calendarId);
        for (Iterator i = calendar.getComponents().iterator(); i.hasNext();) {
            Component component = (Component) i.next();
            if (component instanceof net.fortuna.ical4j.model.component.VTimeZone) {
            } else if (component instanceof net.fortuna.ical4j.model.component.VEvent) {
                Summary summary = (Summary) component.getProperty(Property.SUMMARY);
                Url url = (Url) component.getProperty(Property.URL);
                Location location = (Location) component.getProperty(Property.LOCATION);
                Description description = (Description) component.getProperty(Property.DESCRIPTION);
                DtStart start = (DtStart) component.getProperty(Property.DTSTART);
                DtEnd end = (DtEnd) component.getProperty(Property.DTEND);
                RRule rRule = (RRule) component.getProperty(Property.RRULE);
                if (summary == null || start == null)
                    continue;
                Event event = new Event();
                event.setCalendarId(cal.getId());
                event.setTitle(summary.getValue());
                if (url != null)
                    event.setUrl(url.getValue());
                if (location != null)
                    event.setLocation(location.getValue());
                if (description != null)
                    event.setDescription(description.getValue());
                Date sd = start.getDate();
                event.setStartTime(sd);
                if (end != null) {
                    Date ed = end.getDate();
                    if (DateUtils.getStartOfDay(sd).equals(sd) && DateUtils.addDays(sd, 1).equals(ed))
                        event.setWholeDay(true);
                    else
                        event.setEndTime(ed);
                }
                event.setCreator(cal.getCreator());
                event.setUpdateTime(new Date());
                if (rRule != null) {
                    Recur recur = rRule.getRecur();
                    if (recur.getInterval() < 1)
                        recur.setInterval(1);
                    event.setRecurrenceRule(recur.toString());
                    event.setRepeatEndTime(rRule.getRecur().getUntil());
                }
                eventDAO.saveEvent(event);
                count++;
            }
        }
        return count;
    }

    /**
     * 导出iCalendar格式日历文件
     *
     * @param outputStream 输出流
     * @param calendarId   日历Id
     * @return 成功导出事件数量
     */
    public int exportIcsCalendarEvents(OutputStream outputStream, Long calendarId) throws IOException {
        Calendar cal = calendarDAO.getCalendar(calendarId);
        List events = eventDAO.getEvents(calendarId);
        net.fortuna.ical4j.model.Calendar calendar = new net.fortuna.ical4j.model.Calendar();
        PropertyList ps = calendar.getProperties();
        ComponentList cs = calendar.getComponents();
        ps.add(new ProdId("-//WisEdu Inc//Ccs Calendar 1.0//EN"));
        ps.add(Version.VERSION_2_0);
        ps.add(CalScale.GREGORIAN);
        ps.add(new XProperty("X-WR-CALNAME", cal.getName()));
        ps.add(new XProperty("X-WR-CALDESC", cal.getDescription()));
        ps.add(new XProperty("X-WR-CALCOLOR", cal.getColor()));
        VTimeZone tz = TimeZoneRegistryFactory.getInstance().createRegistry().getTimeZone("Asia/Shanghai").getVTimeZone();
        cs.add(tz);
        int count = 0;
        for (int i = 0; i < events.size(); i++) {
            Event event = (Event) events.get(i);
            VEvent vEvent = new VEvent();
            PropertyList eps = vEvent.getProperties();
            Property summary = new Summary(event.getTitle());
            eps.add(summary);
            if (StringUtils.isNotBlank(event.getUrl())) {
                try {
                    Property url = new Url(new URI(event.getUrl()));
                    eps.add(url);
                } catch (URISyntaxException e) {
                    //ignore
                }
            }
            if (StringUtils.isNotBlank(event.getLocation())) {
                Property location = new Location(event.getLocation());
                eps.add(location);
            }
            if (StringUtils.isNotBlank(event.getDescription())) {
                Property description = new Description(event.getDescription());
                eps.add(description);
            }
            DateProperty start = new DtStart();
            DateProperty end = new DtEnd();
            if (event.isWholeDay()) {
                start.getParameters().add(Value.DATE);
                end.getParameters().add(Value.DATE);
                start.setDate(new net.fortuna.ical4j.model.Date(event.getStartTime()));
                end.setDate(new net.fortuna.ical4j.model.Date(event.getEndTime()));
            } else {
                start.setDate(new DateTime(event.getStartTime()));
                end.setDate(new DateTime(event.getEndTime()));
            }
            eps.add(start);
            eps.add(end);
            if (StringUtils.isNotBlank(event.getRecurrenceRule())) {
                try {
                    Recur recur = new Recur(event.getRecurrenceRule());
                    if (event.getRepeatEndTime() != null)
                        recur.setUntil(new DateTime(event.getRepeatEndTime()));
                    Property rRule = new RRule(recur);
                    eps.add(rRule);
                } catch (ParseException e) {
                    //ignore
                }
            }
            cs.add(vEvent);
            count++;
        }
        CalendarOutputter outputter = new CalendarOutputter(false, FoldingWriter.REDUCED_FOLD_LENGTH);
        outputter.setValidating(false);
        try {
            outputter.output(calendar, outputStream);
        } catch (ValidationException e) {
            //ignore
        }
        return count;
    }
}
