package com.gtis.egov.calendar.service;

import com.gtis.egov.calendar.model.Calendar;
import com.gtis.egov.calendar.model.PersonalCalendar;
import com.wiscom.generic.base.util.PaginationSupport;

import java.util.List;

/**
 * 日历管理接口
 * <p/>
 *
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 * @version V1.0, 2009-5-14 10:30:31
 */
public interface CalendarManager {

    /**
     * 保存基本日历
     *
     * @param calendar 日历
     * @return 日历
     */
    Calendar saveCalendar(Calendar calendar);

    /**
     * 获取基本日历对象
     *
     * @param calendarId 日历Id
     * @return 日历
     */
    Calendar getCalendar(Long calendarId);

    /**
     * 检索基本日历
     *
     * @param term  检索关键字 可为空
     * @param types 日历类型 -1为所有
     * @param start 起始序号
     * @param num   每页记录数
     * @return 分页对象
     */
    PaginationSupport searchCalendars(String term, Short[] types, int start, int num);

    /**
     * 检索好友可见基本日历
     *
     * @param term   检索关键字 可为空
     * @param userId 用户Id
     * @param start  起始序号
     * @param num    每页记录数
     * @return 分页对象
     */
    PaginationSupport searchFriendCalendars(String term, String userId, int start, int num);

    /**
     * 删除基本日历
     *
     * @param calendarId 日历Id
     */
    void removeCalendar(Long calendarId);

    /**
     * 保存个人日历
     *
     * @param personalCalendar 个人日历
     * @return 个人日历
     */
    PersonalCalendar savePersonalCalendar(PersonalCalendar personalCalendar);

    /**
     * @param personalCalendarId 个人日历Id
     * @return 个人日历
     */
    PersonalCalendar getPersonalCalendar(Long personalCalendarId);

    /**
     * 根据基本日历id和用户名获取日历
     *
     * @param calendarId 日历Id
     * @param userId     用户Id
     * @return 个人日历
     */
    PersonalCalendar getPersonalCalendar(Long calendarId, String userId);

    /**
     * 获取日历相关的个人日历列表
     *
     * @param calendarId 日历id
     * @return 个人日历列表
     */
    List/*<PersonalCalendar>*/ getPersonalCalendars(Long calendarId);

    /**
     * 同步标准基本日历到个人日历
     *
     * @param userId 用户Id
     */
    void syncStandardCalendarToPersonalCalendar(String userId);

    /**
     * 同步群组基本日历到个人日历
     *
     * @param userId 用户Id
     */
    void syncGroupCalendarToPersonalCalendar(String userId);

    /**
     * @param userId    用户Id
     * @param isVisible 是否可见 null为全部
     * @return 个人日历列表
     */
    List/*<PersonalCalendar>*/ getPersonalCalendars(String userId, Boolean isVisible);

    /**
     * 获取标准个人日历
     *
     * @param userId 用户Id
     * @return 标准日历列表
     */
    List/*<PersonalCalendar>*/ getStandardPersonalCalendars(String userId);

    /**
     * 获取群组基本日历
     *
     * @param groupId 群组Id
     * @return 群组日历
     */
    List getGroupCalendars(String groupId);

    /**
     * 获取群组个人日历
     *
     * @param userId  用户Id
     * @param groupId 群组Id
     * @return 群组日历列表
     */
    List/*<PersonalCalendar>*/ getGroupPersonalCalendars(String userId, String groupId);

    /**
     * 删除个人日历
     *
     * @param personalCalendarId 个人日历Id
     */
    void removePersonalCalendar(Long personalCalendarId);

    /**
     * @param userId              用户Id
     * @param personalCalendarIds 个人日历Id数组
     * @param isVisible           是否可见
     */
    void changePersonalCalendarVisibleStatus(String userId, Long[] personalCalendarIds, boolean isVisible);

    /**
     * @param userId              用户Id
     * @param personalCalendarIds 个人日历Id数组
     * @param isSelected          是否选中
     */
    void changePersonalCalendarSelectedStatus(String userId, Long[] personalCalendarIds, boolean isSelected);

    /**
     * @param userId             用户Id
     * @param personalCalendarId 个人日历Id
     * @param color              日历颜色
     */
    void changePersonalCalendarColor(String userId, Long personalCalendarId, String color);
}
