package com.gtis.egov.calendar.dao.jdbc;

import com.gtis.egov.calendar.dao.ReminderDAO;
import com.gtis.egov.calendar.model.Event;
import com.gtis.egov.calendar.model.Reminder;
import com.wiscom.generic.base.jdbc.AbstractJdbcDAO;

import java.util.*;

/**
 * 提醒管理DAO实现
 * <p/>
 *
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 * @version V1.0, 2009-5-14 20:22:44
 */
public class ReminderDAOImpl extends AbstractJdbcDAO implements ReminderDAO {

    private static final String T_CALENDAR_REMINDER_SEQ = "T_CALENDAR_REMINDER_SEQ";

    private static Class[] beanClasses = new Class[]{Reminder.class, Event.class};

    /**
     * 保存提醒
     *
     * @param reminder 提醒
     * @return 提醒
     */
    public Reminder saveReminder(Reminder reminder) {
        return reminder.getId() == null ? addReminder(reminder) : updateReminder(reminder);
    }

    private Reminder updateReminder(Reminder reminder) {
        this.updateByBean("update t_calendar_reminder set event_id=:eventId,creator=:creator," +
                "method=:method,seconds=:seconds where id=:id", reminder);
        return reminder;
    }


    private Reminder addReminder(Reminder reminder) {
        reminder.setId(new Long(this.idGenerator.nextLongValue(T_CALENDAR_REMINDER_SEQ)));
        this.updateByBean("insert into t_calendar_reminder (id,event_id,creator,method,seconds) values " +
                "(:id,:eventId,:creator,:method,:seconds)", reminder);
        return reminder;
    }

    /**
     * 获取提醒
     *
     * @param reminderId 提醒Id
     * @return 提醒
     */
    public Reminder getReminder(Long reminderId) {
        final String sql = "select * from t_calendar_reminder where id=:id";
        Map params = new HashMap(1);
        params.put("id", reminderId);
        return (Reminder) this.queryForBean(sql, params, Reminder.class);
    }

    /**
     * 获取指定事件的所有提醒
     *
     * @param eventId 事件Id
     * @return 提醒列表
     */
    public List getReminders(Long eventId) {
        final String sql = "select * from t_calendar_reminder where event_id=:eventId";
        Map params = new HashMap(1);
        params.put("eventId", eventId);
        return this.queryForBeanList(sql, -1, -1, params, Reminder.class);
    }

    /**
     * 获取时间段内需要处理的提醒
     *
     * @param startDate 起始时间
     * @param endDate   结束时间
     * @return 提醒列表
     */
    public List getBaseReminders(Date startDate, Date endDate) {
        final String sql = "select * from t_calendar_reminder t1 " +
                "left join t_calendar_event t2 on t2.id = t1.event_id " +
                "left join t_calendar t3 on t3.id = t2.calendar_id " +
                "where t2.start_time - t1.seconds / 86400 > :startDate and (t2.start_time - t1.seconds / 86400 < :endDate " +
                "or (t2.recurrence_rule is not null and (t2.repeat_end_time is null or " +
                "t2.repeat_end_time > :startDate)))";
        Map params = new HashMap(1);
        params.put("startDate", startDate);
        params.put("endDate", endDate);
        List beansList = this.queryForManyBeanList(sql, -1, -1, params, beanClasses);
        List reminds = new ArrayList(beansList.size());
        for (int i = 0; i < beansList.size(); i++) {
            Object[] objs = (Object[]) beansList.get(i);
            Reminder reminder = (Reminder) objs[0];
            Event event = (Event) objs[1];
            reminder.setEvent(event);
            reminds.add(reminder);
        }
        return reminds;
    }

    /**
     * 删除提醒
     *
     * @param reminderId 提醒Id
     */
    public void removeReminder(Long reminderId) {
        Map params = new HashMap(1);
        params.put("id", reminderId);
        this.update("delete from t_calendar_reminder where id=:id", params);
    }

    /**
     * 删除指定事件的提醒
     *
     * @param eventId 提醒Id
     */
    public void removeReminders(Long eventId) {
        Map params = new HashMap(1);
        params.put("eventId", eventId);
        this.update("delete from t_calendar_reminder where event_id=:eventId", params);
    }

    /**
     * 删除指定日历的提醒
     *
     * @param calendarId 日历Id
     */
    public void removeRemindersByCalendarId(Long calendarId) {
        Map params = new HashMap(1);
        params.put("calendarId", calendarId);
        this.update("delete from t_calendar_reminder where event_id in " +
                "(select event_id from t_calendar_event where calendar_id=:calendarId)", params);
    }
}
