package com.gtis.egov.calendar.dao.jdbc;

import com.gtis.egov.calendar.model.Event;
import com.gtis.egov.calendar.dao.EventDAO;
import com.gtis.egov.calendar.utils.DateUtils;
import com.wiscom.generic.base.jdbc.AbstractJdbcDAO;

import java.util.*;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 * @version V1.0, 2009-5-14 18:40:59
 */
public class EventDAOImpl extends AbstractJdbcDAO implements EventDAO {

    private static final String T_CALENDAR_EVENT_SEQ = "T_CALENDAR_EVENT_SEQ";

    /**
     * 保存事件
     *
     * @param event 事件
     * @return 事件
     */
    public Event saveEvent(Event event) {
        if (event.isWholeDay()) {
            event.setStartTime(DateUtils.getStartOfDay(event.getStartTime()));
            event.setEndTime(DateUtils.addDays(event.getStartTime(), 1));
        }
        return event.getId() == null ? addEvent(event) : updateEvent(event);
    }

    private Event updateEvent(Event event) {
        this.updateByBean("update t_calendar_event set calendar_id=:calendarId,title=:title,url=:url,location=:location," +
                "description=:description,start_time=:startTime,end_time=:endTime,whole_day=:wholeDay,creator=:creator," +
                "update_time=:updateTime,recurrence_rule=:recurrenceRule,repeat_end_time=:repeatEndTime where id=:id", event);
        return event;
    }

    private Event addEvent(Event event) {
        event.setId(new Long(this.idGenerator.nextLongValue(T_CALENDAR_EVENT_SEQ)));
        this.updateByBean("insert into t_calendar_event (id,calendar_id,title,url,location,description,start_time,end_time," +
                "whole_day,creator,update_time,recurrence_rule,repeat_end_time) values " +
                "(:id,:calendarId,:title,:url,:location,:description,:startTime,:endTime," +
                ":wholeDay,:creator,:updateTime,:recurrenceRule,:repeatEndTime)", event);
        return event;
    }

    /**
     * 获取事件
     *
     * @param eventId 事件Id
     * @return 事件
     */
    public Event getEvent(Long eventId) {
        final String sql = "select * from t_calendar_event where id=:id";
        Map params = new HashMap(1);
        params.put("id", eventId);
        return (Event) this.queryForBean(sql, params, Event.class);
    }

    /**
     * 获取某个日历的所有事件
     *
     * @param calendarId 日历Id
     * @return 事件列表
     */
    public List getEvents(Long calendarId) {
        Map params = new HashMap(3);
        final String sql = "select * from t_calendar_event where calendar_id=:calendarId order by start_time";
        params.put("calendarId", calendarId);
        return this.queryForBeanList(sql, -1, -1, params, Event.class);
    }

    /**
     * 获取指定日历时间段内的事件
     *
     * @param userId      用户Id
     * @param calendarIds 日历Ids
     * @param startDate   开始时间
     * @param endDate     结束时间    @return 事件列表
     */
    public List getBaseEvents(String userId, Long[] calendarIds, Date startDate, Date endDate) {
        if (calendarIds == null || calendarIds.length == 0)
            return Collections.EMPTY_LIST;
        final List ids = Arrays.asList(calendarIds);
        final String sql = "select t1.*,t2.color,(select count(id) from t_calendar_reminder t3 where t3.event_id=t1.id) as remind_count from t_calendar_event t1 " +
                "inner join t_calendar_config t2 on t2.calendar_id=t1.calendar_id  " +
                "where t2.creator=:userId and t1.calendar_id in (:calendarIds) and t1.start_time<=:endDate and " +
                "((t1.whole_day=1 or t1.end_time>=:startDate) or (t1.recurrence_rule is not null and (t1.repeat_end_time is null or t1.repeat_end_time>=t1.start_time))) order by t1.start_time";
        Map params = new HashMap(3);
        params.put("userId", userId);
        params.put("calendarIds", ids);
        params.put("startDate", startDate);
        params.put("endDate", endDate);
        return this.queryForBeanList(sql, -1, -1, params, Event.class);
    }

    /**
     * 获取用户时间段内所有事件
     *
     * @param userId    用户Id
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @return 事件列表
     */
    public List getUserBaseEvents(String userId, Date startDate, Date endDate) {
        final String sql = "select t1.*,t2.color,(select count(id) from t_calendar_reminder t3 where t3.event_id=t1.id) as remind_count from t_calendar_event t1 " +
                "inner join t_calendar_config t2 on t2.calendar_id=t1.calendar_id where t2.creator=:userId and t2.visible=1 and t2.selected=1 and t1.start_time<=:endDate and " +
                "((t1.whole_day=1 or t1.end_time>=:startDate) or (t1.recurrence_rule is not null and (t1.repeat_end_time is null or t1.repeat_end_time>=t1.start_time))) order by t1.start_time";
        Map params = new HashMap(3);
        params.put("userId", userId);
        params.put("startDate", startDate);
        params.put("endDate", endDate);
        return this.queryForBeanList(sql, -1, -1, params, Event.class);
    }

    /**
     * 删除事件
     *
     * @param eventId 事件Id
     */
    public void removeEvent(Long eventId) {
        Map params = new HashMap(1);
        params.put("id", eventId);
        this.update("delete from t_calendar_event where id=:id", params);
    }

    /**
     * 删除某个日历的全部事件
     *
     * @param calendarId 日历Id
     */
    public void removeEvents(Long calendarId) {
        Map params = new HashMap(1);
        params.put("calendarId", calendarId);
        this.update("delete from t_calendar_event where calendar_id=:calendarId", params);
    }
}
