package com.gtis.egov.calendar.dao.jdbc;

import com.gtis.egov.calendar.dao.CalendarDAO;
import com.gtis.egov.calendar.model.Calendar;
import com.gtis.egov.calendar.model.PersonalCalendar;
import com.wiscom.generic.base.jdbc.AbstractJdbcDAO;
import com.wiscom.generic.base.util.PaginationSupport;
import org.apache.commons.lang.StringUtils;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 日历管理DAO实现
 * <p/>
 *
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 * @version V1.0, 2009-5-14 14:42:49
 */
public class CalendarDAOImpl extends AbstractJdbcDAO implements CalendarDAO {

    private static final String T_CALENDAR_SEQ = "T_CALENDAR_SEQ";
    private static final String T_CALENDAR_CONFIG_SEQ = "T_CALENDAR_CONFIG_SEQ";

    /**
     * 保存基本日历
     *
     * @param calendar 日历
     * @return 日历
     */
    public Calendar saveCalendar(Calendar calendar) {
        return calendar.getId() == null ? addCalendar(calendar) : updateCalendar(calendar);
    }

    private Calendar updateCalendar(Calendar calendar) {
        this.updateByBean("update t_calendar set name=:name,description=:description,color=:color,owner=:owner," +
                "creator=:creator,create_time=:createTime,type=:type where id=:id", calendar);
        return calendar;
    }

    private Calendar addCalendar(Calendar calendar) {
        calendar.setId(new Long(this.idGenerator.nextLongValue(T_CALENDAR_SEQ)));
        this.updateByBean("insert into t_calendar (id,name,description,color,owner,creator,create_time,type) values " +
                "(:id,:name,:description,:color,:owner,:creator,:createTime,:type)", calendar);
        return calendar;
    }

    /**
     * 获取基本日历对象
     *
     * @param calendarId 日历Id
     * @return 日历
     */
    public Calendar getCalendar(Long calendarId) {
        final String sql = "select * from t_calendar where id=:id";
        Map params = new HashMap(1);
        params.put("id", calendarId);
        return (Calendar) this.queryForBean(sql, params, Calendar.class);
    }

    /**
     * 检索基本日历
     *
     * @param term  检索关键字 可为空
     * @param types 日历类型
     * @param start 起始序号
     * @param num   每页记录数
     * @return 日历列表
     */
    public PaginationSupport searchCalendars(String term, Short[] types, int start, int num) {
        if (types.length == 0)
            return null;
        StringBuffer sql = new StringBuffer("select * from t_calendar where type in (:types) ");
        Map params = new HashMap(2);
        params.put("types", Arrays.asList(types));
        if (StringUtils.isNotBlank(term)) {
            params.put("name", "%" + term + "%");
            sql.append("and name like :name ");
        }
        sql.append("order by create_time desc");
        return this.getBeanPaginationSupport(sql.toString(), null, start, num, params, Calendar.class);
    }

    /**
     * 检索好友可见基本日历
     *
     * @param term   检索关键字 可为空
     * @param userId 用户Id
     * @param start  起始序号
     * @param num    每页记录数
     * @return 分页对象
     */
    public PaginationSupport searchFriendCalendars(String term, String userId, int start, int num) {
        Map params = new HashMap(2);
        StringBuffer sql = new StringBuffer("select t1.* from t_calendar t1 " +
                "left join t_user_friend t2 on t2.friend_user_id=t1.creator where t2.user_id=:userId and t1.type=3 ");
        if (StringUtils.isNotBlank(term)) {
            params.put("name", "%" + term + "%");
            sql.append("and t1.name like :name ");
        }
        sql.append("order by t1.create_time desc");
        params.put("userId", userId);
        return this.getBeanPaginationSupport(sql.toString(), null, start, num, params, Calendar.class);
    }

    /**
     * 删除基本日历
     *
     * @param calendarId 日历Id
     */
    public void removeCalendar(Long calendarId) {
        Map params = new HashMap(1);
        params.put("id", calendarId);
        this.update("delete from t_calendar where id=:id", params);
    }

    /**
     * 删除属于某个基本日历的个人日历
     *
     * @param calendarId 基本日历Id
     */
    public void removePersonalCalendars(Long calendarId) {
        Map params = new HashMap(1);
        params.put("calendarId", calendarId);
        this.update("delete from t_calendar_config where calendar_id=:calendarId", params);
    }

    /**
     * 删除个人日历
     *
     * @param personalCalendarId 个人日历Id
     */
    public void removePersonalCalendar(Long personalCalendarId) {
        Map params = new HashMap(1);
        params.put("id", personalCalendarId);
        this.update("delete from t_calendar_config where id=:id", params);
    }

    /**
     * 保存个人日历
     *
     * @param personalCalendar 个人日历
     * @return 个人日历
     */
    public PersonalCalendar savePersonalCalendar(PersonalCalendar personalCalendar) {
        return personalCalendar.getId() == null ? addPersonalCalendar(personalCalendar) : updatePersonalCalendar(personalCalendar);
    }

    private PersonalCalendar updatePersonalCalendar(PersonalCalendar personalCalendar) {
        this.updateByBean("update t_calendar_config set calendar_id=:calendarId,name=:name,creator=:creator,add_time=:addTime,color=:color," +
                "visible=:visible,selected=:selected,remind=:remind where id=:id", personalCalendar);
        return personalCalendar;
    }

    private PersonalCalendar addPersonalCalendar(PersonalCalendar personalCalendar) {
        personalCalendar.setId(new Long(this.idGenerator.nextLongValue(T_CALENDAR_CONFIG_SEQ)));
        this.updateByBean("insert into t_calendar_config (id,calendar_id,name,creator,add_time,color,visible,selected,remind) values " +
                "(:id,:calendarId,:name,:creator,:addTime,:color,:visible,:selected,:remind)", personalCalendar);
        return personalCalendar;
    }

    /**
     * @param personalCalendarId 个人日历Id
     * @return 个人日历
     */
    public PersonalCalendar getPersonalCalendar(Long personalCalendarId) {
        final String sql = "select t1.*,t2.description,t2.creator as calendar_creator,t2.create_time,t2.type from t_calendar_config t1 " +
                "left join t_calendar t2 on t2.id = t1.calendar_id " +
                "where t1.id = :id";
        Map params = new HashMap(1);
        params.put("id", personalCalendarId);
        return (PersonalCalendar) this.queryForBean(sql, params, PersonalCalendar.class);
    }

    /**
     * 获取日历相关的个人日历列表
     *
     * @param calendarId 日历id
     * @return 个人日历列表
     */
    public List getPersonalCalendars(Long calendarId) {
        final String sql = "select t1.*,t2.description,t2.creator as calendar_creator,t2.create_time,t2.type from t_calendar_config t1 " +
                "left join t_calendar t2 on t2.id = t1.calendar_id " +
                "where t1.calendar_id = :calendarId order by t1.add_time desc";
        Map params = new HashMap(1);
        params.put("calendarId", calendarId);
        return this.queryForBeanList(sql, -1, -1, params, PersonalCalendar.class);
    }

    /**
     * 根据基本日历id和用户名获取日历
     *
     * @param calendarId 日历Id
     * @param userId     用户Id
     * @return 个人日历
     */
    public PersonalCalendar getPersonalCalendar(Long calendarId, String userId) {
        final String sql = "select t1.*,t2.description,t2.creator as calendar_creator,t2.create_time,t2.type from t_calendar_config t1 " +
                "inner join t_calendar t2 on t2.id = t1.calendar_id " +
                "where t2.id = :calendarId and t1.creator=:userId";
        Map params = new HashMap(2);
        params.put("calendarId", calendarId);
        params.put("userId", userId);
        return (PersonalCalendar) this.queryForBean(sql, params, PersonalCalendar.class);
    }

    /**
     * 获取个人日历列表
     *
     * @param userId    用户Id
     * @param isVisible 是否可见 null为全部
     * @return 个人日历列表
     */
    public List getPersonalCalendars(String userId, Boolean isVisible) {
        final String sql = "select t1.*,t2.description,t2.creator as calendar_creator,t2.create_time,t2.type from t_calendar_config t1 " +
                "left join t_calendar t2 on t2.id = t1.calendar_id " +
                "where t1.creator = :userId" + (isVisible != null ? " and visible=" + (isVisible.booleanValue() ? 1 : 0) : "") + " order by t1.add_time desc";
        Map params = new HashMap(1);
        params.put("userId", userId);
        return this.queryForBeanList(sql, -1, -1, params, PersonalCalendar.class);
    }

    /**
     * 根据类型获取用户日历.
     *
     * @param userId 用户Id
     * @param type   日历类型
     * @return 个人日历列表
     */
    public List getPersonalCalendars(String userId, short type) {
        final String sql = "select t1.*,t2.description,t2.creator as calendar_creator,t2.create_time,t2.type from t_calendar_config t1 " +
                "left join t_calendar t2 on t2.id = t1.calendar_id " +
                "where t1.creator = :userId and t2.type=:type order by t1.add_time desc";
        Map params = new HashMap(1);
        params.put("userId", userId);
        params.put("type", Short.valueOf(type));
        return this.queryForBeanList(sql, -1, -1, params, PersonalCalendar.class);
    }

    /**
     * 获取未同步群组基本日历
     *
     * @param userId 用户Id
     * @return 基本日历列表
     */
    public List getAsynchronousGroupCalendars(String userId) {
        final String sql = "select t1.* from t_calendar t1 " +
                "left join pf_user_organ_rel t2 on t2.organ_id = t1.owner " +
                "left join t_calendar_config t3 on t3.creator=t2.user_id and t3.calendar_id=t1.id " +
                "where t2.user_id = :userId and t3.id is null";
        Map params = new HashMap(1);
        params.put("userId", userId);
        return this.queryForBeanList(sql, -1, -1, params, Calendar.class);
    }

    /**
     * 获取标准的个人日历列表
     *
     * @param userId 用户Id
     * @return 标准日历列表
     */
    public List getStandardPersonalCalendars(String userId) {
        final String sql = "select t1.*,t2.description,t2.creator as calendar_creator,t2.create_time,t2.type from t_calendar_config t1 " +
                "left join t_calendar t2 on t2.id = t1.calendar_id " +
                "where t1.creator = :userId and t2.type=5 order by t1.add_time desc";
        Map params = new HashMap(1);
        params.put("userId", userId);
        return this.queryForBeanList(sql, -1, -1, params, PersonalCalendar.class);
    }

    /**
     * 获取群组个人日历列表
     *
     * @param userId  用户Id
     * @param groupId 群组Id
     * @return 群组日历列表
     */
    public List getGroupPersonalCalendars(String userId, String groupId) {
        final String sql = "select t1.*,t2.description,t2.creator as calendar_creator,t2.create_time,t2.type from t_calendar_config t1 " +
                "left join t_calendar t2 on t2.id = t1.calendar_id " +
                "where t1.creator = :userId and t2.type=6 and t2.owner=:groupId order by t1.add_time desc";
        Map params = new HashMap(1);
        params.put("userId", userId);
        params.put("groupId", groupId);
        return this.queryForBeanList(sql, -1, -1, params, PersonalCalendar.class);
    }

    /**
     * 获取未同步的标准日历
     *
     * @param userId 用户Id
     * @return 标准日历列表
     */
    public List getAsynchronousStandardCalendars(String userId) {
        final String sql = "select * from t_calendar where type=5 and id not in (select calendar_id from t_calendar_config where creator=:userId)";
        Map params = new HashMap(1);
        params.put("userId", userId);
        return this.queryForBeanList(sql, -1, -1, params, Calendar.class);
    }

    /**
     * 获取群组基本日历
     *
     * @param groupId 群组Id
     * @return 群组日历
     */
    public List getGroupCalendars(String groupId) {
        final String sql = "select * from t_calendar where type=6 and owner=:groupId";
        Map params = new HashMap(1);
        params.put("groupId", groupId);
        return this.queryForBeanList(sql, -1, -1, params, Calendar.class);
    }
}
