package com.gtis.egov.calendar.dao;

import com.gtis.egov.calendar.model.Calendar;
import com.gtis.egov.calendar.model.PersonalCalendar;
import com.wiscom.generic.base.util.PaginationSupport;

import java.util.List;

/**
 * 日历管理DAO接口
 * <p/>
 *
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 * @version V1.0, 2009-5-14 10:38:55
 */
public interface CalendarDAO {

    /**
     * 保存基本日历
     *
     * @param calendar 日历
     * @return 日历
     */
    Calendar saveCalendar(Calendar calendar);

    /**
     * 获取基本日历对象
     *
     * @param calendarId 日历Id
     * @return 日历
     */
    Calendar getCalendar(Long calendarId);

    /**
     * 检索基本日历
     *
     * @param term  检索关键字 可为空
     * @param types 日历类型 -1为所有
     * @param start 起始序号
     * @param num   每页记录数
     * @return 分页对象
     */
    PaginationSupport searchCalendars(String term, Short[] types, int start, int num);

    /**
     * 检索好友可见基本日历
     *
     * @param term   检索关键字 可为空
     * @param userId 用户Id
     * @param start  起始序号
     * @param num    每页记录数
     * @return 分页对象
     */
    PaginationSupport searchFriendCalendars(String term, String userId, int start, int num);

    /**
     * 删除基本日历
     *
     * @param calendarId 日历Id
     */
    void removeCalendar(Long calendarId);

    /**
     * 删除属于某个基本日历的个人日历
     *
     * @param calendarId 基本日历Id
     */
    void removePersonalCalendars(Long calendarId);

    /**
     * 删除个人日历
     *
     * @param personalCalendarId 个人日历Id
     */
    void removePersonalCalendar(Long personalCalendarId);


    /**
     * 保存个人日历
     *
     * @param personalCalendar 个人日历
     * @return 个人日历
     */
    PersonalCalendar savePersonalCalendar(PersonalCalendar personalCalendar);

    /**
     * @param personalCalendarId 个人日历Id
     * @return 个人日历
     */
    PersonalCalendar getPersonalCalendar(Long personalCalendarId);

    /**
     * 获取日历相关的个人日历列表
     *
     * @param calendarId 日历id
     * @return 个人日历列表
     */
    List/*<PersonalCalendar>*/ getPersonalCalendars(Long calendarId);

    /**
     * 根据基本日历id和用户名获取日历
     *
     * @param calendarId 日历Id
     * @param userId     用户Id
     * @return 个人日历
     */
    PersonalCalendar getPersonalCalendar(Long calendarId, String userId);

    /**
     * 获取个人日历列表
     *
     * @param userId    用户Id
     * @param isVisible 是否可见 null为全部
     * @return 个人日历列表
     */
    List/*<PersonalCalendar>*/ getPersonalCalendars(String userId, Boolean isVisible);


    /**
     * 根据类型获取用户日历.
     *
     * @param userId 用户Id
     * @param type   日历类型
     * @return 个人日历列表
     */
    List getPersonalCalendars(String userId, short type);

    /**
     * 获取未同步群组基本日历
     *
     * @param userId 用户Id
     * @return 基本日历列表
     */
    List getAsynchronousGroupCalendars(String userId);

    /**
     * 获取标准的个人日历列表
     *
     * @param userId 用户Id
     * @return 标准日历列表
     */
    List/*<PersonalCalendar>*/ getStandardPersonalCalendars(String userId);

    /**
     * 获取群组个人日历列表
     *
     * @param userId  用户Id
     * @param groupId 群组Id
     * @return 群组日历列表
     */
    List/*<PersonalCalendar>*/ getGroupPersonalCalendars(String userId, String groupId);

    /**
     * 获取未同步的标准日历
     *
     * @param userId 用户Id
     * @return 标准日历列表
     */
    List getAsynchronousStandardCalendars(String userId);

    /**
     * 获取群组基本日历
     *
     * @param groupId 群组Id
     * @return 群组日历
     */
    List/*<Calendar>*/ getGroupCalendars(String groupId);
}
