package com.gtis.egov.calendar;

import com.gtis.egov.calendar.model.Event;
import com.gtis.egov.calendar.model.Reminder;
import com.gtis.egov.calendar.utils.DateUtils;
import com.gtis.egov.calendar.utils.StringUtils;
import net.fortuna.ical4j.model.DateList;
import net.fortuna.ical4j.model.Recur;
import net.fortuna.ical4j.model.parameter.Value;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * 多个地方用到的方法
 * <p/>
 *
 * @author <a href="mailto:oznyang@163.com">oznyang</a>
 * @version V1.0, 2009-5-15 19:28:00
 */
public class Helper {

    /**
     * Logger.
     */
    private static final Logger logger = LoggerFactory.getLogger(Helper.class);

    /**
     * 由包含重复规则的基本事件根据重复规则生成基本事件
     *
     * @param baseEvent 基本事件
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @return 事件列表
     */
    public static List getRecurrenceEvents(final Event baseEvent, Date startDate, Date endDate) {
        List events = new ArrayList();
        try {
            Recur recur = new Recur(baseEvent.getRecurrenceRule());
            if (startDate.before(baseEvent.getStartTime()))
                startDate = baseEvent.getStartTime();
            if (baseEvent.getRepeatEndTime() != null) {
                recur.setUntil(new net.fortuna.ical4j.model.Date(baseEvent.getRepeatEndTime()));
                if (endDate.after(baseEvent.getRepeatEndTime()))
                    endDate = baseEvent.getRepeatEndTime();
            } else
                recur.setUntil(new net.fortuna.ical4j.model.Date(endDate));
            DateList dateList = recur.getDates(new net.fortuna.ical4j.model.Date(baseEvent.getStartTime()), new net.fortuna.ical4j.model.Date(startDate), new net.fortuna.ical4j.model.Date(DateUtils.addDays(endDate, 1)), Value.DATE);
            for (int i = 0; i < dateList.size(); i++) {
                net.fortuna.ical4j.model.Date date = (net.fortuna.ical4j.model.Date) dateList.get(i);
                int diffDate = (int) Math.ceil((double) (date.getTime() - baseEvent.getStartTime().getTime()) / DateUtils.MILLIS_PER_DAY);//获取相差的天数
                try {
                    Event event = (Event) baseEvent.clone();
                    event.setStartTime(DateUtils.addDays(event.getStartTime(), diffDate));
                    event.setEndTime(DateUtils.addDays(event.getEndTime(), diffDate));
                    event.setRecurDiff(diffDate * DateUtils.MILLIS_PER_DAY);
                    events.add(event);
                } catch (CloneNotSupportedException e) {
                    //not proccess
                }
            }
        } catch (ParseException e) {
            logger.error("Parse recur ({}) error", baseEvent.getRecurrenceRule());
        }
        return events;
    }

    /**
     * 由重复规则生成event，判断是不是在需要提醒的范围
     *
     * @param reminder  提醒
     * @param startDate 起始时间
     * @param endDate   结束时间
     * @return 提醒
     */
    public static Reminder getNearRecurrenceEventReminder(final Reminder reminder, Date startDate, Date endDate) {
        Event event = reminder.getEvent();
        if (StringUtils.isBlank(event.getRecurrenceRule()))
            return reminder;
        try {
            Recur recur = new Recur(event.getRecurrenceRule());
            if (startDate.before(event.getStartTime()))
                startDate = event.getStartTime();
            if (event.getRepeatEndTime() != null) {
                recur.setUntil(new net.fortuna.ical4j.model.Date(event.getRepeatEndTime()));
                if (endDate.after(event.getRepeatEndTime()))
                    endDate = event.getRepeatEndTime();
            } else
                recur.setUntil(new net.fortuna.ical4j.model.Date(endDate));
            DateList dateList = recur.getDates(new net.fortuna.ical4j.model.Date(event.getStartTime()), new net.fortuna.ical4j.model.Date(startDate), new net.fortuna.ical4j.model.Date(DateUtils.addDays(endDate, 1)), Value.DATE);
            for (int i = 0; i < dateList.size(); i++) {
                net.fortuna.ical4j.model.Date date = (net.fortuna.ical4j.model.Date) dateList.get(i);
                int diffDate = (int) Math.ceil((double) (date.getTime() - event.getStartTime().getTime()) / DateUtils.MILLIS_PER_DAY);//获取相差的天数
                Date remindDate = DateUtils.addSeconds(DateUtils.addDays(event.getStartTime(), diffDate), -reminder.getSeconds());
                if (remindDate.after(startDate) && remindDate.before(endDate)) {
                    event.setStartTime(DateUtils.addDays(event.getStartTime(), diffDate));
                    event.setEndTime(DateUtils.addDays(event.getEndTime(), diffDate));
                    return reminder;
                }
            }
        } catch (ParseException e) {
            logger.error("Parse recur ({}) error", event.getRecurrenceRule());
        }
        return null;
    }

    /*public static Map getHasEventDaysMap(List allEvents, Date monthStart, Date monthEnd) {
        Map map = new HashMap(31);
        for (int i = 0; i < allEvents.size(); i++) {
            Event event = (Event) allEvents.get(i);
            int diff = (int) ((event.getEndTime().getTime() - event.getStartTime().getTime()) / com.gtis.egov.calendar.utils.DateUtils.MILLIS_PER_DAY);
            for (int j = 0; j <= diff; j++) {
                Date d = com.gtis.egov.calendar.utils.DateUtils.addDays(event.getStartTime(), j);
                if (monthStart.before(d) && monthEnd.after(d))
                    map.put(new Integer(d.getDate()), Boolean.TRUE);
            }
        }
        return map;
    }

    public static List getNeedShowEvents(List allEvents, Date dayStart, Date dayEnd) {
        List events = new ArrayList(allEvents.size());
        for (int i = 0; i < allEvents.size(); i++) {
            Event event = (Event) allEvents.get(i);
            if (dayStart.before(event.getEndTime()) && dayEnd.after(event.getStartTime()))
                events.add(event);
        }
        return events;
    }*/
}
