package com.gtis.archive.core.impl;

import com.gtis.archive.Constants;
import com.gtis.archive.core.cache.Cache;
import com.gtis.archive.core.dict.Dict;
import com.gtis.archive.core.dict.DictService;
import com.gtis.archive.core.dict.Item;
import com.gtis.archive.core.ex.EntityNotFoundException;
import com.gtis.archive.core.support.xstream.CDATASupportDom4JDriver;
import com.gtis.support.hibernate.HibernateTemplate;
import com.thoughtworks.xstream.XStream;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.SQLQuery;
import org.hibernate.TransactionException;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.orm.hibernate3.HibernateTransactionManager;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-8-12
 */
public class DictServiceImpl extends HibernateTemplate<Dict, String> implements DictService {
    private static final Logger logger = LoggerFactory.getLogger(DictServiceImpl.class);
    public static final String DEFAULT_DICT_FILE = "default-dict.xml";
    private HibernateTemplate<Item, String> itemTemplate;
    @Autowired
    private HibernateTransactionManager transactionManager;
    private Cache cache;

    public void setCache(Cache cache) {
        this.cache = cache;
    }

    public Dict getDict(String name) {
        return findUniqueBy("name", name);
    }

    public Dict getDictById(String id) {
        return get(id);
    }

    public List<Dict> getDicts() {
        return getAll();
    }

    public Item getItem(String itemId) {
        return itemTemplate.get(itemId);
    }

    public List<Item> getItems(String dictName) {
        return getItems(dictName, "");
    }

    @SuppressWarnings("unchecked")
    public List<Item> getItems(String dictName, String parentId) {
        List<Item> list = cache.get(dictName, parentId);
        if (list == null) {
            Dict dict = getDict(dictName);
            if (dict == null) {
                throw new EntityNotFoundException("字典" + dictName + "未找到,请联系管理员检查您的字典设置");
            }
            Criteria criteria = itemTemplate.createCriteria();
            criteria.add(Restrictions.eq("dictId", dict.getId()));
            if (StringUtils.isNotEmpty(parentId))
                criteria.add(Restrictions.eq("parentId", parentId));
            criteria.addOrder(Order.asc("orderNumber"));
            list = criteria.list();
            cache.put(list, dictName, parentId);
        }
        return list;
    }

    public String getItemName(String dictName, String itemValue) {
        List<Item> itemList = getItems(dictName);
        if (itemList == null || itemList.size() == 0) {
            return "";
        }

        String itemName = "";
        for (Item item : itemList) {
            if(item.getValue().equals(itemValue)){
                itemName = item.getName();
                break;
            }
        }
        return itemName;
    }

    public String getItemValue(String dictName, String itemName){
        List<Item> itemList = getItems(dictName);
        if (itemList == null || itemList.size() == 0) {
            return "";
        }

        String itemValue = "";
        for (Item item : itemList) {
            if(item.getName().equals(itemName)){
                itemValue = item.getValue();
                break;
            }
        }

        return itemValue;
    }

    @SuppressWarnings("unchecked")
    public List<Item> getItems(String dictName, Object[] params) {
        Dict dict = getDict(dictName);
        SQLQuery query = getSession().createSQLQuery(dict.getSql());
        for (int i = 0; i < params.length; i++) {
            Object param = params[i];
            if (param instanceof String) {
                query.setString(i, (String) param);
            } else if (param instanceof Integer) {
                query.setInteger(i, (Integer) param);
            } else if (param instanceof Long) {
                query.setLong(i, (Long) param);
            } else if (param instanceof Float) {
                query.setFloat(i, (Float) param);
            } else if (param instanceof Double) {
                query.setDouble(i, (Double) param);
            } else {
                query.setString(i, param.toString());
            }
        }
        List<Item> items = new ArrayList<Item>();
        List<Map> list = (List<Map>) query.setResultTransformer(Criteria.ALIAS_TO_ENTITY_MAP).list();
        for (Map map : list) {
            Item item = new Item();
            item.setName(map.get("NAME").toString());
            item.setValue(map.get("VALUE").toString());
            items.add(item);
        }
        return items;
    }

    @Transactional
    public void removeDict(String id) {
        for (Item item : getItems(getDictById(id).getName())) {
            removeItem(item.getId());
        }
        delete(id);
        cache.removeAll();
    }

    @Transactional
    public void removeItem(String itemId) {
        itemTemplate.delete(itemId);
        cache.removeAll();
    }

    @Transactional
    public void saveDict(Dict dict) {
        save(dict);
        cache.removeAll();
    }

    @Transactional
    public void saveItem(Item item) {
        itemTemplate.save(item);
        cache.removeAll();
    }

    @Override
    protected void afterSessionFactorySet() {
        itemTemplate = new HibernateTemplate<Item, String>(this, Item.class);
    }

    public synchronized void init() {
        TransactionStatus status = null;
        try {
            status = transactionManager.getTransaction(new DefaultTransactionDefinition());
            initDefaultDict();
            transactionManager.commit(status);
        } catch (Exception e) {
            logger.error("initDefaultDict error", e);
            if (status != null)
                try {
                    transactionManager.rollback(status);
                } catch (TransactionException ignored) {
                }
        }
    }

    @SuppressWarnings("unchecked")
    protected void initDefaultDict() {
        XStream sm = new XStream(new CDATASupportDom4JDriver());
        sm.alias("dict", Dict.class);
        sm.alias("item", Item.class);
        Reader reader = null;
        try {
            reader = new InputStreamReader(getClass().getResource(DEFAULT_DICT_FILE).openStream(), Constants.DEFAULT_CHARSET);
            Map<Dict, List<Item>> map = (Map<Dict, List<Item>>) sm.fromXML(reader);
            for (Map.Entry<Dict, List<Item>> entry : map.entrySet()) {
                Dict dict = entry.getKey();
                if (getDict(dict.getName()) == null) {
                    logger.warn("dict [{}] not found,create it", dict.getName());
                    save(dict);
                    for (Item item : entry.getValue()) {
                        item.setDictId(dict.getId());
                        saveItem(item);
                    }
                }
            }
        } catch (IOException e) {
            logger.error(e.getMessage(), e);
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException ignored) {
                }
            }
        }
    }
}
