package com.gtis.archive.service.impl;


import com.gtis.archive.core.cache.Cache;
import com.gtis.archive.entity.Resource;
import com.gtis.archive.service.ResourceService;
import com.gtis.support.hibernate.HibernateTemplate;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.orm.hibernate3.HibernateTransactionManager;
import org.springframework.transaction.TransactionException;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import java.util.*;

/**
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-8-13
 */
public class ResourceServiceImpl extends HibernateTemplate<Resource, String> implements ResourceService {
    private Cache cache;

    @Autowired
    private HibernateTransactionManager transactionManager;

    public void setCache(Cache cache) {
        this.cache = cache;
    }

    @Override
    public List<Resource> getChildrenResource(String parentId, String... name) {
        Resource res = getResource(parentId, name);
        if (res == null) {
            return Collections.emptyList();
        } else {
            return internalGetChildResources(res.getId());
        }
    }

    @Override
    public List<Resource> getChildrenResourceRecursively(String id, String... name) {
        List<Resource> children = getChildrenResource(id);
        for (Resource rs : children) {
            if (rs.getChildren() == null) {
                getChildrenResourceRecursively(rs.getId());
            }
        }
        return children;
    }

    @Override
    public Resource getResource(String id, String... name) {
        Resource res = id != null ? internalGetResource(id) : null;
        if (name == null) {
            return res;
        }
        for (String n : name) {
            res = internalGetChildResource(id, n);
            if (res == null) {
                return null;
            }
        }
        getSession().evict(res);
        return res;
    }

    @Override
    public Resource findChildResource(String id, String targetName, String... name) {
        Resource res = getResource(id, name);
        if (res != null) {
            return internalfindChildResource(res.getId(), targetName);
        }
        return null;
    }

    private Resource internalfindChildResource(String id, String targetName) {
        List<Resource> list = internalGetChildResources(id);
        for (Resource res : list) {
            if (res.getName().equals(targetName)) {
                return res;
            }
        }
        for (Resource res : list) {
            Resource res1 = internalfindChildResource(res.getId(), targetName);
            if (res1 != null) {
                return res1;
            }
        }
        return null;
    }

    @Override
    @Transactional
    public void removeResource(String id) {
        List<Resource> children = getChildrenResource(id);
        for (Resource rs : children) {
            removeResource(rs.getId());
            cache.remove(rs.getId());
        }
        Resource res = getResource(id);
        delete(id);
        cache.remove(id);
        if (res.getParentId() != null) {
            getResource(res.getParentId()).setChildren(null);
        }
    }

    @Override
    @Transactional
    public Resource saveResource(Resource resource) {
        save(resource);
        if (resource.getParentId() != null) {
            getResource(resource.getParentId()).setChildren(null);
        }
        return resource;
    }
    private String patentIdText = "parentId";
    @SuppressWarnings("unchecked")
    private Resource internalGetChildResource(String parentId, String name) {
        Resource res = internalGetResource(parentId);
        List<Resource> children = res == null ? null : res.getChildren();
        if (children != null) {
            for (Resource r : children) {
                if (r.getName().equals(name)) {
                    return r;
                }
            }
        } else {
            if (parentId == null) {
                res = cache.get(name);
                if (res != null) {
                    return res;
                }
            }
            children = (List<Resource>) createCriteria(parentId == null ? Restrictions.isNull(patentIdText) : Restrictions.eq(patentIdText, parentId), Restrictions.eq("name", name)).addOrder(Order.desc("orderNumber")).list();
            if (!children.isEmpty()) {
                res = children.get(0);
                cache.put(res);
                if (parentId == null) {
                    cache.put(name, res);
                }
                return res;
            }
        }
        return null;
    }

    private Resource internalGetResource(String id) {
        if (id == null) {
            return null;
        }
        Resource res = cache.get(id);
        if (res == null) {
            res = get(id);
            if (res != null) {
                getSession().evict(res);
                cache.put(res);
            }
        }
        return res;
    }

    @SuppressWarnings("unchecked")
    private List<Resource> internalGetChildResources(String parentId) {
        Resource res = internalGetResource(parentId);
        if (res != null) {
            List<Resource> children = res.getChildren();
            if (children != null) {
                return children;
            } else {
                children = (List<Resource>) createCriteria(Restrictions.eq(patentIdText, parentId)).addOrder(Order.desc("orderNumber")).list();
                res.setChildren(children);
                for (Resource res1 : children) {
                    cache.put(res1);
                }
                return children;
            }
        }
        return new LinkedList<Resource>();
    }

    public synchronized void init() {
        Resource resource = getResource(null, "model");
        if (resource != null) {
            return;
        }
        TransactionStatus status = null;
        try {
            status = transactionManager.getTransaction(new DefaultTransactionDefinition());
            resource = new Resource();
            resource.setName("model");
            resource.setTitle("模型根节点");
            save(resource);
            transactionManager.commit(status);
        } catch (TransactionException ex) {
            logger.error(ex.getLocalizedMessage());
            if (status != null) {
                try {
                    transactionManager.rollback(status);
                } catch (TransactionException e) {
                    logger.error(e.getMessage());
                }
            }
        }

    }

    /**
     * 根据名称获取资源
     * @param name 名称
     * @return
     */
    @Override
    public Resource getResourceByName(String name){
        String hql = "FROM Resource where name =:name";
        Map map = new HashMap();
        List<Resource> list = new ArrayList<Resource>();
        map.put("name", name);

        try {
            list = find(hql, map);
        } catch (Exception e) {
            logger.error("根据档号查询档案出现异常" + e.toString());
        }

        return list.get(0);
    }
}
