package com.gtis.archive.service.impl;

import com.gtis.archive.core.EntityService;
import com.gtis.archive.core.cache.Cache;
import com.gtis.archive.entity.Position;
import com.gtis.archive.service.ArchiveService;
import com.gtis.archive.service.PositionService;
import com.gtis.support.hibernate.HibernateTemplate;
import org.hibernate.Query;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * Created by Think on 2017/5/3.
 */
public class PositionServiceImpl extends HibernateTemplate<Position, String> implements PositionService {

    private Cache cache;

    @Autowired
    protected EntityService entityService;
    @Autowired
    protected ArchiveService archiveService;

    public void setCache(Cache cache) {
        this.cache = cache;
    }

    /**
     * 保存位置信息
     * @param position 位置信息实体类
     * @param insert 是否是新增
     * @param addRows 新增的行
     * @param addCols 新增的列
     * @return 位置实体类
     */
    @Override
    @Transactional
    public Position savePosition(Position position, Integer addRows, Integer addCols, boolean insert) {
        String key = position.getRoomId() + "-" + position.getCabinetSno();
        Map<String, Boolean> map = null;
        if (insert) {
            int size = position.getRowId() * position.getColumnId();
            map = new HashMap<String, Boolean>(size);
            for (int i = 1; i <= position.getRowId(); i++) {
                for (int j = 1; j <= position.getColumnId(); j++) {
                    map.put(i + "-" + j, false);
                }
            }
            cache.put(key, map);
        } else {
            map = cache.get(key);
            if (addRows > 0) {
                for (int i = addRows; i > (position.getRowId() - addRows); i--) {
                    for (int j = 1; j <= position.getColumnId(); j++) {
                        String cellKey = i + "-" + j;
                        if (!map.containsKey(cellKey)) {
                            map.put(cellKey, false);
                        }
                    }
                }
            }

            if (addCols > 0) {
                for (int i = addCols; i > (position.getColumnId() - addCols); i--) {
                    for (int j = 1; j <= position.getRowId(); j++) {
                        String cellKey = j + "-" + i;
                        if (!map.containsKey(cellKey)) {
                            map.put(cellKey, false);
                        }
                    }
                }
            }
        }

        getSession().saveOrUpdate(position);
        return position;
    }

    /**
     * 获取房间内的柜子数
     *
     * @param roomId 房间号
     * @return 房间的柜子数
     */
    @Override
    @SuppressWarnings("unchecked")
    public List<Position> getCabinetByRoom(String roomId) {
        List<Position> list = new ArrayList<Position>();
        Query query = getSession().createQuery("FROM Position p WHERE p.roomId=? order by p.cabinetSno");
        query.setString(0, roomId);
        list = query.list() == null ? list : query.list();
        return list;
    }

    /**
     * 根据房间号和柜号获取行数
     *
     * @param roomId 房间号
     * @param cabinetSno 鬼子号
     * @return 柜子的行数
     */
    @Override
    public Integer getRowByCabinet(String roomId, String cabinetSno) {
        Query query = getSession().createQuery("SELECT p.rowId FROM Position p WHERE p.roomId=? AND p.cabinetSno=?");
        query.setString(0, roomId);
        query.setString(1, cabinetSno);

        return query.uniqueResult() == null ? 0 : (Integer) query.uniqueResult();
    }

    /**
     * 根据房间号和柜号获取列数
     *
     * @param roomId 房间号
     * @param cabinetSno 柜子号
     * @return 柜子的列数
     */
    @Override
    public Integer getColumnByCabinet(String roomId, String cabinetSno) {
        Query query = getSession().createQuery("SELECT p.columnId FROM Position p WHERE p.roomId=? AND p.cabinetSno=?");
        query.setString(0, roomId);
        query.setString(1, cabinetSno);

        return query.uniqueResult() == null ? 0 : (Integer) query.uniqueResult();
    }

    /**
     * 根据房间号和柜号获取位置
     *
     * @param roomId 房间号
     * @param cabinetSno 柜子号
     * @return 获取柜子
     */
    @Override
    public Position getPositionByCabinet(String roomId, String cabinetSno) {
        Query query = getSession().createQuery("FROM Position p WHERE p.roomId=? AND p.cabinetSno=?");
        query.setString(0, roomId);
        query.setString(1, cabinetSno);
        return (Position) query.uniqueResult();
    }

    /**
     * 获取柜子中的所有格子是否有案卷
     *
     * @param p 柜子
     * @return
     */
    @Override
    public Map getCellHasArchives(Position p) {
        String key = p.getRoomId() + "-" + p.getCabinetSno();
        Map<String, Boolean> map = cache.get(key);
        if (map == null) {
            int flag;
            map = new HashMap<String, Boolean>();
            int row = p.getRowId();
            int col = p.getColumnId();
            for (int i = 1; i <= row; i++) {
                for (int j = 1; j <= col; j++) {
                    flag = archiveService.getArchivesByCell(p.getRoomId(), p.getCabinetSno(), i, j).size();
                    if (flag > 0) {
                        map.put(i + "-" + j, true);
                    } else {
                        map.put(i + "-" + j, false);
                    }
                }
            }
        }
        cache.put(key, map);
        return map;
    }

    /**
     * 更新柜子中的格子的状态
     *
     * @param p 柜子
     * @param isUp 是否是上架
     */
    @Override
    public void updateCabinetStatus(Position p, boolean isUp) {
        boolean b;
        Map<String, Boolean> map = cache.get(p.getRoomId() + "-" + p.getCabinetSno());
        if (isUp) {
            b = true;
        } else {
            int count = archiveService.getArchivesByCell(p.getRoomId(), p.getCabinetSno(), p.getRowId(), p.getColumnId()).size();
            if (count > 0) {
                b = true;
            } else {
                b = false;
            }
        }
        map.put(p.getRowId() + "-" + p.getColumnId(), b);
    }
}
