package com.gtis.archive.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.gtis.archive.Constants;
import com.gtis.archive.Switch;
import com.gtis.archive.core.ModelChangeEvent;
import com.gtis.archive.core.cache.Cache;
import com.gtis.archive.core.environment.EnvHolder;
import com.gtis.archive.entity.Permission;
import com.gtis.archive.service.PermissionService;
import com.gtis.common.Page;
import com.gtis.plat.vo.UserInfo;
import com.gtis.support.hibernate.HibernateTemplate;
import com.gtis.web.SessionUtil;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.hibernate.Criteria;
import org.hibernate.criterion.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationEventPublisher;
import org.springframework.context.ApplicationEventPublisherAware;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ResourceUtils;
import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.*;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-12-8
 */
public class PermissionServiceImpl extends HibernateTemplate<Permission, String> implements PermissionService, ApplicationEventPublisherAware {
    private Cache cache;
    private ApplicationEventPublisher publisher;
    private Logger logger = LoggerFactory.getLogger(ArchiveServiceImpl.class);
    private String resourceIdText = "resourceId";
    private String roleIdText = "roleId";
    private String operationText = "operation";
    private String resourceIdSql = "' AND t.resource_id='";
    public void setCache(Cache cache) {
        this.cache = cache;
    }

    @Override
    public void setApplicationEventPublisher(ApplicationEventPublisher publisher) {
        this.publisher = publisher;
    }

    /**
     * 获取权限配置的JSON
     *
     * @return
     */
    private String permissionJsonInfo = "permissionJson";
    private String leavesIdText = "leavesId";
    @Override
    public String getPermissionJson(String location) {
        logger.info("permission.json的位置:{}", location);
        String s;
        List<String> strings = new ArrayList<String>();
        s = cache.get(permissionJsonInfo);

        try {
            if (StringUtils.isBlank(s)) {
                s = readFileToString(location);
                JSONArray array = JSON.parseArray(s);
                getLeavesId(null, array, strings);
                cache.put(leavesIdText, strings);
                cache.put(permissionJsonInfo, s);
            }
        } catch (IOException e) {
            logger.error("无法读取permission.json文件");
        }

        return s;
    }

    @Override
    @Transactional
    public void grant(Permission[] permissions) {
        if (permissions == null) {
            return;
        }
        for (Permission perm : permissions) {
            if (hasPermission(perm)) {
                continue;
            }
            save(perm);
            cache.put(perm, true);
        }
        publisher.publishEvent(new ModelChangeEvent(permissions));
    }

    @Override
    @Transactional
    public void revoke(Permission[] permissions) {
        if (permissions == null) {
            return;
        }
        for (Permission perm : permissions) {
            if (!hasPermission(perm)) {
                continue;
            }
            delete(cache.<Permission>getKey(perm));
            cache.put(perm, false);
        }
        publisher.publishEvent(new ModelChangeEvent(permissions));
    }

    /**
     * 授权
     *
     * @param permissions 要授予权限列表
     */
    @Override
    @Transactional
    public void grant(List<Permission> permissions) {
        if (permissions == null || permissions.isEmpty()) {
            return;
        }

        for (Permission perm : permissions) {
            if (!"dwdm".equals(perm.getOperation()) && !"nd".equals(perm.getOperation()) && hasPermission(perm)) {
                continue;
            } else if ("dwdm".equals(perm.getOperation()) || "nd".equals(perm.getOperation())) {
                List<Permission> permissionList = getDwdmPermissions(perm.getResourceId(), perm.getRoleId(), perm.getDwdm(),perm.getOperation());
                if (permissionList == null || permissionList.isEmpty()) {
                    save(perm);
                }continue;
            }
            save(perm);
            cache.put(perm, true);
        }
        publisher.publishEvent(new ModelChangeEvent(permissions));
    }

    /**
     * 收回权限
     *
     * @param permissions 要收回的权限列表
     */
    @Override
    @Transactional
    public void revoke(List<Permission> permissions) {
        if (permissions == null || permissions.isEmpty()) {
            return;
        }

        for (Permission perm : permissions) {
            if (!"dwdm".equals(perm.getOperation()) && !"nd".equals(perm.getOperation()) && !hasPermission(perm)) {
                continue;
            } else if ("dwdm".equals(perm.getOperation()) || "nd".equals(perm.getOperation())) {
                List<Permission> permissionList = getDwdmPermissions(perm.getResourceId(), perm.getRoleId(), perm.getDwdm(),perm.getOperation());
                if (!permissionList.isEmpty()) {
                    for (Permission p : permissionList) {
                        delete(p);
                    }
                }
                continue;
            }
            delete(cache.<Permission>getKey(perm));
            cache.put(perm, false);
        }
        publisher.publishEvent(new ModelChangeEvent(permissions));
    }

    @Override
    public boolean hasPermission(String resourceId, String operation) {
        UserInfo user = SessionUtil.getCurrentUser();
        if (user.isAdmin()) {
            return true;
        }
        String roleIds = user.getRoleIds();
        if (roleIds == null) {
            return false;
        }
        return hasPermission(new Permission(resourceId,roleIds,operation));
    }

    @Override
    @SuppressWarnings("unchecked")
    public List<Permission> getPermissions(String resourceId, String roleId) {
        Criteria criteria = createCriteria();
        if (StringUtils.isNotBlank(resourceId)) {
            criteria.add(Restrictions.eq(resourceIdText, resourceId));
        }
        if (StringUtils.isNotBlank(roleId)) {
            String[] roles = roleId.split(",");
            criteria.add(Restrictions.in(roleIdText, roles));
        }

        return criteria.list();
    }

    @Override
    @SuppressWarnings("unchecked")
    public List<Permission> getPermissions(String resourceId, String roleId, String operation) {
        Criteria criteria = createCriteria();
        criteria.setResultTransformer(Criteria.DISTINCT_ROOT_ENTITY);

        if (StringUtils.isNotBlank(resourceId)) {
            criteria.add(Restrictions.eq(resourceIdText, resourceId));
        }
        if (StringUtils.isNotBlank(operation)) {
            criteria.add(Restrictions.eq(operationText, operation));
        }
        if (StringUtils.isNotBlank(roleId)) {
            List<String> roles = Arrays.asList(roleId.split(","));
            Disjunction dis = Restrictions.disjunction();
            for (String role:roles) {
                dis.add(Restrictions.eq(roleIdText, role));
            }
            criteria.add(dis);
        }

        return criteria.list();
    }

    /**
     * 判断用户对某业务是否拥有权限
     *
     * @param resourceId
     * @return
     */
    @Override
    public boolean hasResPermission(String resourceId) {
        UserInfo user = SessionUtil.getCurrentUser();
        boolean b = false;
        if (user.isAdmin()) {
            b = true;
        } else {
            if (!EnvHolder.isEnable(Switch.isNewPermission)) {
                String roleIds = user.getRoleIds();
                if (roleIds != null) {
                    b = hasResPermission(roleIds, resourceId);
                }
            }else {
                String userId = user.getId();
                if (userId != null){
                    b = hasResPermissionNew(userId,resourceId);
                }
            }
        }
        return b;
    }

    @Override
    public Set<String> getReadableRoleIds(String resourceId) {
        Set<String> set = new HashSet<String>();
        for (Permission p : search(0, Page.ALL, new Order[]{}, Restrictions.eq(resourceIdText, resourceId), Restrictions.eq(operationText, Permission.VIEW_PERMISSION)).getItems()) {
            set.add(p.getRoleId());
        }
        return set;
    }

    /**
     * 获取权限条目
     *
     * @return
     */
    @Override
    public List<String> getLeavesId(String location) throws IOException {
        List<String> strings = null;
        JSONArray array = null;
        if (cache.get(leavesIdText) != null) {
            strings = cache.get(leavesIdText);
        } else if (cache.get(permissionJsonInfo) != null) {
            String s = cache.get(permissionJsonInfo);
            strings = new ArrayList<String>();
            array = JSON.parseArray(s);
            getLeavesId(null, array, strings);
            cache.put(leavesIdText, strings);
        } else {
            logger.info("文件位置:{}", location);
            String s = readFileToString(location);
            strings = new ArrayList<String>();
            array = JSONArray.parseArray(s);
            getLeavesId(null, array, strings);
            cache.put(permissionJsonInfo, s);
            cache.put(leavesIdText, strings);
        }
        return strings;
    }

    /**
     * 根据单位代码， 角色， 业务获取权限
     * @param resourceId
     * @param roleId
     * @return
     */
    @Override
    public List<Permission> getDwdmPermissions(String resourceId, String roleId, String dwdm, String operation) {
        Criteria criteria = createCriteria();
        if (StringUtils.isNotBlank(resourceId) && StringUtils.isNotBlank(roleId) && StringUtils.isNotBlank(dwdm)) {
            criteria.add(Restrictions.eq(resourceIdText, resourceId));
            criteria.add(Restrictions.eq(roleIdText, roleId));
            criteria.add(Restrictions.eq(operationText, operation));
            criteria.add(Restrictions.eq("dwdm", dwdm));
        }
        return criteria.list();
    }

    /**
     * 获取用户可访问的业务类型
     * @param resourceId
     * @param roleId
     * @return
     */
    @Override
    public List<Permission> getDwdmPermissions(String resourceId, String roleId, String operation) {
        Criteria criteria = createCriteria();
        if (StringUtils.isNotBlank(resourceId) && StringUtils.isNotBlank(roleId)) {
            criteria.add(Restrictions.eq(resourceIdText, resourceId));
            criteria.add(Restrictions.eq(roleIdText, roleId));
            criteria.add(Restrictions.eq(operationText, operation));
        }
        return criteria.list();
    }

    /**
     * 获取所有权限用户
     * @return
     */
    public List<String> getUserId(){
        List<String> list = getSession().createSQLQuery("select distinct(t.resource_id) from T_PERMISSION t").list();
        if (list == null || list.isEmpty()) {
            return null;
        }

        return list;
    }

    private Boolean hasPermission(Permission perm) {
        Boolean exist = cache.get(perm);
        if (exist == null) {
            Boolean aBoolean = hasResPermission(perm.getRoleId().toString(),perm.getResourceId(),perm.getOperation());
            if (aBoolean) {
                if (!"dwdm".equals(perm.getOperation()) && !"nd".equals(perm.getOperation())) {
                    cache.put(perm, true);
                }
            }
            exist = cache.get(perm);
            if (exist == null) {
                cache.put(perm);
                exist = false;
            }
        }
        return exist;
    }

    /**
     * 获取json中的叶子节点的id
     * @param map
     * @param jsonArray
     * @param list
     */
    private static void getLeavesId(Map map, JSONArray jsonArray, List list) {
        String childrenText = "children";
        if (map != null && map.containsKey(childrenText) && jsonArray == null) {
            getLeavesId(null, (JSONArray) map.get(childrenText), list);
        } else if (map != null && !map.containsKey(childrenText) && jsonArray == null) {
            if (map.containsKey("id")) {
                list.add(map.get("id"));
            }
        } else if (map == null && jsonArray != null) {
            for (int i = 0; i < jsonArray.size(); i++) {
                getLeavesId((Map) jsonArray.get(i), null, list);
            }
        }
    }

    /**
     * 读取资源文件
     * @param location
     * @return
     * @throws IOException
     */
    private static String readFileToString(String location) throws IOException {
        String s;
        if (StringUtils.isNotBlank(location)) {
            File file = ResourceUtils.getFile(location);
            s = FileUtils.readFileToString(file, Constants.DEFAULT_CHARSET);
            s = s.replaceAll("\\s*", "");
        } else {
            throw new IOException("传入的地址为空");
        }
        return s;
    }

    /**
     * 判断角色是否拥某资源的权限
     *
     * @param roleIds
     * @param resourceId
     * @return
     */
    private boolean hasResPermission(String roleIds, String resourceId) {
        String s = roleIds.replaceAll("\'", "");
        logger.info(s);
        String sql = "SELECT count(1) FROM t_permission t WHERE t.role_id ='" + s + resourceIdSql + resourceId + "'";
        if(roleIds.split(",").length>1) {
            sql = "SELECT count(1) FROM t_permission t WHERE t.role_id in (" + roleIds + ") AND t.resource_id='" + resourceId + "'";
        }
        BigDecimal count = (BigDecimal) getSession().createSQLQuery(sql).uniqueResult();
        return (count.intValue() != 0);
    }

    /**
     * 判断角色是否拥某资源的功能权限
     *
     * @param roleIds
     * @param resourceId
     * @return
     */
    private boolean hasResPermission(String roleIds, String resourceId, String operation) {
        String s = roleIds.replaceAll("\'", "");
        String sql = "SELECT count(1) FROM t_permission t WHERE t.role_id= '" + s + "' AND t.operation='" + operation + resourceIdSql + resourceId + "'";
        if(roleIds.split(",").length>1) {
            sql = "SELECT count(1) FROM t_permission t WHERE t.role_id in (" + roleIds + ") AND t.operation='" + operation + resourceIdSql + resourceId + "'";
        }
        BigDecimal count = (BigDecimal) getSession().createSQLQuery(sql).uniqueResult();
        return (count.intValue() != 0);
    }

    /**
     * 获取所有权限用户
     * @return
     */
    public List<Permission> getUsers(){
        Criteria criteria = createCriteria();
        criteria.add(Restrictions.isNotNull("userId"));
        criteria.add(Restrictions.sqlRestriction("rowid in (select min(a.rowid) from t_permission a group by a.USER_ID)"));
        logger.error(criteria.toString());
        return criteria.list();
    }

    /**
     * 获取用户本业务权限
     * @return
     */
    public List<Permission> getUserPermissions(String resourceId, String userId) {
        Criteria criteria = createCriteria();
        if (StringUtils.isNotBlank(resourceId)) {
            criteria.add(Restrictions.eq(resourceIdText, resourceId));
        }
        if (StringUtils.isNotBlank(userId)) {
            criteria.add(Restrictions.eq("userId", userId));
        }
        return criteria.list();
    }

    public void delete(String userId,String resIds){
        try {
            if (resIds.equalsIgnoreCase("") || resIds.equalsIgnoreCase("null")) {
                batchExecute("delete from Permission where userId=?", userId);
                logger.info("delete " + userId);
            } else {
                for (String resId:resIds.split(","))
                batchExecute("delete from Permission t where t.userId=? and t.resourceId=?", userId, resId);
                logger.error("delete " + userId + " + " + resIds);
            }
        }catch (Exception e){
            logger.error(e.getMessage());
        }
    }

    /**
     * 授权
     *
     * @param permissions 要授予权限列表
     */
    @Transactional
    public void grantNew(List<Permission> permissions) {
        if (permissions != null && !permissions.isEmpty()) {
            for (Permission perm : permissions) {
                if ("dwdm".equals(perm.getOperation()) || "nd".equals(perm.getOperation())) {
                   List<Permission> permissionList = getDwdmPermissionsNew(perm.getResourceId(), perm.getUserId(), perm.getDwdm(),perm.getOperation());
                    if (permissionList == null || permissionList.isEmpty()) {
                        save(perm);
                    }
                    continue;
                }
                if ("sys".equals(perm.getResourceId())) {
                    List<Permission> permissionList = getSysPermissionsNew(perm.getResourceId(), perm.getUserId(), perm.getOperation());
                    if (permissionList == null || permissionList.isEmpty()) {
                        save(perm);
                    }
                    continue;
                }
                save(perm);
            }
            publisher.publishEvent(new ModelChangeEvent(permissions));
        }
    }

    /**
     *判断是否已有权限
     * @param perm 要授予权限列表
     */
    private Boolean hasPermissionNew(Permission perm) {
        Boolean exist = false;
        try {
            exist = cache.get(perm);
            if (exist == null) {
                Boolean aBoolean = hasResPermissionNew(perm.getUserId().toString(), perm.getResourceId(), perm.getOperation());
                if (aBoolean) {
                    if (!"dwdm".equals(perm.getOperation()) && !"nd".equals(perm.getOperation())) {
                        cache.put(perm, true);
                    }
                }
                exist = cache.get(perm);
                if (exist == null) {
                    cache.put(perm);
                    exist = false;
                }
            }
        }catch (Exception e){
            logger.error(e.getMessage());
        }
        return exist;
    }

    /**
     * 判断角色是否拥某资源的功能权限
     *
     * @param userIds
     * @param resourceId
     * @return
     */
    private boolean hasResPermissionNew(String userIds, String resourceId, String operation) {
        String s = userIds.replaceAll("\'", "");
        String sql = "SELECT count(1) FROM t_permission t WHERE t.user_id= '" + s + "' AND t.operation='" + operation + resourceIdSql + resourceId + "'";
        if(userIds.split(",").length>1) {
            sql = "SELECT count(1) FROM t_permission t WHERE t.user_id in (" + userIds + ") AND t.operation='" + operation + resourceIdSql + resourceId + "'";
        }
        BigDecimal count = (BigDecimal) getSession().createSQLQuery(sql).uniqueResult();
        return (count.intValue() != 0);
    }

    /**
     * 判断用户是否拥某资源的权限
     *
     * @param userIds
     * @param resourceId
     * @return
     */
    private boolean hasResPermissionNew(String userIds, String resourceId) {
        String s = userIds.replaceAll("\'", "");
        logger.info("hasResPermissionNew方法获取userId:{}",s);
        String sql = "SELECT count(1) FROM t_permission t WHERE t.user_id ='" + userIds + resourceIdSql + resourceId + "'";
        if(userIds.split(",").length>1) {
            sql = "SELECT count(1) FROM t_permission t WHERE t.user_id in (" + userIds + ") AND t.resource_id='" + resourceId + "'";
        }
        BigDecimal count = (BigDecimal) getSession().createSQLQuery(sql).uniqueResult();
        return (count.intValue() != 0);
    }

    /**
     * 根据单位代码， 角色， 业务获取权限
     * @param resourceId
     * @param userId
     * @return
     */
    private List<Permission> getDwdmPermissionsNew(String resourceId, String userId, String dwdm, String operation) {
        Criteria criteria = createCriteria();
        if (StringUtils.isNotBlank(resourceId) && StringUtils.isNotBlank(userId) && StringUtils.isNotBlank(dwdm)) {
            criteria.add(Restrictions.eq(resourceIdText, resourceId));
            criteria.add(Restrictions.eq("userId", userId));
            criteria.add(Restrictions.eq(operationText, operation));
            criteria.add(Restrictions.eq("dwdm", dwdm));
        }
        return criteria.list();
    }

    /**
     * 根据单位代码， 角色， 业务获取权限
     * @param resourceId
     * @param userId
     * @return
     */
    private List<Permission> getSysPermissionsNew(String resourceId, String userId, String operation) {
        Criteria criteria = createCriteria();
        if (StringUtils.isNotBlank(resourceId) && StringUtils.isNotBlank(userId) && StringUtils.isNotBlank(operation)) {
            criteria.add(Restrictions.eq(resourceIdText, resourceId));
            criteria.add(Restrictions.eq("userId", userId));
            criteria.add(Restrictions.eq(operationText, operation));
        }
        return criteria.list();
    }

    /**
     * 获取用户可访问的数据
     * @param resourceId
     * @param userId
     * @return
     */
    public List<Permission> getDwdmPermissionsNew(String resourceId, String userId, String operation) {
        Criteria criteria = createCriteria();
        if (StringUtils.isNotBlank(resourceId) && StringUtils.isNotBlank(userId)) {
            criteria.add(Restrictions.eq(resourceIdText, resourceId));
            criteria.add(Restrictions.eq("userId", userId));
            criteria.add(Restrictions.eq(operationText, operation));
        }
        return criteria.list();
    }

    /**
     * 判断用户是否拥某资源的功能权限
     *
     * @param userId
     * @param resourceId
     * @return
     */
    public boolean hasUserResPermission(String userId, String resourceId, String operation) {
        String sql = "SELECT count(1) FROM t_permission t WHERE t.user_id= '" + userId + "' AND t.operation='" + operation + resourceIdSql + resourceId + "'";
        BigDecimal count = (BigDecimal) getSession().createSQLQuery(sql).uniqueResult();
        return (count.intValue() != 0);
    }
}
