package com.gtis.archive.service.impl;

import com.gtis.archive.core.EntityService;
import com.gtis.archive.entity.Archive;
import com.gtis.archive.entity.RecordAjhRange;
import com.gtis.archive.service.ArchiveService;
import com.gtis.archive.service.FondsManagerService;
import com.gtis.archive.service.LogManagerService;
import com.gtis.archive.service.RecordAjhRangeService;
import com.gtis.common.Page;
import com.gtis.support.hibernate.HibernateTemplate;
import com.gtis.web.SessionUtil;
import org.hibernate.Query;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * 全宗卷业务处理类
 * Created by linlong on 2017/6/15.
 */
@Service
public class FondsManagerServiceImpl extends HibernateTemplate<Archive, String> implements FondsManagerService {

    @Autowired
    private EntityService entityService;
    @Autowired
    private ArchiveService archiveService;
    @Autowired
    private RecordAjhRangeService recordAjhRangeService;
    @Autowired
    private LogManagerService logManagerService;

    /**
     * 保存全宗卷
     *
     * @param fondsManager 要保存的全宗卷
     * @return 保存的全宗卷
     */
    @Override
    @Transactional
    public Archive saveFondsManager(Archive fondsManager) {
        getSession().save(fondsManager);
        return fondsManager;
    }

    /**
     * 更新全宗卷
     *
     * @param fondsManager
     * @return
     */
    @Override
    @Transactional
    public Archive updateFondsManager(Archive fondsManager) {
        getSession().update(fondsManager);
        return fondsManager;
    }

    /**
     * 删除全宗卷
     *
     * @param ids       要删除的全宗卷的id数组
     */
    @Override
    @Transactional
    public void removeFondsManager(String[] ids, String modelName) {
        for (String id : ids) {
            RecordAjhRange recordAjhRange = new RecordAjhRange();
            Archive archive = getFondsManager(id);
            recordAjhRange.setBgqx(archive.getBgqx());
            recordAjhRange.setAjh(archive.getAjh());
            recordAjhRange.setModelName(archive.getModelName());
            recordAjhRange.setMlh(archive.getModelName());
            recordAjhRange.setDwdm(archive.getDwdm());
            recordAjhRangeService.saveRecordAjh(recordAjhRange);
            Archive fondsManager = archiveService.getArchive(modelName, id);
            delete(id);
            logManagerService.saveArchiveLogManager(SessionUtil.getCurrentUserId(), SessionUtil.getCurrentUser().getUsername(), fondsManager, "delete");
        }
    }

    /**
     * 根据业务名和id获取全宗卷
     *
     * @param id  全宗卷id
     * @return 获取的全宗卷
     */
    @Override
    public Archive getFondsManager(String id) {
        return entityService.load(Archive.class, id);
    }

    /**
     * 查找全宗卷
     *
     * @param modelName  业务名
     * @param criterions 查找语句
     * @param orders     字段排序
     * @param start      起始
     * @param limit      页面大小
     * @return 全宗卷页面
     */
    @Override
    public Page<Archive> searchFondsManager(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit) {
        List<Criterion> list = new ArrayList<Criterion>(criterions);
        return entityService.search(modelName, list, orders == null ? Collections.singletonList(Order.desc("id")) : orders, start, limit);
    }

    /**
     * 获取某个业务的最大值
     *
     * @param modelName 业务名
     * @return 顺序号的最大值
     */
    @Override
    public Integer generateSxh(String modelName) {
        RecordAjhRange recordAjhRange = recordAjhRangeService.getAvailable(modelName, modelName, SessionUtil.getCurrentUser().getRegionCode());
        if (recordAjhRange != null) {
            recordAjhRangeService.remove(recordAjhRange.getId());
            return recordAjhRange.getAjh();
        } else {
            Query query = getSession().createQuery("SELECT MAX(ajh) FROM "+ modelName);
            return (query.uniqueResult() == null) ? 1 : (Integer.parseInt(query.uniqueResult().toString()) + 1);
        }
    }

    /**
     * 通过id获取全宗卷对象
     *
     * @param id
     * @return
     */
    @Override
    public Archive getFondsManagerById(String modelName, String id) {
        return archiveService.getArchive(modelName, id);
    }
}
