package com.gtis.archive.service.impl;

import com.gtis.archive.core.environment.EnvHolder;
import com.gtis.archive.entity.Original;
import com.gtis.archive.service.ArchiveService;
import com.gtis.archive.service.FileToPicService;
import com.gtis.archive.service.OriginalService;
import com.jacob.activeX.ActiveXComponent;
import com.jacob.com.ComThread;
import com.jacob.com.Dispatch;
import com.jacob.com.Variant;
import com.sun.image.codec.jpeg.JPEGCodec;
import com.sun.image.codec.jpeg.JPEGEncodeParam;
import com.sun.image.codec.jpeg.JPEGImageEncoder;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.NameValuePair;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.icepdf.core.pobjects.Document;
import org.icepdf.core.util.GraphicsRenderingHints;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.List;

/**
 * 将其他文件转换成图片
 * @author wangxinyi
 * @date 2018/10/08
 */
@Service
public class FileToPicServiceImpl implements FileToPicService {

    private static final Logger logger = LoggerFactory.getLogger(FileToPicServiceImpl.class);

    private static final String PDF_SUFFIX = "_pdf.pdf";

    @Autowired
    private OriginalService originalService;
    @Autowired
    private ArchiveService archiveService;

    /**
     * 将excel转换为图片
     * @param excelPath excel路径
     * @param path      图片路径
     * @param fileName  文件名
     * @param ownerId
     * @param ownerModelName
     */
    @Override
    public int excelToJpg(String excelPath, String path, String fileName, String ownerId, String ownerModelName) {
        String pdfPath = excelPath + PDF_SUFFIX;
        excelToPdf(excelPath, pdfPath);

        return pdfToJpg(pdfPath, path, fileName, ownerId, ownerModelName);
    }

    /**
     * 将excel转成pdf
     *
     * @param excelPath 源文件
     * @param pdfPath   目标文件
     */
    private String visible = "Visible";
    @Override
    public void excelToPdf(String excelPath, String pdfPath) {
        ActiveXComponent app = new ActiveXComponent("Excel.Application");
        try {
            app.setProperty(visible, false);
            Dispatch workbooks = app.getProperty("Workbooks").toDispatch();
            logger.info("打开文档, {}", excelPath);
            Dispatch workbook = Dispatch.invoke(workbooks, "Open", Dispatch.Method, new Object[]{excelPath, new Variant(false), new Variant(false)}, new int[3]).toDispatch();
            Dispatch.invoke(workbook, "SaveAs", Dispatch.Method, new Object[]{
                    pdfPath, new Variant(57), new Variant(false),
                    new Variant(57), new Variant(57), new Variant(false),
                    new Variant(true), new Variant(57), new Variant(true),
                    new Variant(true), new Variant(true)}, new int[1]);
            Variant f = new Variant(false);
            logger.info("转换文档到PDF, {} ", pdfPath);
            Dispatch.call(workbook, "Close", f);
            logger.info("转换完成");
        } catch (Exception e) {
            logger.error("========Error:文档转换失败: {}", e.getMessage());
            logger.error("转换失败");
        } finally {
            app.invoke("Quit", new Variant[]{});
        }
    }

    /**
     * 将word转换成pdf
     *
     * @param docPath doc文件路径
     * @param pdfPath pdf文件路径
     */
    @Override
    public void wordToPdf(String docPath, String pdfPath) {
        // 启动word应用程序(Microsoft Office Word 2003)
        logger.info("========启动word================");
        ActiveXComponent app = new ActiveXComponent("Word.Application");
        try {
            ComThread.InitSTA();
            if (app.m_pDispatch == 0) {
                app = new ActiveXComponent("Word.Application");
                app.setProperty(visible, new Variant(false));
                app.setProperty("DisplayAlerts", new Variant(false));
            }
            app.setProperty(visible, false);
            logger.info("============word启动成功========");
            logger.info("*****正在转换...*****");
            Dispatch docs = app.getProperty("Documents").toDispatch();
            File diskFile = new File(docPath);
            if (!diskFile.exists()) {
                logger.error("文件未找到, {}", docPath);
                throw new FileNotFoundException("文件未找到!");
            }
            Dispatch doc = Dispatch.call(docs, "Open", docPath, false, true).toDispatch();
            File tofile = new File(pdfPath);
            if (tofile.exists() && !tofile.delete()) {
                    logger.error("upload方法删除临时文件失败！" + tofile.getPath());
            }
            Dispatch.call(doc, "ExportAsFixedFormat", pdfPath, 17);
            Dispatch.call(doc, "Close", false);
            logger.info("转换完成");
        } catch (Exception e) {
            logger.error("========Error:文档转换失败：" + e.getMessage(),e);
            ComThread.Release();
        } finally {
            app.invoke("Quit", 0);
            ComThread.Release();
        }
    }

    /**
     * 将word转换为图片
     * @param docPath  word路径
     * @param path     文件转换后的路径
     * @param fileName 文件名
     * @param ownerId
     * @param ownerModelName
     */
    @Override
    public int wordToJpg(String docPath, String path, String fileName, String ownerId, String ownerModelName) {
        String pdfPath = docPath + PDF_SUFFIX;
        logger.info("doc转pdf路径为, {}", pdfPath);
        wordToPdf(docPath, pdfPath);
        return pdfToJpg(pdfPath, path, fileName, ownerId, ownerModelName);
    }

    /**
     * 将ceb文件转为图片
     * @param fileName 源文件
     * @param fileUrl 文件路径
     * @param archiveId
     * @param modelName
     */
    @Override
    public void cebToJpg(String fileName, String fileUrl, String archiveId, String modelName) {

        if (StringUtils.isBlank(fileName) && !fileName.endsWith(".ceb") && StringUtils.isBlank(fileUrl)) {
            logger.error("源文件获取错误");
            logger.error("源文件获取错误");
        }

        String url = EnvHolder.getSysProps().get("archive.url");
        String ip = null;
        logger.info("url = {}", url);
        if (url.contains(":")) {
            ip = url.split(":")[1];
        }

        File src = new File(fileName);
        url = "http:" + ip + ":8800";
        HttpClient client = new HttpClient();
        GetMethod method = new GetMethod(url);
        try {
            method.setQueryString(new NameValuePair[]{new NameValuePair("filename", src.getName()),
                    new NameValuePair("srcfile", URLEncoder.encode(fileUrl, "UTF-8")),
                    new NameValuePair("uploadurl", URLEncoder.encode(url + "/gateway!getCebPic.action?archiveId=" + archiveId + "&modelName=" + modelName, "UTF-8"))});
            client.executeMethod(method);
            logger.info(method.getResponseBodyAsString());
            Thread.currentThread().sleep(8000);
        } catch (IOException e) {
            logger.error("转换错误, {}", e.getMessage());
        } catch (InterruptedException e) {
            logger.error("转换错误, {}", e.getMessage());
            Thread.currentThread().interrupt();
        } finally {
                method.releaseConnection();
        }
    }

    /**
     * 将pdf转换为jpg
     *
     * @param pdfPath  pdf路径
     * @param path     路径
     * @param fileName 文件名
     * @param ownerId
     * @param ownerModelName
     * @return pdf的页数
     */
    @Override
    public int pdfToJpg(String pdfPath, String path, String fileName, String ownerId, String ownerModelName) {
        Document document = new Document();
        FileOutputStream fos = null;
        int pageNum = 0;
        int size = 0;

        try {
            logger.info("pdf path = {}", pdfPath);
            logger.info("fileName = {}", fileName);
            document.setFile(pdfPath);
            pageNum = document.getNumberOfPages();
            List<Original> originals = originalService.getOriginals(ownerId);
            size = originals == null ? 0 : originals.size();
            logger.info("pageNum = {}", pageNum);
            for (int i = 0; i < pageNum; i++) {
                BufferedImage image = (BufferedImage)
                        document.getPageImage(i, GraphicsRenderingHints.SCREEN, org.icepdf.core.pobjects.Page.BOUNDARY_CROPBOX, 0f, 2.5f);
                int j = size + i + 1;
                String imgName = fileName + "-" + addPrefixZero(4, j) + ".jpg";
                fos = new FileOutputStream(path + File.separator + imgName);
                JPEGImageEncoder jpegEncoder = JPEGCodec.createJPEGEncoder(fos);
                JPEGEncodeParam jpegEncodeParam = jpegEncoder.getDefaultJPEGEncodeParam(image);
                jpegEncodeParam.setDensityUnit(JPEGEncodeParam.DENSITY_UNIT_DOTS_INCH);
                jpegEncodeParam.setXDensity(300);
                jpegEncodeParam.setYDensity(300);
                jpegEncoder.encode(image, jpegEncodeParam);
                IOUtils.closeQuietly(fos);
                image.flush();
            }
        } catch (Exception e) {
            logger.error("获取pdf失败,{}", e.getMessage());

            Thread.currentThread().interrupt();
        } finally {
            document.dispose();
        }
        return pageNum + size;
    }

    /**
     * 生成表格图片
     *
     * @param cellValue  单元格的值
     * @param path
     * @param fontSize   字体大小
     * @param imageWidth 图像宽度
     * @param rowHeight  行高度
     */
    @Override
    public void generateTableGraphics(String[][] cellValue, String path, int fontSize, int imageWidth, int rowHeight) {
//        fontSize = 32;
//        imageWidth = 504;
//        rowHeight = 81;
        // 横线的行数
        int totalRow = cellValue.length;
        // 竖线的个数
        int totalCol = 0;

        if (cellValue[0] != null) {
            totalCol = cellValue[0].length;
        }

        if (imageWidth <= 0) {
            throw new IllegalArgumentException("图片宽度不能小于等于0");
        }

        if (rowHeight <= 0) {
            throw new IllegalArgumentException("表格行高度不能小于等于0");
        }

        if (fontSize <= 0) {
            throw new IllegalArgumentException("字体大小不能小于等于0");
        }

        // 图片高度
        int imageHeight = totalRow * rowHeight + 10;
        // 起始高度
        int startHeight = 0;
        // 起始宽度
        int startWidth = 0;
        // 单元格的列宽度
        int colWidth = (imageWidth - 10) / totalCol;
        BufferedImage image = new BufferedImage(imageWidth, imageHeight, BufferedImage.TYPE_INT_RGB);
        Graphics2D graphics = image.createGraphics();
        image = graphics.getDeviceConfiguration().createCompatibleImage(imageWidth, imageHeight, Transparency.TRANSLUCENT);
        graphics = image.createGraphics();
        graphics.setStroke(new BasicStroke(3.0f));
        graphics.fillRect(0, 0, imageWidth, imageHeight);

        // 画横线
        for (int i = 0; i <= totalRow; i++) {
            graphics.setColor(Color.RED);
            graphics.drawLine(startWidth, startHeight + i * rowHeight, startWidth + colWidth * totalCol, startHeight + i * rowHeight);
        }

        // 画竖线
        for (int j = 0; j <= totalCol; j++) {
            if(j == 0){
                graphics.setColor(Color.RED);
                graphics.drawLine(startWidth, startHeight, startWidth, startHeight + rowHeight * totalRow);
            }else if(j == 1){
                graphics.setColor(Color.RED);
                graphics.drawLine(startWidth + (imageWidth-10)*3/10, startHeight, startWidth + (imageWidth-10)*3/10, startHeight + rowHeight * totalRow);
            }else if(j == 2){
                graphics.setColor(Color.RED);
                graphics.drawLine(startWidth + (imageWidth-10)*7/10, startHeight, startWidth + (imageWidth-10)*7/10, startHeight + rowHeight * totalRow);
            }else {
                graphics.setColor(Color.RED);
                graphics.drawLine(imageWidth-10, startHeight, imageWidth-10, startHeight + rowHeight * totalRow);
            }
        }

        Font font = new Font("宋体", Font.PLAIN, fontSize);
        for (int k = 0; k < cellValue.length; k++) {
            for (int l = 0; l < cellValue[k].length; l++) {
                if(l == 0){
                    graphics.setFont(font);
                    graphics.setColor(Color.RED);
                    graphics.drawString(cellValue[k][l], startWidth + colWidth * l + 29, startHeight + rowHeight * k + 37);
                }else if(l == 1){
                    graphics.setFont(font);
                    graphics.setColor(Color.RED);
                    graphics.drawString(cellValue[k][l], startWidth + (imageWidth-10)*3/10 + 29, startHeight + rowHeight * k + 37);
                }else if(l == 2){
                    graphics.setFont(font);
                    graphics.setColor(Color.RED);
                    graphics.drawString(cellValue[k][l], startWidth + (imageWidth-10)*7/10 + 29, startHeight + rowHeight * k + 37);
                }else {
                    graphics.setFont(font);
                    graphics.setColor(Color.RED);
                    graphics.drawString(cellValue[k][l], startWidth + colWidth * l + 25, startHeight + rowHeight * (k + 1) - 30);
                    graphics.drawString(cellValue[k][l], startWidth, startHeight);
                }
            }
        }

        createImage(image, path, "png");
    }

    /**
     * 生成电子签名
     *
     * @param src         被签名文件
     * @param digitalFile 电子签名文件
     * @param alpha
     * @return
     */
    @Override
    public BufferedImage digitalSign(File src, File digitalFile, float alpha) {
        // 获取底图
        BufferedImage buffImg = null;
        BufferedImage waterImg = null;
        Graphics2D g2d = null;
        try {
            // 获取层图
            buffImg = ImageIO.read(src);
            waterImg = ImageIO.read(digitalFile);
            // x轴起始位置
            int x = (buffImg.getWidth() - waterImg.getWidth()) / 2;
            // y轴起始位置
            int y = 50;
            // 创建Graphics2D对象，用在底图对象上绘图
            g2d = buffImg.createGraphics();
            // 获取层图的宽度
            int waterImgWidth = waterImg.getWidth();
            // 获取层图的高度
            int waterImgHeight = waterImg.getHeight();
            // 在图形和图像中实现混合和透明效果
            g2d.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_ATOP, alpha));
            // 绘制
            g2d.drawImage(waterImg, x, y, waterImgWidth, waterImgHeight, null);
        } catch (IOException e) {
            logger.error("生成签名图片错误, {}", e.getMessage());
            logger.error("生成签名图片错误");
        } finally {
            // 释放图形上下文使用的系统资源
            if (g2d != null) {
                g2d.dispose();
            }
        }
        return buffImg;
    }

    /**
     * 合成签名图片
     *
     * @param bi       签名图片
     * @param savePath 保存路径
     */
    @Override
    public void generateSignFile(BufferedImage bi, String savePath) {

        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(savePath);
            JPEGImageEncoder jpegEncoder = JPEGCodec.createJPEGEncoder(fos);
            JPEGEncodeParam jpegEncodeParam = jpegEncoder.getDefaultJPEGEncodeParam(bi);
            jpegEncodeParam.setDensityUnit(JPEGEncodeParam.DENSITY_UNIT_DOTS_INCH);
            jpegEncodeParam.setXDensity(300);
            jpegEncodeParam.setYDensity(300);
            jpegEncoder.encode(bi, jpegEncodeParam);
        } catch (IOException e) {
            logger.error("生成图片错误, {}", e.getMessage());
            logger.error("生成图片错误");
        } finally {
            IOUtils.closeQuietly(fos);
        }
    }

    /**
     * 生成图片
     *
     * @param image    图片
     * @param location 地址
     */
    private void createImage(BufferedImage image, String location, String type) {
        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(location);
            ImageIO.write(image, type, fos);
        } catch (IOException e) {
            logger.error("创建图片出错, {}", e.getMessage());
            logger.error("创建图片出错");
        } finally {
            IOUtils.closeQuietly(fos);
        }
    }



    /**
     * 数字补齐0
     * @param count 位数
     * @param i 数字
     * @return 补齐后的字符串
     */
    private String addPrefixZero(int count, Integer i) {
        if (i == null || count <= 0) {
            return null;
        }
        return String.format("%0" + count + "d", i);
    }
}
