package com.gtis.archive.service.impl;

import com.google.common.base.*;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.gtis.archive.core.environment.EnvHolder;
import com.gtis.archive.core.environment.Environment;
import com.gtis.archive.entity.ConfigProp;
import com.gtis.archive.service.ConfigPropService;
import com.gtis.support.hibernate.HibernateTemplate;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.transaction.annotation.Transactional;

import java.util.Arrays;
import java.util.List;
import java.util.Map;


/**
 * 配置项service
 *
 * @author linlong
 * @since 2019.05.06
 */
public class ConfigPropServiceImpl extends HibernateTemplate<ConfigProp, String> implements ConfigPropService {

    /**
     * 忽略的配置的key
     */
    private Map<String, String> ignoreConfigProps;

    /**
     * 内部维护的缓存避免每次都从数据库查找
     */
    private Map<String, ConfigProp> cache;

    /**
     * {@inheritDoc}
     */
    @Override
    public int count() {
        String countSql = "SELECT count(*) FROM ConfigProp";
        return ((Long) getSession().createQuery(countSql).uniqueResult()).intValue();
    }

    /**
     * 批量添加
     *
     * @param list 待添加列表
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public void batchSave(List<ConfigProp> list) {
        for (ConfigProp p : list) {
            getSession().save(p);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void updatePair(ConfigProp p) {
        update(p);
        cache.put(p.getKey(), p);
        Environment env = EnvHolder.getAppEnv();
        while (env != null) {
            if (env.getProps().containsKey(p.getKey())) {
                env.getProps().put(p.getKey(), p.getValue());
                break;
            }else {
                env = env.getParent();
            }
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void update(ConfigProp p) {
        getSession().saveOrUpdate(p);
    }

    private void initIgnoreStr() {
        // 忽视掉的配置，大部分仍需要重启才能生效
        String[] ignoreStr = {
                "transmit.db.url",
                "transmit.db.driver",
                "transmit.db.password",
                "transmit.db.username",
                "sqlserver.db.password",
                "sqlserver.db.username",
                "sqlserver.db.url",
                "sqlserver.db.driver",
                "hibernate.format_sql",
                "archive.db.driver",
                "archive.db.password",
                "archive.db.username",
                "archive.db.url",
                "hibernate.hbm2ddl.auto",
                "hibernate.dialect",
                "hibernate.show_sql",
                "hibernate.generate_statistics",
                "web.root",
                "models.file"};
        ignoreConfigProps = Maps.uniqueIndex(Arrays.asList(ignoreStr), new Function<String, String>() {
            @Override
            public String apply(String s) {
                return s;
            }
        });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<ConfigProp> getConfigProps() {
        List<ConfigProp> list = Lists.newArrayList(cache.values());
        if (CollectionUtils.isEmpty(list)) {
            list = getAll();
            cache = toMap(getAll());
        }
        Iterable<ConfigProp> itr = Iterables.filter(list, new Predicate<ConfigProp>() {
            @Override
            public boolean apply(ConfigProp p) {
                return !ignoreConfigProps.containsKey(p.getKey());
            }
        });
        list = Lists.newArrayList(itr);
        return list;
    }

    /**
     * 更新配置
     *
     * @param enables 开关型配置，名称：通过逗号连接要获取开关列表用逗号分隔
     * @param inputs  输入型配置
     * @return 更新结果
     */
    @Override
    public void updateProps(String enables, Map<String, String> inputs) throws Exception {
        List<ConfigProp> listToUpdate = Lists.newArrayList();

        if (!Strings.isNullOrEmpty(enables)) {
            String[] enableArr = enables.split(",");
            for (String enable : enableArr) {
                ConfigProp prop = cache.get(enable);
                if (!Strings.isNullOrEmpty(prop.getValue())) {
                    if (prop.getValue().equals(ConfigProp.ON_VALUE)) {
                        prop.setValue(ConfigProp.OFF_VALUE);
                    } else {
                        prop.setValue(ConfigProp.ON_VALUE);
                    }
                    listToUpdate.add(prop);
                }
            }
        }

        if (inputs != null) {
            for (Map.Entry<String, String> entry : inputs.entrySet()) {
                if (!Strings.isNullOrEmpty(entry.getValue())) {
                    ConfigProp prop = cache.get(caseCamelToProp(entry.getKey()));
                    if (prop != null) {
                        prop.setValue(entry.getValue());
                        listToUpdate.add(prop);
                    }
                }
            }
        }

        for (ConfigProp p : listToUpdate) {
            updatePair(p);
        }
    }

    private String caseCamelToProp(String propName) {
        StringBuilder sb = new StringBuilder(propName);
        // 定位
        int temp = 0;
        if (!propName.contains(".")) {
            for (int i = 0; i < propName.length(); i++) {
                if (Character.isUpperCase(propName.charAt(i))) {
                    sb.insert(i + temp, ".");
                    temp++;
                }
            }
        }
        return Ascii.toLowerCase(sb.toString());
    }


    /**
     * 将pair列表转换为map
     *
     * @param list 待转换list
     * @return 转换后的map
     */
    private Map<String, ConfigProp> toMap(List<ConfigProp> list) {
        Map<String, ConfigProp> map = Maps.newLinkedHashMapWithExpectedSize(list.size());
        for (ConfigProp prop : list) {
            map.put(prop.getKey(), prop);
        }
        return map;
    }


    /*
     * note: 使用了@Transactional的方法，对同一个类里面的方法调用， @Transactional无效。
     * 比如有一个类Test，它的一个方法A，A再调用Test本类的方法B（不管B是否public还是private），
     * 但A没有声明注解事务，而B有。则外部调用A之后，B的事务是不会起作用的。（经常在这里出错）
     *
     * 调用同个类的方法，直接使用原来类的方法不会使用生成的代理类的方法，所以事务往往无效
     *
     * reference: https://blog.csdn.net/wsk1103/article/details/84666050
     *
     * 解决方法：
     * 1. 将b()方法抽出来，重新声明一个类，并且该类交由spring管理控制。
     * 2. 同时在a()上添加@Transactional注解或者在类上添加。
     * 3. 在原A类中的a()方法，改为 ((A)AopContext.currentProxy).b() (要将expose-proxy设为true，对应的spring boot注解为@EnableAspectJAutoProxy(exposeProxy=true))
     *
     * reference: https://blog.csdn.net/JustForSS/article/details/83008824
     */

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void init() {
        initIgnoreStr();
        Environment env = EnvHolder.getAppEnv();
        Map<Object, Object> appProps = env.getParent().getProps();
        if (count() <= 0) {
            List<ConfigProp> listToAdd = Lists.newArrayListWithExpectedSize(appProps.size());
            for (Map.Entry<Object, Object> entry : appProps.entrySet()) {
                listToAdd.add(new ConfigProp(entry.getKey().toString(), entry.getValue().toString()));
            }
            batchSave(listToAdd);
            cache = toMap(listToAdd);
        } else {
            // 生成新的map
            Map<String, Map.Entry<Object, Object>> stringAppProps = Maps.uniqueIndex(appProps.entrySet(), new Function<Map.Entry<Object,Object>, String>() {

                @Override
                public String apply(Map.Entry<Object, Object> entry) {
                    return entry.getKey().toString();
                }
            });
            List<ConfigProp> list = getAll();
            // 启动时修改配置，读取仍然是配置文件中的配置，但应当以数据库保存的为准
            for (ConfigProp prop : list) {
                if (!ignoreConfigProps.containsKey(prop.getKey())) {
                    Map.Entry<Object, Object> entry = stringAppProps.get(prop.getKey());
                    if (!Strings.isNullOrEmpty(prop.getValue()) && !prop.getValue().equals(entry.getValue().toString())) {
                        updateProp(env, prop);
                    }
                }
            }
            cache = toMap(list);
        }
    }

    /**
     * 更新配置
     *
     * @param env 环境
     * @param prop 待修改配置
     */
    private void updateProp(Environment env, ConfigProp prop) {
        env.put(prop.getKey(), prop.getValue());
    }
}
