package com.gtis.archive.service.impl;

import com.gtis.archive.Constants;
import com.gtis.archive.Switch;
import com.gtis.archive.core.EntityService;
import com.gtis.archive.core.Model;
import com.gtis.archive.core.ModelService;
import com.gtis.archive.core.dict.DictService;
import com.gtis.archive.core.dict.Item;
import com.gtis.archive.core.environment.EnvHolder;
import com.gtis.archive.core.support.hibernate.UUIDHexGenerator;
import com.gtis.archive.core.support.hibernate.envers.FixedAuditReaderFactory;
import com.gtis.archive.entity.*;
import com.gtis.archive.service.*;
import com.gtis.archive.util.ReportCache;
import com.gtis.archive.util.Struts2Utils;
import com.gtis.common.Page;
import com.gtis.plat.service.SysUserService;
import com.gtis.plat.vo.PfOrganVo;
import com.gtis.plat.vo.PfRoleVo;
import com.gtis.plat.vo.UserInfo;
import com.gtis.support.hibernate.HibernateDao;
import com.gtis.web.SessionUtil;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.annotation.Transactional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.math.BigDecimal;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.Collections;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-10-26
 */
public class ArchiveServiceImpl extends HibernateDao implements ArchiveService {
    private static final Logger logger = LoggerFactory.getLogger(ArchiveServiceImpl.class);
    private static String datePattern = "yyyyMMdd";

    private static ArrayList<String> dwdms;

    private static Date today;

    private String dhText = "根据档号查询档案出现异常";
    private String mlhSql = " and t.mlh = \'";
    private String modelNameText = "model_name='";
    private String andText = " AND ";
    private String ndText = "nd = '";
    private String dhLike = "dh like '%";
    private String mlhText = "mlh = '";

    static {
        dwdms = new ArrayList<String>();
        dwdms.add("220122");
        dwdms.add("220181");
        dwdms.add("220182");
        dwdms.add("220183");
        today = new Date();
    }

    @Autowired
    private EntityService entityService;
    @Autowired
    private ModelService modelService;
    @Autowired
    private OriginalService originalService;
    @Autowired
    private SysUserService userService;
    @Autowired
    private DictService dictService;
    @Autowired
    private PermissionService permissionService;
    @Autowired
    private ResourceService resourceService;
    @Autowired
    private ArchiveMetadataService archiveMetadataService;
    @Autowired
    public ReportCache reportCache;

    private boolean dwdmEnable = true;
    private boolean jiangyinEnable = false;

    public void setDwdmEnable(boolean dwdmEnable) {
        this.dwdmEnable = dwdmEnable;
    }

    public void setJiangyinEnable(boolean jiangyinEnable) {
        this.jiangyinEnable = jiangyinEnable;
    }


    @Override
    @Transactional
    public Archive getArchive(String modelName, String id) {
        return entityService.load(Document.toArchiveModelName(modelName), id);
    }

    @Override
    public Document getDocument(String modelName, String id) {
        return entityService.load(Archive.toDocumentModelName(modelName), id);
    }

    @Override
    public Chapter getChapter(String modelName, String id) {
        modelName = modelName.replace("_jn","_cp");
        modelName = modelName.substring(0,7);
        return entityService.load(modelName,id);
    }

    @Override
    public Archive getSimpleArchive(String id) {
        return (Archive) getSession().createSQLQuery("select t.*,0 clazz_ from t_archive t where t.id=?").addEntity(Archive.class).setParameter(0, id).uniqueResult();
    }

    @Override
    public Archive getSimpleArchiveByProId(String proid) {
        return (Archive) getSession().createSQLQuery("select t.*,0 clazz_ from t_archive t where t.pro_id=?").addEntity(Archive.class).setParameter(0, proid).list().get(0);
    }

    @Override
    public Document getSimpleDocument(String id) {
        return (Document) getSession().createSQLQuery("select t.*,0 clazz_ from t_document t where t.id=?").addEntity(Document.class).setParameter(0, id).uniqueResult();
    }

    @Override
    public Document getSimpleDocumentBySxh(String tm,int sxh) {
        String hql = "from Document where tm=:tm and sxh=:sxh ";
        Map map = new HashMap();
        List<Document> list = new ArrayList<Document>();
        map.put("tm", tm);
        map.put("sxh", sxh);
        try {
            list = find(hql, map);
        } catch (Exception e) {
            logger.error(dhText + e.toString(),e);
        }
        return list.get(0);
    }

    @Override
    public Chapter findChapterById(String id) {
        return entityService.load(Chapter.class, id);
    }

    @Override
    public Archive getDocumentArchive(String modelName, String documentId) {
        return getArchive(modelName, getDocument(modelName, documentId).getId());
    }

    @Override
    public Archive getDocumentArchive(Document document) {
        return getArchive(document.getModelName(), document.getArchiveId());
    }

    @Override
    public List<Document> getArchiveDocuments(String modelName, String archiveId) {
        List<Order> orders = new ArrayList<Order>(2);
        orders.add(Order.asc("sxh"));
        orders.add(Order.asc("id"));
        return entityService.<Document>search(Archive.toDocumentModelName(modelName), Collections.singletonList(Restrictions.eq("archiveId", archiveId)), orders, 0, Page.ALL).getItems();
    }

    public List<Chapter> getDocumentChapters(String modelName, String archiveId) {
        modelName = modelName.replace("_jn","_cp");
        List<Order> orders = new ArrayList<Order>(2);
        orders.add(Order.asc("sxh"));
        orders.add(Order.asc("id"));
        return entityService.<Chapter>search(modelName, Collections.singletonList(Restrictions.eq("documentId", archiveId)), orders, 0, Page.ALL).getItems();
    }

    @Override
    public List<Document> getArchiveDocuments(Archive archive) {
        return getArchiveDocuments(archive.getModelName(), archive.getId());
    }

    @Override
    public List<Chapter> getDocumentChapters(Document document) {
        return getDocumentChapters(document.getModelName(), document.getId());
    }

    @Override
    public Model getArchiveModel(String modelName) {
        return modelService.getModel(Document.toArchiveModelName(modelName));
    }

    @Override
    public Model getDocumentModel(String modelName) {
        return modelService.getModel(Archive.toDocumentModelName(modelName));
    }

    @Override
    public Archive newArchive(String modelName) {
        return (Archive) entityService.newInstance(Document.toArchiveModelName(modelName));
    }

    @Override
    public Document newDocument(String modelName) {
        return (Document) entityService.newInstance(Archive.toDocumentModelName(modelName));
    }

    @Override
    public Chapter newChapter(String modelName) {
        return (Chapter) entityService.newInstance(modelName);
    }

    @Transactional
    @Override
    public void removeArchive(String modelName, String[] ids) {
        Archive archive = null;
        for (String id : ids) {
            archive = getSimpleArchive(id);
            List<Document> docs = getArchiveDocuments(archive);
            String[] dids = new String[docs.size()];
            for (int i = 0, size = docs.size(); i < size; i++) {
                dids[i] = docs.get(i).getId();
            }
            removeDocument(modelName, dids);
            List<Original> ogs = originalService.getOriginals(id);
            String[] oids = new String[ogs.size()];
            for (int i = 0, size = ogs.size(); i < size; i++) {
                oids[i] = ogs.get(i).getId();
            }
            originalService.removeOriginal(oids);

            getSession().delete(entityService.load(archive.getModelName(),id));
        }
        logger.debug("remove archive [{}] ids [{}]", modelName, ids);
    }

    @Transactional
    @Override
    public void removeDocument(String modelName, String[] ids) {
        for (String id : ids) {
            List<Original> ogs = originalService.getOriginals(id);
            String[] oids = new String[ogs.size()];
            for (int i = 0, size = ogs.size(); i < size; i++) {
                oids[i] = ogs.get(i).getId();
            }
            originalService.removeOriginal(oids);
        }
        entityService.remove(Archive.toDocumentModelName(modelName), ids);
        logger.debug("remove document [{}] ids [{}]", modelName, ids);
    }

    @Transactional
    @Override
    public void removeChapter(String modelName, String[] ids) {
        for (String id : ids) {
            List<Original> ogs = originalService.getOriginals(id);
            String[] oids = new String[ogs.size()];
            for (int i = 0, size = ogs.size(); i < size; i++) {
                oids[i] = ogs.get(i).getId();
            }
            originalService.removeOriginal(oids);
        }
        entityService.remove(modelName, ids);
        logger.debug("remove chapter [{}] ids [{}]", modelName, ids);
    }

    @Transactional
    @Override
    public Archive saveArchive(Archive archive) {
        if (dwdmEnable) {
            if (StringUtils.isBlank(archive.getDwdm())) {
//                archive.setDwdm("321283"); //泰兴内容
                archive.setDwdm(getDwdm());
            }
            if (jiangyinEnable) {
                setQzhByDwdm(archive);
            }
        }
        if (StringUtils.isEmpty(archive.getId())) {
            archive.setId(UUIDHexGenerator.generate());
        }
        return entityService.save(archive);
    }

    @Transactional
    @Override
    public Document saveDocument(Document document) {
        if (StringUtils.isBlank(document.getDwdm())) {
            document.setDwdm(getDwdm());
        }
        return entityService.save(document);
    }

    @Transactional
    @Override
    public Chapter saveChapter(Chapter chapter) {
        if (dwdmEnable) {
            chapter.setDwdm(getDwdm());
        }
        return entityService.save(chapter);
    }

    @Override
    public Page<Archive> searchArchive(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit) {
        List<Criterion> list = new ArrayList<Criterion>(criterions);
        Page<Archive> page = new Page<Archive>();
        if (dwdmEnable && !SessionUtil.getCurrentUser().isAdmin() && !hasPermissionToViewAllArchive()) {
            Set<String> dwdms = getAvailableDwdms(modelName); // 获取单位代码列表
            Disjunction dis = Restrictions.disjunction();
            dis.add(Restrictions.isNull("dwdm"));
            dis.add(Restrictions.in("dwdm", dwdms));
            dis.add(Restrictions.like("dwdm", "%" + getDwdm() + "%"));
            list.add(dis);
        }

        if(EnvHolder.isEnable("nd.enable") && !SessionUtil.getCurrentUser().isAdmin() && !hasPermissionToViewAllArchive()) {
            Set<String> nds = getAvailableNds(modelName); // 获取年度列表
            Disjunction disjunction = Restrictions.disjunction();
            disjunction.add(Restrictions.isNull("nd"));
            disjunction.add(Restrictions.in("nd", nds));
            list.add(disjunction);
        }
        try {
            page = entityService.search(modelName, list, orders == null ? Collections.singletonList(Order.desc("id")) : orders, start, limit);
        }catch (ClassCastException e) {
            logger.error(e.getMessage());
        }
        return page;
    }

    @Override
    public Page<Document> searchDocument(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit) {
        List<Criterion> list = new ArrayList<Criterion>(criterions);
        if (dwdmEnable && !SessionUtil.getCurrentUser().isAdmin()) {
            Set<String> dwdms = getAvailableDwdms(modelName);
            list.add(Restrictions.or(Restrictions.isNull("dwdm"), Restrictions.in("dwdm", dwdms)));
        }
        return entityService.search(Archive.toDocumentModelName(modelName), list, orders == null ? Collections.singletonList(Order.desc("id")) : orders, start, limit);
    }

    @Override
    public int getArchiveCount(String modelName, List<? extends Criterion> criterions) {
        return entityService.getCount(modelName, criterions);
    }

    @Override
    public Page<Archive> searchArchive(String modelName, List<? extends Criterion> criterions, List<Order> orders, String dwdm, int start, int limit) {
        List<Criterion> list = new ArrayList<Criterion>(criterions);
        list.add(Restrictions.or(Restrictions.isNull("dwdm"), Restrictions.eq("dwdm", dwdm)));
        return entityService.search(modelName, list, orders == null ? Collections.singletonList(Order.desc("id")) : orders, start, limit);
    }

    @Override
    public List<Object> searchArchiveList(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit) {
        List<Criterion> list = new ArrayList<Criterion>(criterions);
        return entityService.search(modelName, list, orders == null ? Collections.singletonList(Order.desc("id")) : orders, start, limit).getItems();
    }

    @Override
    public String flhjs(String flhjs, String modelName) {
        String hql = "select flh from Archive where flh like :flhcx and modelName= :modelName group by flh order by flh ";
        String flhtj = "";
        List<Object> list = null;
        Map map = new HashMap();
        String flhcs = "%" + flhjs + "%";
        map.put("flhcx", flhcs);
        map.put("modelName", modelName);
        try {
            list = find(hql, map);
            StringBuilder flhsz = new StringBuilder();
            for (int i = 0; i < list.size(); i++) {
                try {
                    flhsz.append(list.get(i).toString());
                    flhsz.append(";");
                } catch (Exception e) {
                    logger.error(e.getMessage());
                }
            }
            flhtj = flhsz.toString();
        } catch (Exception e) {
            logger.error(e.getMessage());
        }
        return flhtj;
    }

    private String getDwdm() {
        String dwdm = (String) Struts2Utils.getSessionAttribute("__dwdm");
            ////单位代码用部门代码进行替换
            if (EnvHolder.isEnable(Switch.ORAGAN_NO)) {
                String oraganNo = null;
                List<PfOrganVo> pfOrganVoList = userService.getOrganListByUser(SessionUtil.getCurrentUserId());
                if (pfOrganVoList != null && !pfOrganVoList.isEmpty()) {
                    PfOrganVo pfOrganVo = pfOrganVoList.get(0);
                    oraganNo = pfOrganVo.getOraganNo();
                }
                dwdm = oraganNo;
            } else {
                dwdm = userService.getUserRegionCode(SessionUtil.getCurrentUserId());
            }
        return dwdm;
    }

    /**
     * 根据档号获取档案主表信息
     *
     * @param dh 档号
     * @return
     */

    @Override
    public Archive getArchiveByDh(String dh) {
        String hql = "FROM Archive where dh =:dh";
        Map map = new HashMap();
        List<Archive> list = new ArrayList<Archive>();
        map.put("dh", dh);
        try {
            list = find(hql, map);
        } catch (Exception e) {
            logger.error(dhText + e.toString());
        }
        return list.get(0);
    }

    /**
     * 对档号进行模糊查询
     *
     * @param dh
     * @return
     */
    @Override
    public List<Archive> getArchivesByDh(String dh) {
        Query query = getSession().createQuery("FROM Archive a WHERE a.dh LIKE ?");
        query.setString(0, dh);
        return query.list();
    }

    /**
     * 根据档号获取档案信息
     *
     * @param ajh 案卷号
     * @return
     */
    @Override
    public Archive getArchiveByAjh(Integer ajh, String mlh) {
        String hql = "FROM Archive where ajh=:ajh and mlh=:mlh ";
        Map map = new HashMap();
        List<Archive> list = new ArrayList<Archive>();
        map.put("dh", ajh);
        map.put("mh", mlh);
        try {
            list = find(hql, map);
        } catch (Exception e) {
            logger.error(dhText , e.toString());
        }
        return list.get(0);
    }

    /**
     * 获取ajhs
     *
     * @param room
     * @param carbinet
     * @param side
     * @param cell
     * @param box
     * @return
     */
    @Override
    public String getAjhs(String room, String carbinet, Integer side, Integer cell, Integer box) {
        String hql = "FROM Archive where roomId =:room and cabinetSno =:carbinet and sideId =:side and cellId =:cell and boxId =:box";
        List<Archive> list = new ArrayList<Archive>();
        StringBuilder stringBuffer = new StringBuilder();
        Map map = new HashMap();
        map.put("room", room);
        map.put("carbinet", carbinet);
        map.put("side", side);
        map.put("cell", cell);
        map.put("box", box);
        try {
            list = find(hql, map);
        } catch (Exception e) {
            logger.error("查询ajhs异常" + e.toString(),e);
        }
        for (Archive archive : list) {
            stringBuffer.append(archive.getAjh().toString());
            stringBuffer.append(",");
        }

        return stringBuffer.toString();
    }

    /**
     * 检查是否有权限查看所有的档案
     *
     * @return
     */
    private boolean hasPermissionToViewAllArchive() {
        boolean flag = false;
        flag = permissionService.hasPermission("sys", "viewAllArchive");
        return flag;
    }

    /**
     * 可选功能：根据单位代码设置全宗号，其对应关系由字典表中qzhdwdm决定
     *
     * @param archive
     */
    private void setQzhByDwdm(Archive archive) {
        if (StringUtils.isBlank(archive.getDwdm())) {
            return;
        }
        String dwdm = archive.getDwdm();
        try {
            List<Item> items = dictService.getItems(Constants.QZH_DWDM);
            for (Item item : items) {
                if (dwdm.equals(item.getName())) {
                    archive.setQzh(item.getValue());
                    break;
                }
            }
        } catch (Exception ex) {
            logger.info("未配置[{}]单位代码全宗号字典项", Constants.QZH_DWDM);
        }
    }

    /**
     * 获取数据库中顺序号最大的卷内档案。
     *
     * @param modelName
     * @param archiveId
     * @return
     */
    @Override
    public Document getMaxSxhDocuments(String modelName, String archiveId) {
        List<Order> orders = new ArrayList<Order>(2);
        orders.add(Order.desc("sxh"));

        List<Document> documents = entityService.<Document>search(Archive.toDocumentModelName(modelName), Collections.singletonList(Restrictions.eq("archiveId", archiveId)), orders, 0, Page.ALL)
                .getItems();
        return documents.isEmpty() ? null : documents.get(0);
    }

    /**
     * 获取数据库中顺序号最大的卷内档案。
     *
     * @param modelName
     * @param archiveId
     * @return
     */
    @Override
    public Chapter getMaxSxhChp(String modelName, String archiveId) {
        List<Order> orders = new ArrayList<Order>(2);
        orders.add(Order.desc("sxh"));

        List<Chapter> documents = entityService.<Chapter>search(modelName, Collections.singletonList(Restrictions.eq("documentId", archiveId)), orders, 0, Page.ALL).getItems();
        return documents.isEmpty() ? null : documents.get(0);
    }

    /**
     * 获取数据库中的最大盒号
     * @return
     */
    @Override
    public Integer getMaxBoxNum(String modelName,String nd,String mlh) {
        String hql = "SELECT max(t.BOX_ID) FROM t_archive t where t.model_Name= \'" + modelName + "\'";

        if(mlh != null){
            hql = hql + mlhSql + mlh + "\'";
        }

        if(nd != null){
            hql = hql + " and nd = \'" + nd + "\'";
        }

        BigDecimal count = (BigDecimal) getSession().createSQLQuery(hql).uniqueResult();
        return count == null? 0:count.intValue();
    }

    public void getMaxBoxId(String hql, Object... objects) {
        Query query = getSession().createSQLQuery(hql);
        for (int i = 0; i < objects.length; i++) {
            query.setParameter(i, objects[i]);
        }
        query.executeUpdate();
    }

    /**
     * 自动获取生成盒号
     * 盒号格式按照芜湖县的要求
     * 格式：H-年-流水号
     *
     * @return 生成的流水号
     */
    @Override
    public String autoGenerateBoxId() {
        Integer box = getMaxBoxNum("Archive",null,null);
        Calendar c = Calendar.getInstance();
        String boxId = null;
        int curYear = c.get(Calendar.YEAR);
        if (box != null) {
            if (box == 0) {
                boxId = curYear + "001";
            } else {
                String s = box.toString();
                int year = Integer.parseInt(s.substring(0, 4));
                int num = Integer.parseInt(s.substring(4, s.length()));
                if (year != curYear) {
                    year = curYear;
                }
                num++;
                boxId = year + String.format("%03d", num);
            }
        } else {
            logger.error("生成盒号错误");
        }
        return boxId;
    }

    /**
     * 根据房间号、柜号、格子号获取档案
     *
     * @param currentRoom
     * @param currentCab
     * @param rowId
     * @param colId
     * @return
     */
    @Override
    public List<Archive> getArchivesByCell(String currentRoom, String currentCab, Integer rowId, Integer colId) {
        Query query = getSession().createQuery("FROM Archive a WHERE a.roomId=? AND a.cabinetSno=? AND a.rowId=? AND a.columnId=?");
        query.setString(0, currentRoom);
        query.setString(1, currentCab);
        query.setInteger(2, rowId);
        query.setInteger(3, colId);
        return query.list();
    }

    /**
     * 根据盒号获取案卷
     *
     * @param boxId
     * @return
     */
    @Override
    public List<Archive> getArchivesByBox(Integer boxId) {
        Query query = getSession().createQuery("FROM Archive a WHERE a.boxId=?");
        query.setInteger(0, boxId);
        return query.list();
    }

    /**
     * 根据存放位置获取案卷
     *
     * @param cfwz
     * @return
     */
    @Override
    public List<Archive> getArchivesByCfwz(String cfwz) {
        Query query = getSession().createQuery("FROM Archive a WHERE a.cfwz=?");
        query.setString(0, cfwz);
        return query.list();
    }

    /**
     * 获取档案最大柜号
     */
    @Override
    public Archive getArchiveByPosition(String modelName) {
        List<Archive> list = getSession().createSQLQuery("select t.*,0 clazz_ from t_archive t where t.cabinet_sno is not null and t.side_id is not null and t.cell_id is not null and t.model_name = " + modelName + " and t.box_id is not null order by to_number(t.cabinet_sno) desc,t.side_id desc,t.cell_id desc,t.box_id desc").addEntity(Archive.class).list();

        if (list == null || list.isEmpty()) {
            return null;
        }

        return (Archive)list.get(0);
    }

    /**
     * 获取档案最大柜号
     */
    @Override
    public Archive getArchiveByPosition() {
        List<Archive> list = getSession().createSQLQuery("select t.*,0 clazz_ from t_archive t where t.cabinet_sno is not null and t.side_id is not null and t.cell_id is not null and t.box_id is not null order by to_number(t.cabinet_sno) desc,t.side_id desc,t.cell_id desc,t.box_id desc").addEntity(Archive.class).list();

        if (list == null || list.isEmpty()) {
            return null;
        }

        return (Archive)list.get(0);
    }

    /**
     * 获取档案最大柜号
     */
    @Override
    public Integer getArchivePosition() {
        List<String> list = getSession().createSQLQuery("select t.cabinet_sno from t_archive t where t.cabinet_sno is not null  group by t.cabinet_sno order by t.cabinet_sno desc").list();

        if (list == null || list.isEmpty()) {
            return 1;
        }

        return Integer.valueOf(list.get(0));
    }

    /**
     * 根据柜号获取柜号中最大格号
     *
     * @param setCabinetSno
     * @return
     */
    @Override
    public Integer getMaxCellId(String setCabinetSno) {
        Object object = getSession().createSQLQuery("select  max (a.cell_id) from t_archive a where a.cabinet_sno  = " + setCabinetSno).uniqueResult();
        return object == null ? 1 : ((BigDecimal) object).intValue();
    }

    /**
     * 根据格号、柜号获取最大的盒号
     *
     * @param cellId
     * @param cabinetSno
     * @return
     */
    @Override
    public Integer getMaxBox(Integer cellId, String cabinetSno) {
        Object object = getSession().createSQLQuery("select  max (a.box_id) from t_archive a where a.cell_id = " + cellId + " and a.cabinet_sno = " + cabinetSno).uniqueResult();
        return object == null ? 1 : ((BigDecimal) object).intValue();
    }

    /**
     * 判断盒子是否已满
     *
     * @param boxId
     * @param cellId
     * @param cabinetSno
     * @return
     */
    @Override
    public boolean isBoxFull(Integer boxId, Integer cellId, String cabinetSno) {
        Integer count = ((BigDecimal) (getSession().createSQLQuery("select count(t.id) from t_archive t where t.cell_id = " + cellId + " and t.box_id = " + boxId + " and t.cabinet_sno = " + cabinetSno).uniqueResult())).intValue();
        return count >= 15;
    }

    /**
     * 判断格子是否已满
     *
     * @param cellId
     * @param cabinetSno
     * @return
     */
    @Override
    public boolean isCellFull(Integer cellId, Integer cabinetSno) {
        Integer count = ((BigDecimal) (getSession().createSQLQuery("select count(t.id) from t_archive t where t.cell_id = " + cellId + " and t.cabinet_sno = " + cabinetSno).uniqueResult())).intValue();
        return count >= 180;
    }

    @Override
    public int maxYjdh(String tableName, String nd, String ywjc, String dwjc) {
        String hql = "select max(yjdhstr) from (select substr(yjdh,13,4) yjdhstr from " + tableName + " where substr(yjdh,1,4) = \'" + nd + "\' and substr(yjdh,11,2) = \'" + ywjc + "\'and substr(yjdh,9,2) = \'" + dwjc + "')";
        String yjdh = (String) getSession().createSQLQuery(hql).uniqueResult();
        String num = null;
        char[] ch = new char[1];
        int j = 0;
        if (!StringUtils.isEmpty(yjdh)) {
            for (int i = 0; i < yjdh.length(); i++) {
                char c = yjdh.charAt(i);
                if (c != 0){
                    if (j > ch.length) {
                        Arrays.copyOf(ch, j);
                    }
                    ch[j++] = c;
                }
            }
        } else {
            return 0;
        }
        num = String.valueOf(ch);
        return Integer.valueOf(num);
    }

    @Transactional
    @Override
    public void updateYjdhById(String tableName, String id, String value, String type) {
        String hql = "update " + tableName + " set yjdh = \'" + value + "\' where id = \'" + id + "\'";
        if("js".equals(type)){
            hql = "update " + tableName + " set jsdh = \'" + value + "\' where id = \'" + id + "\'";
        }
        getSession().createSQLQuery(hql).executeUpdate();
    }

    @Override
    public Archive createDhByArchive(Archive archive, String userId) {
        String dwjc = null;
        String nd = archive.getNd();
        String modelName = archive.getModelName();
        if (nd == null) {
            DateFormat df = new SimpleDateFormat("yyyy");
            nd = df.format(new Date());
        }
        try {
            String oraganNo = null;
            if (userId == null || userId == "") {
                logger.error("用户Id为空");
            }
            List<PfOrganVo> pfOrganVoList = userService.getOrganListByUser(userId);
            if (pfOrganVoList != null && !pfOrganVoList.isEmpty()) {
                PfOrganVo pfOrganVo = pfOrganVoList.get(0);
                oraganNo = pfOrganVo.getOraganNo();
            }
            String dwdm = oraganNo;
            if (dwdm != null) {
                List<Item> items = this.dictService.getItems("dwjc");
                for (Item item : items) {
                    String value = item.getValue();
                    if (value.equals(dwdm)) {
                        dwjc = item.getName();
                        break;
                    }
                }
                StringBuilder buffer = new StringBuilder();
                buffer.append(nd).append(dwjc);
                buffer.append(modelName.toUpperCase());
                String newDh = buffer.toString();
                Integer ajh = archive.getAjh();
                int length = ajh.toString().length();
                for (int i = newDh.length(); i < 13 - length; i++) {
                    buffer.append("0");
                }
                buffer.append(ajh);
                newDh = buffer.toString();
                archive.setDh(newDh);
            } else {
                logger.error("找不到对应的部门简称");
            }
        } catch (Exception e) {
            logger.error("dwjc的字典项不存在", e);
        }
        return archive;
    }

    /**
     * 根据案卷所属卷内的页数获得案卷的页数
     *
     * @param sum
     * @param archiveId
     */
    @Transactional
    @Override
    public void updateArchiveYsByDoc(int sum, String archiveId) {
        String hql = "update t_archive set ys= " + sum + " where id= \'" + archiveId + "\'";
        getSession().createSQLQuery(hql).executeUpdate();
    }

    /**
     * 判断boxId在是否已占用
     *
     * @param boxId
     * @return
     */
    @Override
    public boolean isExistBoxId(Integer boxId, String mlh, String cabinetSno, Integer columnId, Integer cellId, String modelName, String nd) {
        String hql = "select t.id from t_archive t where t.box_id = " + boxId + " and t.model_name = '" + modelName + "\'";

        if(mlh != null){
            hql = hql + mlhSql + mlh + "\'";
        }

        if(cabinetSno != null){
            hql = hql + " and t.CABINET_SNO = \'" + cabinetSno + "\'";
        }

        if(columnId != null){
            hql = hql + " and t.column_Id = \'" + columnId + "\'";
        }

        if(cellId != null){
            hql = hql + " and t.CELL_ID = \'" + cellId + "\'";
        }

        if(nd != null){
            hql = hql + " and t.nd = \'" + nd + "\'";
        }

        hql = hql + " and rownum <= 1";

        String count = (String) getSession().createSQLQuery(hql).uniqueResult();

        return count != null || count != "";
    }

    /**
     * 判断boxId在是否已占用
     *
     * @param boxId
     * @return
     */
    @Override
    public boolean isExistBoxId(Integer boxId, String mlh, String cabinetSno, Integer columnId, Integer cellId, String modelName) {
        String hql = "select t.id from t_archive t where t.box_id = " + boxId + " and t.model_name = '" + modelName + "\'";

        if(mlh != null){
            hql = hql + mlhSql + mlh + "\'";
        }

        if(cabinetSno != null){
            hql = hql + " and t.CABINET_SNO = \'" + cabinetSno + "\'";
        }

        if(columnId != null){
            hql = hql + " and t.column_Id = \'" + columnId + "\'";
        }

        if(cellId != null){
            hql = hql + " and t.CELL_ID = \'" + cellId + "\'";
        }

        hql = hql + " and rownum <= 1";

        String count = (String) getSession().createSQLQuery(hql).uniqueResult();
        return (StringUtils.isBlank(count)||count == null);
    }

    /**
     * 根据Hql语句对archive进行更新
     *
     * @param hql     语句
     * @param objects 参数
     */
    @Transactional
    @Override
    public void upDateArchiveByHql(String hql, Object... objects) {
        Query query = getSession().createSQLQuery(hql);
        for (int i = 0; i < objects.length; i++) {
            query.setParameter(i, objects[i]);
        }
        query.executeUpdate();
    }

    /**
     * 获取档案的年度
     *
     * @param modelName 模型名
     * @return
     */
    @Override
    public List getArchiveNd(String modelName) {
        String sSQL = "";
        if (StringUtils.isBlank(modelName)) {
            sSQL = "select t.nd from t_archive t group by t.nd";
        } else {
            sSQL = "select t.nd from t_archive t where t.model_name = '" + modelName + "' group by t.nd order by t.nd";
        }

        return getSession().createSQLQuery(sSQL).list();
    }

    /**
     * 获取档案的目录号
     *
     * @param modelName 业务模型名
     * @return
     */
    @Override
    public List getArchiveMlh(String modelName) {
        String sSQL = "";
        if (StringUtils.isBlank(modelName)) {
            sSQL = "select t.mlh from t_archive t group by t.mlh";
        } else {
            sSQL = "select t.mlh from t_archive t where t.model_name = '" + modelName + "' group by t.mlh order by t.mlh";
        }

        return getSession().createSQLQuery(sSQL).list();
    }

    /**
     * 获取档案的盒号
     *
     * @param modelName 模型名
     * @param nd        年度
     * @param dh        档号
     * @param mlh       目录号
     * @param boxId     盒号
     * @param cabinetSno     密集架号
     * @param cellId     列号
     * @param cellId     层号
     * @param cfwz      存放位置
     * @param fiedsSQL     查询字段
     * @param orderSQL     排序字段
     * @return
     */

    @Override
    public List getBox(String modelName, String nd, String dh, String mlh, String boxId, String cabinetSno, String columnId, String cellId, String cfwz, String fiedsSQL, String orderSQL) {
        String sSQL = "SELECT count(*),max(ajh),min(ajh),box_id, cfwz, nd, mlh, sjsj FROM t_archive";
        if(!"count(*),max(ajh),min(ajh),box_id, cfwz, sjsj".equals(fiedsSQL)){
            sSQL = "SELECT " + fiedsSQL + " FROM t_archive";
        }
        String sWhere = "";
        if (StringUtils.isNotBlank(modelName)) {
            sWhere = sWhere + modelNameText + modelName + "'";
        }

        if (StringUtils.isNotBlank(nd)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + ndText + nd + "'";
        }

        if (StringUtils.isNotBlank(dh)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + dhLike + dh + "%'";
        }

        if (StringUtils.isNotBlank(mlh)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + mlhText + mlh + "'";
        }

        if (StringUtils.isNotBlank(boxId)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + "box_id = " + boxId;
        }

        if (StringUtils.isNotBlank(cabinetSno)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + "cabinet_sno = " + cabinetSno;
        }

        if (StringUtils.isNotBlank(columnId)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + "side_id = " + columnId;
        }

        if (StringUtils.isNotBlank(cellId)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + "cell_id = " + cellId;
        }

        if (StringUtils.isNotBlank(cfwz) && "not".equals(cfwz)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + "cfwz IS NOT NULL ";
        } else if (StringUtils.isNotBlank(cfwz) && "null".equals(cfwz)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + "cfwz IS NULL ";
        }

        if ("".equals(sWhere) && StringUtils.isNotBlank(sWhere)) {
            sSQL = sSQL + " WHERE box_id is not NULL AND " + sWhere;
        } else {
            sSQL = sSQL + " WHERE box_id is not NULL ";
        }

        if("".equals(orderSQL)) {
            sSQL = sSQL + " GROUP BY box_id, cfwz, nd, mlh, sjsj ORDER BY nd DESC, mlh DESC, box_id DESC, sjsj DESC";
        } else if("count(*),max(ajh),min(ajh),box_id, cfwz, sjsj".equals(fiedsSQL)) {
            sSQL = sSQL + " GROUP BY box_id, cfwz, nd, mlh, sjsj ORDER BY " + orderSQL;
        }else {
            sSQL = sSQL + " GROUP BY " + fiedsSQL.substring(27, fiedsSQL.length()) + " ORDER BY " + orderSQL;
        }

        return getSession().createSQLQuery(sSQL).list();
    }

    /**
     * 删除档案盒号信息
     *
     * @param boxIds    盒号信息
     * @param modelName 业务模型名
     * @param nd        年度
     * @param dh        档号
     * @param mlh       目录号
     * @param cfwz      存放位置
     */
    @Transactional
    @Override
    public void deleteArchiveBoxId(String[] boxIds, String modelName, String nd, String dh, String mlh, String cfwz) {
        if (boxIds == null || boxIds.length == 0) {
            return;
        }

        StringBuilder sb = new StringBuilder();
        for (String boxId : boxIds) {
            sb.append("\'").append(boxId).append("\'").append(",");
        }

        String sbs = sb.toString();
        String idStr = sbs.substring(0, sbs.length() - 1);
        String sSQL = "UPDATE t_archive SET box_id = '' WHERE box_id in (" + idStr + ") ";

        String sWhere = "";
        if (modelName != null && StringUtils.isNotBlank(modelName)) {
            sWhere = sWhere + modelNameText + modelName + "'";
        }

        if (nd != null && StringUtils.isNotBlank(nd)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + ndText + nd + "'";
        }

        if (dh != null && StringUtils.isNotBlank(dh)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + dhLike + dh + "%'";
        }

        if (mlh != null && StringUtils.isNotBlank(mlh)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + mlhText + mlh + "'";
        }

        if (cfwz != null && StringUtils.isNotBlank(cfwz) && "not".equals(cfwz)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + "cfwz IS NOT NULL ";
        } else if (cfwz != null && StringUtils.isNotBlank(cfwz) && "null".equals(cfwz)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + "cfwz IS NULL ";
        }

        if (sWhere != "" && StringUtils.isNotBlank(sWhere)) {
            sSQL = sSQL + andText + sWhere;
        }

        Query query = getSession().createSQLQuery(sSQL);
        query.executeUpdate();
    }

    /**
     * 保存档案的存放位置
     *
     * @param modelName 模型名
     * @param nd        年度
     * @param dh        档号
     * @param mlh       目录号
     * @param boxId     盒号
     * @param cfwz      存放位置
     */
    @Transactional
    @Override
    public void updateArchiveBoxIDd(String modelName, String nd, String dh, String mlh, String boxId, String cfwz) {
        if (boxId == null || StringUtils.isBlank(boxId)) {
            return;
        }

        String sSQL = "UPDATE t_archive SET cfwz = '" + ((cfwz == null || StringUtils.isBlank(cfwz)) ? "" : cfwz) + "' WHERE box_id = " + boxId + "";
        String sWhere = "";
        if (modelName != null && StringUtils.isNotBlank(modelName)) {
            sWhere = sWhere + modelNameText + modelName + "'";
        }

        if (nd != null && StringUtils.isNotBlank(nd)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + ndText + nd + "'";
        }

        if (dh != null && StringUtils.isNotBlank(dh)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + dhLike + dh + "%'";
        }

        if (mlh != null && StringUtils.isNotBlank(mlh)) {
            sWhere = sWhere + (sWhere != "" && StringUtils.isNotBlank(sWhere) ? andText : "") + mlhText + mlh + "'";
        }

        if (sWhere != "" && StringUtils.isNotBlank(sWhere)) {
            sSQL = sSQL + andText + sWhere;
        }

        Query query = getSession().createSQLQuery(sSQL);
        query.executeUpdate();
    }

    /**
     * 获取最大sideId
     *
     * @param cabinetsno
     * @param boxId
     * @param cellId
     * @return
     */
    @Override
    public Integer getMaxSideId(String cabinetsno, Integer boxId, Integer cellId) {
        Object object = getSession().createSQLQuery("select max(t.side_id) from t_archive t where t.cabinet_sno =  \'" + cabinetsno + "\' and t.box_id = " + boxId + "and t.cell_id = " + cellId).uniqueResult();
        return object == null ? 0 : ((BigDecimal) object).intValue();
    }

    @Override
    public boolean isExistArchiveByCfwz(String cfwz) {
        String hql = "select count(1) from t_archive t where t.cfwz = \'" + cfwz + "\'";
        BigDecimal count = (BigDecimal) getSession().createSQLQuery(hql).uniqueResult();
        return count.intValue() != 0;
    }

    @Override
    public List<Object> getFieldValue(String sql, Object... objects) {
        Query query = getSession().createSQLQuery(sql);

        for (int i = 0 ; i< objects.length;i++){
            query.setParameter(i,objects[i]);
        }
        return query.list();
    }

    /**
     * 获取当前用户可访问的单位代码
     * @return
     */
    public Set<String> getAvailableDwdms(String modelName) {
        List<PfRoleVo> roles = SessionUtil.getCurrentUser().getLstRole();
        Set<String> dwdms = new HashSet<String>();
        Resource resource = resourceService.getResourceByName(modelName);
        if (!EnvHolder.isEnable(Switch.isNewPermission)) {
            for (PfRoleVo role : roles) {
                List<Permission> permissions = permissionService.getDwdmPermissions(resource.getId(), role.getRoleId(), "dwdm");
                Iterator<Permission> itr = permissions.iterator();
                while (itr.hasNext()) {
                    dwdms.add(itr.next().getDwdm());
                }
            }
        }else {
            UserInfo userInfo = SessionUtil.getCurrentUser();
            String userId = userInfo.getId();
            List<Permission> permissions = permissionService.getDwdmPermissionsNew(resource.getId(),userId,"dwdm");
            for (Permission permission : permissions) {
                dwdms.add(permission.getDwdm());
            }
        }
        dwdms.add(getDwdm());
        return dwdms;
    }

    /**
     * 获取当前用户可访问的年度
     * @return
     */
    public Set<String> getAvailableNds(String modelName) {
        List<PfRoleVo> roles = SessionUtil.getCurrentUser().getLstRole();
        Set<String> dwdms = new HashSet<String>();
        Resource resource = resourceService.getResourceByName(modelName);
        if (!EnvHolder.isEnable(Switch.isNewPermission)) {
            for (PfRoleVo role : roles) {
                List<Permission> permissions = permissionService.getDwdmPermissions(resource.getId(), role.getRoleId(), "nd");
                Iterator<Permission> itr = permissions.iterator();
                while (itr.hasNext()) {
                    dwdms.add(itr.next().getDwdm());
                }
            }
        }else {
            UserInfo userInfo = SessionUtil.getCurrentUser();
            String userId = userInfo.getId();
            List<Permission> permissions = permissionService.getDwdmPermissionsNew(resource.getId(),userId,"nd");
            for (Permission permission : permissions) {
                dwdms.add(permission.getDwdm());
            }
        }
        return dwdms;
    }

    /**
     * 获取最大的流水号(长春不动产档案)
     * @return
     */
    private String sqlDwdm = "220112";
    @Override
    @Transactional
    public String getMaxLsh(String modelName, Archive archive) throws ClassNotFoundException {
        String s;
        String sql = "";

        if (org.apache.commons.lang3.StringUtils.equals(archive.getDwdm(), sqlDwdm)) { // 双阳区
            if (org.apache.commons.lang3.StringUtils.equals(modelName, "Fdzs") || org.apache.commons.lang3.StringUtils.equals(modelName, "Fdzm")) {
                sql = "SELECT MAX(to_number(t.lsh)) FROM t_archive t WHERE t.dwdm = '220112' AND (t.model_name = 'Fdzs' OR t.model_name = 'Fdzm')";
            } else {
                sql = "SELECT MAX(to_number(t.lsh)) FROM t_archive t WHERE t.dwdm = '220112' AND (t.model_name <> 'Fdzs' AND t.model_name <> 'Fdzm')";
            }
        } else if (dwdms.contains(archive.getDwdm())) {
            if (!org.apache.commons.lang3.StringUtils.equals(modelName, "Fdzs") && !org.apache.commons.lang3.StringUtils.equals(modelName, "Fdzm")) {
                sql = "SELECT MAX(to_number(t.lsh)) FROM t_archive t WHERE t.dwdm IN ('220122', '220181', '220182', '220183') AND (t.model_name <> 'Fdzs' AND t.model_name <> 'Fdzm') AND t.dwdm= " + archive.getDwdm();
            }
        } else { // 长春市
            if (org.apache.commons.lang3.StringUtils.equals(modelName, "Fdzs") || org.apache.commons.lang3.StringUtils.equals(modelName, "Fdzm")) {
                sql = "SELECT MAX(to_number(t.lsh)) FROM t_archive t WHERE t.dwdm NOT IN ('220112', '220122', '220181', '220182', '220183') AND (t.model_name = 'Fdzs' OR t.model_name = 'Fdzm')";
            } else {
                sql = "SELECT MAX(to_number(t.lsh)) FROM t_archive t WHERE t.dwdm NOT IN ('220112', '220122', '220181', '220182', '220183') AND (t.model_name <> 'Fdzs' AND t.model_name <> 'Fdzm')";
            }
        }

        Object object = getSession().createSQLQuery(sql).uniqueResult();

        if (object == null) {
            s = "";
        } else {
            s = object.toString();
        }

        return s;
    }

    /**
     * 生成流水号，长春地区需求
     * 流水号形式：
     * 房地一体业务
     * 1.长春市：年月日 + 5 + 四位流水号 例：2016090250001
     * 2.双阳区：9 + 年月日 + 5 + 四位流水号 例：92016090250001
     * 非房地一体业务
     * 1.长春市：年 + 六位流水号 例：2016000001
     * 2.双阳区：同房产业务 例：92016000001
     * 3.其他区：单位代码 + 年 + 流水号 例：2201822016000001
     * @param modelName
     * @param archive
     * @return
     * @throws ClassNotFoundException
     */
    @Override
    @Transactional
    public synchronized String generateAndUpdateLsh(String modelName, Archive archive){
        String s = null;
        try {
            if (modelName.equals("Fdzs") || modelName.equals("Fdzm")) {
                s = generateFcywLsh(modelName, archive);
            } else {
                s = generateTdywLsh(modelName, archive);
            }
        }catch (ClassNotFoundException e) {
            logger.error("赋值错误:{}", e.getMessage());
            throw new ClassCastException("赋值错误");
        } catch (ParseException e) {
            logger.error("日期转换错误{}", e.getMessage());
        }
        archive.setLsh(s);
        saveArchive(archive);
        return s;
    }

    /**
     * 生成房产业务的流水号
     * 长春和双阳区
     * @param modelName
     * @param archive
     * @return
     * @throws ClassNotFoundException
     */
    private String generateFcywLsh(String modelName, Archive archive) throws ClassNotFoundException, ParseException {
        Session session = getSession();
        // 前缀,双阳区流水号加9
        String suffix ;
        // 根据地区确定前缀
        String seqPrefix;
        String s;
        Integer i;

        if (archive.getDwdm().equals(sqlDwdm)) {
            seqPrefix = "SY";
            suffix = "9";
        } else {
            seqPrefix = "CC";
            suffix = "";
        }

        Date current = new Date();

        if (!DateUtils.isSameDay(today, current)) {
            i = getFcywLsh(archive);
            String maxLsh = getMaxLsh(modelName, archive);
            if (org.apache.commons.lang3.StringUtils.isNotBlank(maxLsh) && !DateUtils.isSameDay(current, getFcywDate(maxLsh, archive.getDwdm()))) {
                logger.info("初始化序列{}", seqPrefix + "_FCYW_LSH");
                // 初始化序列的值
                String sql = "ALTER SEQUENCE " + seqPrefix + "_FCYW_LSH INCREMENT BY " + (50001 - i);
                session.createSQLQuery(sql).executeUpdate();
                i = getFcywLsh(archive);
                // 重新设置序列的增长为1
                sql = "ALTER SEQUENCE " + seqPrefix + "_FCYW_LSH INCREMENT BY 1";
                session.createSQLQuery(sql).executeUpdate();
            }
            today = current;
            s = suffix + DateFormatUtils.format(today, datePattern) + i;
        } else {  // 最大流水号有且是今天的话直接加一
            i = getFcywLsh(archive);
            s = suffix + DateFormatUtils.format(today, datePattern) + i;
        }
        return s;
    }

    /**
     * 生成土地业务的流水号
     * @param modelName
     * @param archive
     * @return
     * @throws ClassNotFoundException
     */
    private String generateTdywLsh(String modelName, Archive archive) throws ClassNotFoundException {
        String s = getMaxLsh(modelName, archive);
        Calendar c = Calendar.getInstance();
        int curYear = c.get(Calendar.YEAR);
        String base = null;

        if (org.apache.commons.lang3.StringUtils.isNotBlank(s)) {
            String year = getTdywLshYear(s, archive.getDwdm());
            if (Integer.parseInt(year) == curYear) {
                Long l = Long.parseLong(s) + 1;
                base = l.toString();
            }
        }

        if (org.apache.commons.lang3.StringUtils.isBlank(base)) {
            base = curYear + "000001";
            if (archive.getDwdm().equals(sqlDwdm)) {
                base = "9" + base;
            } else if (dwdms.contains(archive.getDwdm())) {
                base = archive.getDwdm() + base;
            }
        }

        return base;
    }

    @Override
    @Transactional
    public Integer getFcywLsh(Archive archive) {
        String prefix;
        if (archive.getDwdm().equals(sqlDwdm)) {
            prefix = "SY";
        } else {
            prefix = "CC";
        }
        Session session = getSession();
        String sql = "SELECT " + prefix + "_FCYW_LSH.nextval FROM dual";
        Query query = session.createSQLQuery(sql);
        BigDecimal b = (BigDecimal) query.uniqueResult();
        return b.intValue();
    }

    /**
     * 获取土地业务流水号中的年度
     * @param lsh 流水号
     * @param dwdm 单位代码
     * @return
     */
    private String getTdywLshYear(String lsh, String dwdm) {
        String s;

        if (dwdms.contains(dwdm)) {
            s = lsh.substring(6, 10);
        } else if (sqlDwdm.equals(dwdm)) {
            s = lsh.substring(1, 5);
        } else {
            s = lsh.substring(0, 4);
        }

        return s;
    }

    /**
     * 获取流水号中的日期
     * @param lsh
     * @param dwdm
     * @return
     */
    private Date getFcywDate(String lsh, String dwdm) throws ParseException {
        String s;
        if (sqlDwdm.equals(dwdm)) {
            s = lsh.substring(1, 9);
        } else {
            s = lsh.substring(0, 8);
        }
        return DateUtils.parseDate(s, datePattern);
    }

    /**
     * 根据hql语句进行查询
     * @param hql hql语句
     * @param objects 传递来的不定长度参数
     * @return
     */
    public List<Archive> getArchiveByHQL(String hql,Object...objects){
        Query query = getSession().createQuery(hql);
        for (int i = 0;i<objects.length;i++){
            query.setParameter(0,objects[i]);
        }
        return query.list();
    }

    /**
     * 马鞍山触摸屏查询生成随机码
     * @param id
     * @return
     */
    public String getCode(String id){
        Map<String,Object> map = new HashMap<String, Object>();
        map.put("time",600);//设置操作时间
        Archive archive = this.getSimpleArchive(id);
        map.put("archive",archive);
        StringBuilder sb = new StringBuilder();
        for (int i = 0;i<6;i++){
            sb.append((int) (Math.random()*10));
        }
        reportCache.setReportCache(sb.toString(),map);
        return sb.toString();
    }

    @Override
    public Archive getPreVersionArchive(int version, String modelName, String id) throws ClassNotFoundException {
        return FixedAuditReaderFactory.get(getSession(), modelService.getClassLoader())
                .find(Archive.class, id, version);
    }
}
