package com.gtis.archive.service.impl;

import com.google.common.base.Strings;
import com.gtis.archive.core.ModelService;
import com.gtis.archive.core.support.hibernate.envers.FixedAuditReaderFactory;
import com.gtis.archive.core.support.xstream.CDATASupportDom4JDriver;
import com.gtis.archive.entity.Archive;
import com.gtis.archive.entity.ArchiveMetaData;
import com.gtis.archive.service.ArchiveMetadataService;
import com.gtis.common.util.UUIDGenerator;
import com.gtis.support.hibernate.HibernateTemplate;
import com.thoughtworks.xstream.XStream;
import org.apache.commons.collections.CollectionUtils;
import org.hibernate.envers.AuditReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.FileCopyUtils;

import javax.annotation.PostConstruct;
import java.io.*;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * 档案元数据
 *
 * @author linlong
 * @date 2019.03.18
 */
@Service
public class ArchiveMetadataServiceImpl extends HibernateTemplate<ArchiveMetaData, String> implements ArchiveMetadataService, InitializingBean {

    private static Logger logger = LoggerFactory.getLogger(ArchiveMetadataServiceImpl.class);

    private XStream sm;

    @Autowired
    private ModelService modelService;

    private AuditReader auditReader;

    @PostConstruct
    public void init() {
        logger.debug("ArchiveMetadataServiceImpl init");
        sm = new XStream(new CDATASupportDom4JDriver());
        sm.processAnnotations(ArchiveMetaData.class);
    }

    /**
     * 将对象转换为xml
     *
     * @param metaData 档案元数据
     * @return xml字符串
     */
    @Override
    public String convertObjToXml(ArchiveMetaData metaData) {
        return sm.toXML(metaData);
    }

    @Override
    public ArchiveMetaData toBean(String xmlStr) {
        @SuppressWarnings("unchecked")
        ArchiveMetaData archiveMetaData = (ArchiveMetaData) sm.fromXML(xmlStr);
        return archiveMetaData;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public ArchiveMetaData getArchiveMetaDataInfoFromArchive(Archive archive, ArchiveMetaData metaData) {
        ArchiveMetaData.ArchiveMetadataBuilder builder;

        if (metaData == null) {
            builder = new ArchiveMetaData.ArchiveMetadataBuilder();
        } else {
            builder = new ArchiveMetaData.ArchiveMetadataBuilder(metaData);
        }

        metaData = builder
                .year(archive.getNd())
                .archiveId(archive.getId())
                .archivesName(archive.getTm())
                .totalPages(archive.getYs())
                .retentionPeriod(archive.getBgqx())
                .rq(archive.getGdsj())
                .fondsId(archive.getQzh())
                .archiveCode(archive.getDh())
                .classCode(archive.getFlh())
                .author(archive.getZrz())
                .securityClassification(archive.getMj())
                .totalPages(archive.getYs())
                .totalNumberOfItems(archive.getJs())
                .storageLocation(archive.getCfwz())
                .build();
        saveOrUpdate(metaData);
        return metaData;
    }

    @Override
    public ArchiveMetaData getArchiveMetaDataInfoFromArchive(Archive archive) {
        return getArchiveMetaDataInfoFromArchive(archive, null);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<File> batch(List<String> metadataIds, String parentPath) throws IOException {
        List<File> files = new ArrayList<File>(metadataIds.size());
        for (String id : metadataIds) {
            ArchiveMetaData archiveMetaData = load(id);
            File f = generateXmlFile(parentPath, archiveMetaData);
            files.add(f);
        }
        return files;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public File generateXmlFile(String parentPath, ArchiveMetaData archiveMetaData) throws IOException {
        String xml = convertObjToXml(archiveMetaData);
        String path = parentPath + File.separator + archiveMetaData.getId() + ".xml";
        File dest = new File(path);
        if (!dest.getParentFile().exists()) {
            dest.getParentFile().mkdirs();
        }
        FileCopyUtils.copy(xml, new BufferedWriter(new FileWriter(dest)));
        return dest;
    }

    /**
     * {@inheritDoc}
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public ArchiveMetaData saveOrUpdate(ArchiveMetaData archiveMetaData) {
        if (Strings.isNullOrEmpty(archiveMetaData.getId())) {
            archiveMetaData.setId(UUIDGenerator.generate());
        }
        getSession().saveOrUpdate(archiveMetaData);
        return archiveMetaData;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    @SuppressWarnings("unchecked")
    public ArchiveMetaData getArchiveMetadataByArchiveId(String archiveId) {
        String hql = "FROM ArchiveMetaData a WHERE a.archiveId=:archiveId";
        List<ArchiveMetaData> list = getSession().createQuery(hql)
                .setString("archiveId", archiveId)
                .list();
        if (CollectionUtils.isNotEmpty(list)) {
            return list.get(0);
        } else {
            return null;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Number> getVersions(String modelName, String archiveId) throws ClassNotFoundException {
        List<Number> list = auditReader.getRevisions(modelService.getClass(modelName), archiveId);
        if (list == null) {
            return Collections.emptyList();
        }

        return list;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Archive getPreVersionArchive(String modelName, String archiveId, Number version) throws ClassNotFoundException {
        return (Archive) auditReader.find(modelService.getClass(modelName), archiveId, version);
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        auditReader = FixedAuditReaderFactory.get(getSession(), modelService.getClassLoader());
    }
}
