package com.gtis.archive.service.impl;

import com.gtis.archive.Constants;
import com.gtis.archive.core.EntityService;
import com.gtis.archive.core.Model;
import com.gtis.archive.core.ModelChangeEvent;
import com.gtis.archive.core.ModelService;
import com.gtis.archive.core.environment.Environment;
import com.gtis.archive.core.ex.TemplateNotFoundException;
import com.gtis.archive.entity.Archive;
import com.gtis.archive.entity.Document;
import com.gtis.archive.entity.Original;
import com.gtis.archive.entity.Resource;
import com.gtis.archive.service.ResourceService;
import com.gtis.archive.service.SecurityService;
import com.gtis.search.*;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.ApplicationListener;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * .
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 12-2-12
 */
public class ArchiveIndexProviderImpl extends EntityIndexProvider<Object> implements ApplicationListener {

    protected static final String[] TPLS = {com.gtis.search.Constants.TPL_TEXT_URL, com.gtis.search.Constants.TPL_TEXT_BODY, com.gtis.search.Constants.TPL_FILE_URL, com.gtis.search.Constants.TPL_FILE_BODY};
    @Autowired
    private ModelService modelService;
    @Autowired
    private EntityService entityService;
    @Autowired
    private SecurityService securityService;
    @Autowired
    private IndexManager indexManager;
    @Autowired
    private ResourceService resourceService;
    private CategoryFactory defaultCategoryFactory;
    private ScheduledExecutorService executorService;
    private Future reloadFuture;

    public void setDefaultBusinessFactory(BusinessFactory defaultBusinessFactory) {
        defaultBusiness = defaultBusinessFactory.getBusiness();
    }

    public void setDefaultCategoryFactory(CategoryFactory defaultCategoryFactory) {
        this.defaultCategoryFactory = defaultCategoryFactory;
    }

    @Override
    public List<Business> getBusinesses() {
        List<Category> categories = new ArrayList<Category>();
        for (Model model : modelService.getModels()) {
            if (!model.isHidden()) {
                Category cat = defaultCategoryFactory.getCategory(defaultBusiness).clone();
                cat.setId(Constants.ARCHIVE_PREFIX + model.getName());
                cat.setName(model.getTitle());
                cat.setIcon(model.getName());
                cat.setRoleIds(getReadableRoleIds(model.getName()));
                for (String tpl : TPLS) {
                    try {
                        cat.addTpl(tpl, model.getTemplate("search" + StringUtils.capitalize(tpl)));
                    } catch (TemplateNotFoundException ignored) {
                        logger.info(ignored.getMessage());
                    }
                }
                categories.add(cat);
            }
        }
        defaultBusiness.setCategories(categories);
        return Collections.singletonList(defaultBusiness);
    }

    private String[] getReadableRoleIds(String name) {
        Resource res = resourceService.findChildResource(null, name, Constants.MODEL_ROOT);
        if (res != null) {
            Set<String> ids = securityService.getReadableRoleIds(res.getId());
            return ids.toArray(new String[ids.size()]);
        }
        return new String[0];
    }

    @Override
    public int getTotalCount(String categoryId) {
        return entityService.getCount(getModelName(categoryId), null);
    }

    @Override
    protected Index createIndex(Object entity) {
        if (entity instanceof Archive || entity instanceof Document || entity instanceof Original) {
            return super.createIndex(entity);
        } else {
            return null;
        }
    }

    @Override
    protected Index prepareIndex(Index index) {
        return index;
    }

    @Override
    protected Object getEntity(String categoryId, String id) {
        return entityService.load(getModelName(categoryId), id);
    }

    @Override
    protected void extractEntity(Object entity, Index index) {
        index.setBusinessId(defaultBusiness.getId());
        index.setCategoryId(Constants.ARCHIVE_PREFIX + modelService.getModelName(entity));
        if (entity instanceof Archive) {
            extractArchive((Archive) entity, index);
        } else if (entity instanceof Document) {
            extractDocument((Document) entity, index);
        } else if (entity instanceof Original) {
            extractOriginal((Original) entity, index);
        }
        index.appendBody(renderBody(entity));
    }

    private String renderBody(Object entity) {
        try {
            Model model = modelService.getModel(entity);
            Environment env = new Environment(entity, model.getEnv());
            env.put("fields", StringUtils.join(model.getInheritfieldsMap().keySet(), " "));
            String indexTpl = modelService.getModel(entity).getTemplate("index");
            return env.getExpr(indexTpl);
        } catch (Exception ignored) {
            logger.error(ignored.getMessage());
        }
        return null;
    }

    @Override
    protected List<Object> getEntities(String categoryId, int start, int size) {
        return entityService.search(getModelName(categoryId), null, null, start, size).getItems();
    }

    private String getModelName(String categoryId) {
        return categoryId.substring(2);
    }

    protected void extractArchive(Archive entity, Index index) {
        index.setId(entity.getId());
        index.setTitle(entity.getTm());
        index.setDate(entity.getQrq());
    }

    protected void extractDocument(Document entity, Index index) {
        index.setId(entity.getId());
        index.setTitle(entity.getTm());
        index.setDate(entity.getRq());

        index.addSearchableField("aid", entity.getArchiveId());
    }

    public Date dateFormat(String rq) {
        //注意：SimpleDateFormat构造函数的样式与strDate的样式必须相符
        SimpleDateFormat sDateFormat=new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"); //加上时间
        //必须捕获异常
        try {
            return sDateFormat.parse(rq);
        } catch(ParseException px) {
            logger.error(px.getMessage());
        }
        return new Date();
    }

    protected void extractOriginal(Original entity, Index index) {
        index.setId(entity.getId());
        index.setTitle(entity.getName());
        index.setDate(entity.getUpdateTime());
        index.setMimeType(MimeTypeHelper.getMimeType(entity.getName()));
        index.appendBody(entity.getContent(), 5000);
        index.addField("size", (int) entity.getFileSize());
        index.addSearchableField("oid", entity.getOwnerId());
        Archive archive = entityService.load(Archive.class, entity.getOwnerId());
        if (archive != null) {
            index.addField("ownerIsArchive", true);
            String modelName = modelService.getModelName(archive);
            if (Archive.MODEL_NAME.equals(modelName)) {
                modelName = Original.MODEL_NAME;
            }
            index.setCategoryId(Constants.ARCHIVE_PREFIX + modelName);
        } else {
            Document document = entityService.load(Document.class, entity.getOwnerId());
            if (document != null) {
                String modelName = modelService.getModelName(document);
                if (Document.MODEL_NAME.equals(modelName)) {
                    modelName = Original.MODEL_NAME;
                }
                index.setCategoryId(Constants.ARCHIVE_PREFIX + modelName);
            }
        }
    }

    @Override
    public void onApplicationEvent(ApplicationEvent event) {
        if (event instanceof ModelChangeEvent) {
            defaultBusiness.setLastModified(event.getTimestamp());
            synchronized (this) {
                if (executorService == null) {
                    executorService = Executors.newSingleThreadScheduledExecutor();
                }
                if (reloadFuture != null) {
                    reloadFuture.cancel(false);
                }
                reloadFuture = executorService.schedule(new Runnable() {
                    @Override
                    public void run() {
                        indexManager.reloadBusiness(defaultBusiness.getId());
                        reloadFuture.cancel(false);
                        reloadFuture = null;
                    }
                }, 10, TimeUnit.SECONDS);
            }
        }
    }
}
