package com.gtis.archive.service;

import com.gtis.archive.core.Model;
import com.gtis.archive.entity.Archive;
import com.gtis.archive.entity.Chapter;
import com.gtis.archive.entity.Document;
import com.gtis.common.Page;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;

import java.util.List;


/**
 * 档案服务.
 * <p/>
 *
 * @author <a href="mailto:oxsean@gmail.com">sean yang</a>
 * @version V1.0, 2010-10-26
 */
public interface ArchiveService {
    /**
     * 获取案卷
     *
     * @param modelName 案卷模型名
     * @param id        案卷id
     * @return 案卷
     */
    Archive getArchive(String modelName, String id);

    /**
     * 获取卷内
     *
     * @param modelName 卷内模型名
     * @param id        卷内id
     * @return 卷内
     */
    Document getDocument(String modelName, String id);

    /**
     * 获取二级卷内
     *
     * @param modelName 卷内模型名
     * @param id        卷内id
     * @return 卷内
     */
    Chapter getChapter(String modelName, String id);

    /**
     * 获取只有基本信息的案卷对象
     *
     * @param id 案卷id
     * @return 案卷
     */
    Archive getSimpleArchive(String id);

    /**
     * 根据项目Id获取只有基本信息的案卷对象
     *
     * @param proid
     * @return
     */
    Archive getSimpleArchiveByProId(String proid);

    /**
     * 获取只有基本信息的卷内对象
     *
     * @param id 卷内id
     * @return 案卷
     */
    Document getSimpleDocument(String id);

    /**
     * 获取只有基本信息的卷内对象
     *
     * @param tm 卷内题名
     * @return 案卷
     */
    Document getSimpleDocumentBySxh(String tm,int sxh);

    /**
     * 获取只有基本信息的卷内对象
     *
     * @param id 卷内id
     * @return 案卷
     */
    Chapter findChapterById(String id);

    /**
     * 获取卷内的案卷信息
     *
     * @param modelName  卷内模型名
     * @param documentId 卷内id
     * @return 案卷
     */
    Archive getDocumentArchive(String modelName, String documentId);

    /**
     * 获取卷内的案卷信息
     *
     * @param document 卷内
     * @return 案卷
     */
    Archive getDocumentArchive(Document document);

    /**
     * 获取案卷的卷内列表
     *
     * @param modelName 案卷模型名
     * @param archiveId 案卷id
     * @return 卷内列表
     */
    List<Document> getArchiveDocuments(String modelName, String archiveId);

    /**
     * 获取案卷的卷内列表
     *
     * @param archive 案卷
     * @return 卷内列表
     */
    List<Document> getArchiveDocuments(Archive archive);

    /**
     * 获取案卷的卷内列表
     *
     * @param document 案卷
     * @return 卷内列表
     */
    List<Chapter> getDocumentChapters(Document document);

    /**
     * 根据模型名获取案卷模型
     *
     * @param modelName 模型名
     * @return 字段列表
     */
    Model getArchiveModel(String modelName);

    /**
     * 根据模型名获取卷内模型
     *
     * @param modelName 模型名
     * @return 字段列表
     */
    Model getDocumentModel(String modelName);

    /**
     * 根据模型名创建一个案卷实例
     *
     * @param modelName 案卷的模型名
     * @return 案卷
     */
    Archive newArchive(String modelName);

    /**
     * 根据模型名创建一个卷内实例
     *
     * @param modelName 案卷的模型名
     * @return 案卷
     */
    Document newDocument(String modelName);

    /**
     * 根据模型名创建一个卷内实例
     *
     * @param modelName 案卷的模型名
     * @return 案卷
     */
    Chapter newChapter(String modelName);

    /**
     * 删除案卷
     *
     * @param modelName 案卷的模型名
     * @param ids       案卷id列表
     */
    void removeArchive(String modelName, String[] ids);

    /**
     * 删除卷内
     *
     * @param modelName 案卷的模型名
     * @param ids       卷内id列表
     */
    void removeDocument(String modelName, String[] ids);

    /**
     * 删除卷内
     *
     * @param modelName 案卷的模型名
     * @param ids       卷内id列表
     */
    void removeChapter(String modelName, String[] ids);

    /**
     * 保存案卷
     *
     * @param archive 案卷
     * @return 安敢
     */
    Archive saveArchive(Archive archive);

    /**
     * 保存卷内
     *
     * @param document 卷内
     * @return 卷内
     */
    Document saveDocument(Document document);

    /**
     * 保存二级卷内
     *
     * @param chapter 卷内
     * @return 卷内
     */
    Chapter saveChapter(Chapter chapter);

    /**
     * 检索案卷
     *
     * @param modelName  案卷的模型名
     * @param criterions 检索条件
     * @param orders     排序
     * @param start      位置
     * @param limit      分页大小
     * @return 案卷列表page对象
     */
    Page<Archive> searchArchive(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit);

    /**
     * 检索卷内
     *
     * @param modelName  案卷的模型名
     * @param criterions 检索条件
     * @param orders     排序
     * @param start      位置
     * @param limit      分页大小
     * @return 卷内列表page对象
     */
    Page<Document> searchDocument(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit);

    /**
     * 获取案卷数量
     *
     * @param modelName  案卷的模型名
     * @param criterions 查询条件
     * @return 数量
     */
    int getArchiveCount(String modelName, List<? extends Criterion> criterions);

    /**
     * @param modelName
     * @param criterions
     * @param orders
     * @param dwdm
     * @param start
     * @param limit
     * @return
     */
    Page<Archive> searchArchive(String modelName, List<? extends Criterion> criterions, List<Order> orders, String dwdm, int start, int limit);

    /**
     * 获取满足条件的档案序列
     *
     * @return
     */
    List<Object> searchArchiveList(String modelName, List<? extends Criterion> criterions, List<Order> orders, int start, int limit);

    /**
     * 获取分类号种类
     *
     * @return
     */
    String flhjs(String flhjs, String modelName);

    /**
     * 通过档号获取档案
     *
     * @param dh 档号
     * @return archive
     */
    Archive getArchiveByDh(String dh);

    /**
     * 对档号进行模糊查询
     * @param dh
     * @return
     */
    List<Archive> getArchivesByDh(String dh);

    /**
     * 通过档号获取档案
     *
     * @param ajh 案卷号
     * @return archive
     */
    Archive getArchiveByAjh(Integer ajh, String mlh);

    /**
     * 获取ajhs
     *
     * @param room
     * @param carbinet
     * @param side
     * @param cell
     * @param box
     * @return
     */
    String getAjhs(String room, String carbinet, Integer side, Integer cell, Integer box);

    /**
     * 获取卷内最大顺序号的卷内档案
     *
     * @param modelName
     * @param archiveId
     * @return
     */
    public Document getMaxSxhDocuments(String modelName, String archiveId);

    /**
     * 获取二级卷内最大顺序号的卷内档案
     *
     * @param modelName
     * @param archiveId
     * @return
     */
    public Chapter getMaxSxhChp(String modelName, String archiveId);

    /**
     * 获取档案最大柜号
     *
     * @return
     */
    public Archive getArchiveByPosition(String modelName);

    /**
     * 获取档案最大柜号
     *
     * @return
     */
    public Archive getArchiveByPosition();

    /**
     * 获取档案最大柜号
     *
     * @return
     */
    public Integer getArchivePosition();

    /**
     * 根据柜号获取柜号中最大格号
     *
     * @param setCabinetSno
     * @return
     */
    public Integer getMaxCellId(String setCabinetSno);

    /**
     * 根据格号获取最大盒号
     *
     * @param cellId
     * @return
     */
    public Integer getMaxBox(Integer cellId, String cabinetSno);

    /**
     * 根据盒号判断该盒是否放满
     *
     * @param boxId
     * @return
     */
    public boolean isBoxFull(Integer boxId, Integer cellId, String cabinetSno);

    /**
     * 判断格子是否已满
     *
     * @param cellId
     * @param cabinetSno
     * @return
     */
    public boolean isCellFull(Integer cellId, Integer cabinetSno);

    /**
     * 获取字表中移交单号最大值
     *
     * @param tableName 字表名称
     * @return
     */
    public int maxYjdh(String tableName, String nd, String ywjc, String dwjc);

    /**
     * 根据id更新字表的移交单号
     *
     * @param tableName
     * @param id
     */
    public void updateYjdhById(String tableName, String id, String value, String type);

    /**
     * 根据用户的部门编码生成13位档号
     *
     * @param archive
     * @param userId
     * @return
     */
    public Archive createDhByArchive(Archive archive, String userId);

    /**
     * 根据卷内更新案卷的页数
     *
     * @param sum
     */
    public void updateArchiveYsByDoc(int sum, String archiveId);

    /**
     * 判断boxId在是否已占用
     *
     * @param boxId
     * @return
     */
    public boolean isExistBoxId(Integer boxId, String mlh,String cabinetSno,Integer columnId,Integer cellId, String modelName,String nd);


    /**
     * 判断boxId在是否已占用
     *
     * @param boxId
     * @return
     */
    public boolean isExistBoxId(Integer boxId, String mlh,String cabinetSno,Integer columnId,Integer cellId, String modelName);

    /**
     * 根据Hql语句对archive进行更新
     *
     * @param hql     语句
     * @param objects 参数
     */
    public void upDateArchiveByHql(String hql, Object... objects);

    /**
     * 获取档案的年度
     *
     * @param modelName 模型名
     * @return
     */
    List getArchiveNd(String modelName);

    /**
     * 获取档案的目录号
     * @param modelName 业务模型名
     * @return
     */
    List getArchiveMlh(String modelName);

    /**
     * 获取档案的盒号
     *
     * @param modelName 模型名
     * @param nd        年度
     * @param dh        档号
     * @param mlh       目录号
     * @param boxId     盒号
     * @param cabinetSno     密集架号
     * @param columnId     列号
     * @param cellId     层号
     * @param cfwz      存放位置
     * @param fiedsSQL     查询字段
     * @param orderSQL     排序字段
     * @return
     */
    List getBox(String modelName, String nd, String dh, String mlh, String boxId, String cabinetSno, String columnId, String cellId, String cfwz, String fiedsSQL, String orderSQL);

    /**
     * 删除档案的盒号
     *
     * @param boxIds    盒号信息
     * @param modelName 业务模型名
     * @param nd        年度
     * @param dh        档号
     * @param mlh       目录号
     * @param cfwz      存放位置
     * @return
     */
    void deleteArchiveBoxId(String[] boxIds, String modelName, String nd, String dh, String mlh, String cfwz);

    /**
     * 保存档案盒号的存放位置
     * @param modelName 模型名
     * @param nd 年度
     * @param dh 档号
     * @param mlh 目录号
     * @param boxId 盒号
     * @param cfwz 存放位置
     */
    void updateArchiveBoxIDd(String modelName, String nd, String dh, String mlh, String boxId, String cfwz);

    Integer getMaxSideId(String cabinetsno,Integer boxId,Integer cellId);

    /**
     * 判断该存放位置是否已存放档案
     * @param cfwz
     * @return
     */
    boolean isExistArchiveByCfwz(String cfwz);

    /**
     * 根据sql获取archive相应的字段值
     * @param sql
     * @param objects
     * @return
     */
    List<Object> getFieldValue(String sql,Object...objects);

    /**
     * 获取最大盒号
     * @return
     */
    Integer getMaxBoxNum(String modelName,String nd,String mlh);

    /**
     * 自动获取生成盒号
     * 盒号格式按照芜湖县的要求
     * 格式：H-年-流水号
     * @return
     */
    String autoGenerateBoxId();

    /**
     * 根据房间号、柜号、格子号获取档案
     * @param currentRoom
     * @param currentCab
     * @param rowId
     * @param colId
     * @return
     */
    List<Archive> getArchivesByCell(String currentRoom, String currentCab, Integer rowId, Integer colId);

    /**
     * 根据盒号获取案卷
     * @param boxId
     * @return
     */
    List<Archive> getArchivesByBox(Integer boxId);

    /**
     * 根据存放位置获取案卷
     * @param cfwz
     * @return
     */
    List<Archive> getArchivesByCfwz(String cfwz);

    /**
     * 获取最大的流水号
     * @param modelName
     * @param archive
     * @return
     * @throws ClassNotFoundException
     */
    String getMaxLsh(String modelName, Archive archive) throws ClassNotFoundException;

    /**
     * 生成流水号
     * @param modelName
     * @param archive
     * @param modelName 业务模型名
     * @param archive 档案
     * @return
     * @throws ClassNotFoundException
     */
    String generateAndUpdateLsh(String modelName, Archive archive);

    /**
     * 获取房产业务的流水号
     * @param archive
     * @return
     */
    Integer getFcywLsh(Archive archive);

    /**
     * 根据hql语句进行查询
     * @param hql hql语句
     * @param objects 传递来的不定长度参数
     * @return
     */
    List<Archive> getArchiveByHQL(String hql,Object...objects);

    /**
     * 马鞍山触摸屏查询生成随机码
     * @param id
     * @return
     */
    String getCode(String id);

    /**
     * 获取先前版本的档案记录
     *
     * @return 先前版本的档案实例
     */
    Archive getPreVersionArchive(int version, String modelName, String id) throws ClassNotFoundException;
}
