package com.gtis.archive.core.support.hibernate.envers;

import org.hibernate.envers.RevisionType;
import org.hibernate.envers.configuration.AuditConfiguration;
import org.hibernate.envers.configuration.AuditEntitiesConfiguration;
import org.hibernate.envers.exception.AuditException;
import org.hibernate.envers.query.criteria.AuditCriterion;
import org.hibernate.envers.query.impl.AbstractAuditQuery;
import org.hibernate.envers.reader.AuditReaderImplementor;
import org.hibernate.proxy.HibernateProxy;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * @author linlong
 * @since 2019.04.09
 */
public class FixedRevisionOfEntityQuery extends AbstractAuditQuery {

    private final boolean selectEntitiesOnly;
    private final boolean selectDeletedEntities;
    private final ClassLoader classLoader;
    private final FixedEntityInstantiator fixedEntityInstantiator;

    public FixedRevisionOfEntityQuery(AuditConfiguration verCfg,
                                  AuditReaderImplementor versionsReader,
                                  Class<?> cls, boolean selectEntitiesOnly,
                                  boolean selectDeletedEntities, ClassLoader classLoader) {
        super(verCfg, versionsReader, cls);

        this.selectEntitiesOnly = selectEntitiesOnly;
        this.selectDeletedEntities = selectDeletedEntities;
        this.classLoader = classLoader;
        this.fixedEntityInstantiator = new FixedEntityInstantiator(verCfg, versionsReader, classLoader);
    }

    public FixedRevisionOfEntityQuery(AuditConfiguration verCfg,
                                  AuditReaderImplementor versionsReader, Class<?> cls, String entityName,
                                  boolean selectEntitiesOnly, boolean selectDeletedEntities,
                                      ClassLoader classLoader) {
        super(verCfg, versionsReader, cls, entityName);

        this.selectEntitiesOnly = selectEntitiesOnly;
        this.selectDeletedEntities = selectDeletedEntities;
        this.classLoader = classLoader;
        this.fixedEntityInstantiator = new FixedEntityInstantiator(verCfg, versionsReader, classLoader);
    }

    private Number getRevisionNumber(Map versionsEntity) {
        AuditEntitiesConfiguration verEntCfg = verCfg.getAuditEntCfg();

        String originalId = verEntCfg.getOriginalIdPropName();
        String revisionPropertyName = verEntCfg.getRevisionFieldName();

        Object revisionInfoObject = ((Map) versionsEntity.get(originalId)).get(revisionPropertyName);

        if (revisionInfoObject instanceof HibernateProxy) {
            return (Number) ((HibernateProxy) revisionInfoObject).getHibernateLazyInitializer().getIdentifier();
        } else {
            // Not a proxy - must be read from cache or with a join
            return verCfg.getRevisionInfoNumberReader().getRevisionNumber(revisionInfoObject);
        }
    }

    @SuppressWarnings({"unchecked"})
    @Override
    public List list() throws AuditException {
        AuditEntitiesConfiguration verEntCfg = verCfg.getAuditEntCfg();

        /*
        The query that should be executed in the versions table:
        SELECT e (unless another projection is specified) FROM ent_ver e, rev_entity r WHERE
          e.revision_type != DEL (if selectDeletedEntities == false) AND
          e.revision = r.revision AND
          (all specified conditions, transformed, on the "e" entity)
          ORDER BY e.revision ASC (unless another order or projection is specified)
         */
        if (!selectDeletedEntities) {
            // e.revision_type != DEL AND
            qb.getRootParameters().addWhereWithParam(verEntCfg.getRevisionTypePropName(), "<>", RevisionType.DEL);
        }

        // all specified conditions, transformed
        for (AuditCriterion criterion : criterions) {
            criterion.addToQuery(verCfg, entityName, qb, qb.getRootParameters());
        }

        if (!hasProjection && !hasOrder) {
            String revisionPropertyPath = verEntCfg.getRevisionNumberPath();
            qb.addOrder(revisionPropertyPath, true);
        }

        if (!selectEntitiesOnly) {
            qb.addFrom(verCfg.getAuditEntCfg().getRevisionInfoEntityName(), "r");
            qb.getRootParameters().addWhere(verCfg.getAuditEntCfg().getRevisionNumberPath(), true, "=", "r.id", false);
        }

        List<Object> queryResult = buildAndExecuteQuery();
        if (hasProjection) {
            return queryResult;
        } else {
            List entities = new ArrayList();
            String revisionTypePropertyName = verEntCfg.getRevisionTypePropName();

            for (Object resultRow : queryResult) {
                Map versionsEntity;
                Object revisionData;

                if (selectEntitiesOnly) {
                    versionsEntity = (Map) resultRow;
                    revisionData = null;
                } else {
                    Object[] arrayResultRow = (Object[]) resultRow;
                    versionsEntity = (Map) arrayResultRow[0];
                    revisionData = arrayResultRow[1];
                }

                Number revision = getRevisionNumber(versionsEntity);

                Object entity = fixedEntityInstantiator.createInstanceFromVersionsEntity(entityName, versionsEntity, revision);

                if (!selectEntitiesOnly) {
                    entities.add(new Object[] { entity, revisionData, versionsEntity.get(revisionTypePropertyName) });
                } else {
                    entities.add(entity);
                }
            }

            return entities;
        }
    }
}
