package com.gtis.archive.core.support.hibernate.envers;

import org.hibernate.envers.configuration.AuditConfiguration;
import org.hibernate.envers.entities.EntityConfiguration;
import org.hibernate.envers.entities.mapper.id.IdMapper;
import org.hibernate.envers.exception.AuditException;
import org.hibernate.envers.reader.AuditReaderImplementor;
import org.hibernate.envers.tools.reflection.ReflectionTools;
import org.hibernate.util.ReflectHelper;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * 主要为了解决档案系统中的动态加载的类通过默认的类加载器无法加载的问题
 *
 * @author Lin.L
 */
public class FixedEntityInstantiator {

    /**
     * 配置类
     */
    private final AuditConfiguration verCfg;
    private final AuditReaderImplementor versionsReader;
    /**
     * 类加载器
     */
    private final ClassLoader classLoader;

    public FixedEntityInstantiator(AuditConfiguration verCfg, AuditReaderImplementor versionsReader, ClassLoader classLoader) {
        this.verCfg = verCfg;
        this.versionsReader = versionsReader;
        this.classLoader = classLoader;
    }

    /**
     * Creates an entity instance based on an entry from the versions table.
     * @param entityName Name of the entity, which instances should be read
     * @param versionsEntity An entry in the versions table, from which data should be mapped.
     * @param revision Revision at which this entity was read.
     * @return An entity instance, with versioned properties set as in the versionsEntity map, and proxies
     * created for collections.
     */
    public Object createInstanceFromVersionsEntity(String entityName, Map versionsEntity, Number revision) {
        if (versionsEntity == null) {
            return null;
        }

        // The $type$ property holds the name of the (versions) entity
        String type = verCfg.getEntCfg().getEntityNameForVersionsEntityName((String) versionsEntity.get("$type$"));

        if (type != null) {
            entityName = type;
        }

        // First mapping the primary key
        IdMapper idMapper = verCfg.getEntCfg().get(entityName).getIdMapper();
        Map originalId = (Map) versionsEntity.get(verCfg.getAuditEntCfg().getOriginalIdPropName());

        Object primaryKey = idMapper.mapToIdFromMap(originalId);

        // Checking if the entity is in cache
        if (versionsReader.getFirstLevelCache().contains(entityName, revision, primaryKey)) {
            return versionsReader.getFirstLevelCache().get(entityName, revision, primaryKey);
        }

        // If it is not in the cache, creating a new entity instance
        Object ret;
        try {
            EntityConfiguration entCfg = verCfg.getEntCfg().get(entityName);
            if (entCfg == null) {
                // a relation marked as RelationTargetAuditMode.NOT_AUDITED
                entCfg = verCfg.getEntCfg().getNotVersionEntityConfiguration(entityName);
            }

            Class<?> cls;
            // 通过指定的classloader获取类
            if (classLoader != null) {
                cls = classLoader.loadClass(entCfg.getEntityClassName());
            } else {
                cls = ReflectionTools.loadClass(entCfg.getEntityClassName());
            }
            ret = ReflectHelper.getDefaultConstructor(cls).newInstance();
        } catch (Exception e) {
            throw new AuditException(e);
        }

        // Putting the newly created entity instance into the first level cache, in case a one-to-one bidirectional
        // relation is present (which is eagerly loaded).
        versionsReader.getFirstLevelCache().put(entityName, revision, primaryKey, ret);

        verCfg.getEntCfg().get(entityName).getPropertyMapper().mapToEntityFromMap(verCfg, ret, versionsEntity, primaryKey,
                versionsReader, revision);
        idMapper.mapToEntityFromMap(ret, originalId);

        // Put entity on entityName cache after mapping it from the map representation
        versionsReader.getFirstLevelCache().putOnEntityNameCache(primaryKey, revision, ret, entityName);

        return ret;
    }

    @SuppressWarnings({"unchecked"})
    public void addInstancesFromVersionsEntities(String entityName, Collection addTo, List<Map> versionsEntities, Number revision) {
        for (Map versionsEntity : versionsEntities) {
            addTo.add(createInstanceFromVersionsEntity(entityName, versionsEntity, revision));
        }
    }
}
